/*
 * EventHandlers.cs - Event handler delegate types for X#.
 *
 * This file is part of the X# library.
 * Copyright (C) 2002  Southern Storm Software, Pty Ltd.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace XWindows
{

using System;

/// <summary>
/// <para>The <see cref="T:XWindows.MovedEventHandler"/> delegate is
/// used to describe widget move events.</para>
/// </summary>
///
/// <param name="widget">
/// <para>The widget that was moved.</para>
/// </param>
///
/// <param name="x">
/// <para>The new X co-ordinate for the widget.</para>
/// </param>
///
/// <param name="y">
/// <para>The new Y co-ordinate for the widget.</para>
/// </param>
public delegate void MovedEventHandler(Widget widget, int x, int y);

/// <summary>
/// <para>The <see cref="T:XWindows.ResizedEventHandler"/> delegate is
/// used to describe widget resize events.</para>
/// </summary>
///
/// <param name="widget">
/// <para>The widget that was resized.</para>
/// </param>
///
/// <param name="width">
/// <para>The new width for the widget.</para>
/// </param>
///
/// <param name="height">
/// <para>The new height for the widget.</para>
/// </param>
public delegate void ResizedEventHandler(Widget widget, int width, int height);

/// <summary>
/// <para>The <see cref="T:XWindows.MapStateEventHandler"/> delegate is
/// used to describe widget map state change events.</para>
/// </summary>
///
/// <param name="widget">
/// <para>The widget that was mapped or unmapped.</para>
/// </param>
///
/// <param name="mapped">
/// <para>Set to <see langword="true"/> if the widget has just been
/// mapped; <see langword="false"/> otherwise.</para>
/// </param>
public delegate void MapStateEventHandler(Widget widget, bool mapped);

/// <summary>
/// <para>The <see cref="T:XWindows.PaintEventHandler"/> delegate is
/// used to describe widget paint events.</para>
/// </summary>
///
/// <param name="widget">
/// <para>The widget that is being painted.</para>
/// </param>
///
/// <param name="canvas">
/// <para>The canvas to use to repaint the widget.  This
/// canvas will have been initialised with the widgets foreground and
/// background colors, and with the clipping region set to the area
/// that needs to be repainted.</para>
/// </param>
public delegate void PaintEventHandler(Widget widget, Canvas canvas);

/// <summary>
/// <para>The <see cref="T:XWindows.MapStateEventHandler"/> delegate is
/// used to describe widget iconic state change events.</para>
/// </summary>
///
/// <param name="widget">
/// <para>The widget that was iconified or de-iconified.</para>
/// </param>
///
/// <param name="iconified">
/// <para>Set to <see langword="true"/> if the widget has just been
/// iconified; <see langword="false"/> if it was de-iconified.</para>
/// </param>
public delegate void IconicStateEventHandler(Widget widget, bool iconified);

/// <summary>
/// <para>The <see cref="T:XWindows.ClosedEventHandler"/> delegate is
/// used to describe widget close requests.</para>
/// </summary>
///
/// <param name="widget">
/// <para>The widget that is to be closed.</para>
/// </param>
///
/// <returns>
/// <para>Returns <see langword="true"/> if the window can be closed,
/// or <see langword="false"/> if some condition is preventing to
/// window from being closed.</para>
/// </returns>
public delegate bool ClosedEventHandler(Widget widget);

} // namespace XWindows
