/*
 * InputOnlyWidget.cs - Widget handling for input-only widgets.
 *
 * This file is part of the X# library.
 * Copyright (C) 2002  Southern Storm Software, Pty Ltd.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace XWindows
{

using System;
using XWindows.Types;

/// <summary>
/// <para>The <see cref="T:XWindows.InputOnlyWidget"/> class manages widgets
/// that occupy screen real estate for the purpose of keyboard and pointer
/// handling, but which do not have output functionality.</para>
/// </summary>
public class InputOnlyWidget : Widget
{
	/// <summary>
	/// <para>Constructs a new <see cref="T:XWindows.InputOnlyWidget"/>
	/// instance underneath a specified parent widget.</para>
	/// </summary>
	///
	/// <param name="parent">
	/// <para>The parent of the new widget.</para>
	/// </param>
	///
	/// <param name="x">
	/// <para>The X co-ordinate of the top-left corner of
	/// the new widget.</para>
	/// </param>
	///
	/// <param name="y">
	/// <para>The Y co-ordinate of the top-left corner of
	/// the new widget.</para>
	/// </param>
	///
	/// <param name="width">
	/// <para>The width of the new widget.</para>
	/// </param>
	///
	/// <param name="height">
	/// <para>The height of the new widget.</para>
	/// </param>
	///
	/// <exception cref="T:System.ArgumentNullException">
	/// <para>Raised if <paramref name="parent"/> is <see langword="null"/>.
	/// </para>
	/// </exception>
	///
	/// <exception cref="T:XWindows.XException">
	/// <para>Raised if <paramref name="x"/>, <paramref name="y"/>,
	/// <paramref name="width"/>, or <paramref name="height"/> are
	/// out of range.</para>
	/// </exception>
	///
	/// <exception cref="T.XWindows.XInvalidOperationException">
	/// <para>Raised if <paramref name="parent"/> is disposed or the
	/// root window.</para>
	/// </exception>
	public InputOnlyWidget(Widget parent, int x, int y, int width, int height)
			: base(GetDisplay(parent, false), GetScreen(parent),
				   DrawableKind.InputOnlyWidget, parent)
			{
				try
				{
					// Validate the position and size.
					if(x < -32768 || x > 32767 ||
					   y < -32768 || y > 32767)
					{
						throw new XException(S._("X_InvalidPosition"));
					}
					if(width < 1 || width > 32767 ||
					   height < 1 || height > 32767 ||
					   !ValidateSize(width, height))
					{
						throw new XException(S._("X_InvalidSize"));
					}

					// Set the initial position and size of the widget.
					this.x = x;
					this.y = y;
					this.width = width;
					this.height = height;

					// Lock down the display and create the window handle.
					try
					{
						IntPtr display = dpy.Lock();
						Xlib.Window pwindow = parent.GetWidgetHandle();
						Xlib.Window window = Xlib.XCreateWindow
								(display, pwindow,
								 x, y, (uint)width, (uint)height, (uint)0,
								 screen.DefaultDepth, 2 /* InputOnly */,
								 screen.DefaultVisual,
								 (uint)0, IntPtr.Zero);
						SetWidgetHandle(window);
						if(parent.AutoMapChildren)
						{
							Xlib.XMapWindow(display, window);
						}
					}
					finally
					{
						dpy.Unlock();
					}

					// Push the widget down to the default layer.
					layer = 0x7FFFFFFF;
					Layer = 0;
				}
				finally
				{
					// Creation failed, so detach ourselves from
					// the parent's widget tree.
					Detach();
				}
			}

	// Internal constructor that is used by the "InputOutputWidget" subclass.
	internal InputOnlyWidget(Widget parent, int x, int y,
							 int width, int height, Color background,
							 bool rootAllowed, bool overrideRedirect)
			: base(GetDisplay(parent, rootAllowed), GetScreen(parent),
				   DrawableKind.Widget, parent)
			{
				try
				{
					// Validate the position and size.
					if(x < -32768 || x > 32767 ||
					   y < -32768 || y > 32767)
					{
						throw new XException(S._("X_InvalidPosition"));
					}
					if(width < 1 || width > 32767 ||
					   height < 1 || height > 32767 ||
					   !ValidateSize(width, height))
					{
						throw new XException(S._("X_InvalidSize"));
					}

					// Set the initial position and size of the widget.
					this.x = x;
					this.y = y;
					this.width = width;
					this.height = height;

					// Lock down the display and create the window handle.
					try
					{
						IntPtr display = dpy.Lock();
						Xlib.Window pwindow = parent.GetWidgetHandle();
						/* eve include  = new XSetWindowAttributes()*/
						XSetWindowAttributes attrs = new XSetWindowAttributes();
						attrs.override_redirect = overrideRedirect;
						Xlib.Window window = Xlib.XCreateWindow
								(display, pwindow,
								 x, y, (uint)width, (uint)height, (uint)0,
								 screen.DefaultDepth, 1 /* InputOutput */,
								 screen.DefaultVisual,
								 (uint)(CreateWindowMask.CWOverrideRedirect),
								 ref attrs);
						SetWidgetHandle(window);
						Xlib.XSetWindowBackground(display, window,
												  ToPixel(background));
						if(parent.AutoMapChildren)
						{
							Xlib.XMapWindow(display, window);
						}
					}
					finally
					{
						dpy.Unlock();
					}

					// Push the widget down to the default layer.
					layer = 0x7FFFFFFF;
					Layer = 0;
				}
				finally
				{
					// Creation failed, so detach ourselves from
					// the parent's widget tree.
					Detach();
				}
			}

	// Get the display associated with a parent widget, and also
	// validate the widget.
	private static Display GetDisplay(Widget parent, bool rootAllowed)
			{
				if(parent == null)
				{
					throw new ArgumentNullException("parent");
				}
				if(!rootAllowed && parent is RootWindow)
				{
					throw new XInvalidOperationException
						(S._("X_NonRootParent"));
				}
				return parent.Display;
			}

	// Get the screen associated with a parent widget.
	private static Screen GetScreen(Widget parent)
			{
				if(parent != null)
				{
					return parent.Screen;
				}
				else
				{
					return null;
				}
			}

} // class InputOnlyWidget

} // namespace XWindows
