/*
 * InputOutputWidget.cs - Widget handling for input-output widgets.
 *
 * This file is part of the X# library.
 * Copyright (C) 2002  Southern Storm Software, Pty Ltd.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace XWindows
{

using System;
using XWindows.Types;
using XWindows.Events;

/// <summary>
/// <para>The <see cref="T:XWindows.InputOutputWidget"/> class manages widgets
/// that occupy screen real estate for the purpose of keyboard and pointer
/// handling, and which have output functionality.</para>
/// </summary>
public class InputOutputWidget : InputOnlyWidget
{
	// Internal state.
	private Color foreground;
	private Color background;
	private PaintEventHandler paint;
	private Region exposeRegion;
	internal InputOutputWidget nextExpose;

	/// <summary>
	/// <para>Constructs a new <see cref="T:XWindows.InputOutputWidget"/>
	/// instance underneath a specified parent widget.</para>
	/// </summary>
	///
	/// <param name="parent">
	/// <para>The parent of the new widget.</para>
	/// </param>
	///
	/// <param name="x">
	/// <para>The X co-ordinate of the top-left corner of
	/// the new widget.</para>
	/// </param>
	///
	/// <param name="y">
	/// <para>The Y co-ordinate of the top-left corner of
	/// the new widget.</para>
	/// </param>
	///
	/// <param name="width">
	/// <para>The width of the new widget.</para>
	/// </param>
	///
	/// <param name="height">
	/// <para>The height of the new widget.</para>
	/// </param>
	///
	/// <exception cref="T:System.ArgumentNullException">
	/// <para>Raised if <paramref name="parent"/> is <see langword="null"/>.
	/// </para>
	/// </exception>
	///
	/// <exception cref="T:XWindows.XException">
	/// <para>Raised if <paramref name="x"/>, <paramref name="y"/>,
	/// <paramref name="width"/>, or <paramref name="height"/> are
	/// out of range.</para>
	/// </exception>
	///
	/// <exception cref="T.XWindows.XInvalidOperationException">
	/// <para>Raised if <paramref name="parent"/> is disposed, the
	/// root window, or an input-only window.</para>
	/// </exception>
	public InputOutputWidget(Widget parent, int x, int y,
							 int width, int height)
			: base(parent, x, y, width, height,
			       new Color(StandardColor.Background), false, false)
			{
				foreground = new Color(StandardColor.Foreground);
				background = new Color(StandardColor.Background);
			}

	/// <summary>
	/// <para>Constructs a new <see cref="T:XWindows.InputOutputWidget"/>
	/// instance underneath a specified parent widget, with
	/// specified foreground and background colors.</para>
	/// </summary>
	///
	/// <param name="parent">
	/// <para>The parent of the new widget.</para>
	/// </param>
	///
	/// <param name="x">
	/// <para>The X co-ordinate of the top-left corner of
	/// the new widget.</para>
	/// </param>
	///
	/// <param name="y">
	/// <para>The Y co-ordinate of the top-left corner of
	/// the new widget.</para>
	/// </param>
	///
	/// <param name="width">
	/// <para>The width of the new widget.</para>
	/// </param>
	///
	/// <param name="height">
	/// <para>The height of the new widget.</para>
	/// </param>
	///
	/// <param name="foreground">
	/// <para>The foreground color for the widget.</para>
	/// </param>
	///
	/// <param name="background">
	/// <para>The background color for the widget.</para>
	/// </param>
	///
	/// <exception cref="T:System.ArgumentNullException">
	/// <para>Raised if <paramref name="parent"/> is <see langword="null"/>.
	/// </para>
	/// </exception>
	///
	/// <exception cref="T:XWindows.XException">
	/// <para>Raised if <paramref name="x"/>, <paramref name="y"/>,
	/// <paramref name="width"/>, or <paramref name="height"/> are
	/// out of range.</para>
	/// </exception>
	///
	/// <exception cref="T.XWindows.XInvalidOperationException">
	/// <para>Raised if <paramref name="parent"/> is disposed, the
	/// root window, or an input-only window.</para>
	/// </exception>
	public InputOutputWidget(Widget parent, int x, int y,
							 int width, int height,
							 Color foreground, Color background)
			: base(parent, x, y, width, height, background, false, false)
			{
				this.foreground = foreground;
				this.background = background;
			}

	// Internal constructor that is used by "TopLevelWindow" and
	// "OverrideWindow".
	internal InputOutputWidget(Widget parent, int x, int y,
							   int width, int height,
							   Color foreground, Color background,
							   bool rootAllowed, bool overrideRedirect)
			: base(parent, x, y, width, height, background,
				   rootAllowed, overrideRedirect)
			{
				this.foreground = foreground;
				this.background = background;
			}

	/// <summary>
	/// <para>Get or set the foreground color for this widget.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The foreground color to use.</para>
	/// </value>
	///
	/// <remarks>
	/// <para>The foreground color does not become active until the
	/// next time a <see cref="T:XWindows.Canvas"/> object is created
	/// for the widget.  Usually this happens the next time the
	/// widget is repainted.</para>
	/// </remarks>
	public Color Foreground
			{
				get
				{
					return foreground;
				}
				set
				{
					foreground = value;
				}
			}

	/// <summary>
	/// <para>Get or set the background color for this widget.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The background color to use.</para>
	/// </value>
	///
	/// <remarks>
	/// <para>The background color becomes active immediately.  If the
	/// widget is mapped, a repaint will be forced.</para>
	/// </remarks>
	public Color Background
			{
				get
				{
					return background;
				}
				set
				{
					if(background != value)
					{
						try
						{
							IntPtr display = dpy.Lock();
							Xlib.Window window = GetWidgetHandle();
							background = value;
							Xlib.XSetWindowBackground
								(display, window, ToPixel(value));
							if(mapped && AncestorsMapped)
							{
								Xlib.XClearArea(display, window,
												0, 0, (uint)0, (uint)0,
												Xlib.Bool.True);
							}
						}
						finally
						{
							dpy.Unlock();
						}
					}
				}
			}

	/// <summary>
	/// <para>Force a repaint of this widget.</para>
	/// </summary>
	public void Repaint()
			{
				try
				{
					IntPtr display = dpy.Lock();
					if(mapped && AncestorsMapped)
					{
						Xlib.XClearArea(display, GetWidgetHandle(),
										0, 0, (uint)0, (uint)0,
										Xlib.Bool.True);
					}
				}
				finally
				{
					dpy.Unlock();
				}
			}

	/// <summary>
	/// <para>Process a color theme change for this widget.</para>
	/// </summary>
	public override void ThemeChange()
			{
				// Pass the theme change on to all of the child widgets.
				base.ThemeChange();

				// Change the background pixel to match the new theme
				// settings, and then force a repaint on the widget.
				try
				{
					IntPtr display = dpy.Lock();
					Xlib.Window window = GetWidgetHandle();
					Xlib.XSetWindowBackground
						(display, window, ToPixel(background));
					if(mapped && AncestorsMapped)
					{
						Xlib.XClearArea(display, window,
										0, 0, (uint)0, (uint)0,
										Xlib.Bool.True);
					}
				}
				finally
				{
					dpy.Unlock();
				}
			}

	/// <summary>
	/// <para>Event that is raised when the widget needs to be
	/// painted in reponse to an <c>Expose</c> event.</para>
	/// </summary>
	public event PaintEventHandler Paint
			{
				add
				{
					// Add ExposureMask to the widget's select mask if this
					// is the first delegate that was added to the event.
					if(paint == null)
					{
						SelectInput(EventMask.ExposureMask);
					}
					paint += value;
				}
				remove
				{
					// Remove ExposureMask from the widget's select mask if
					// this was the last delegate to be removed from the event.
					paint -= value;
					if(paint == null)
					{
						DeselectInput(EventMask.ExposureMask);
						RemovePendingExpose();
					}
				}
			}

	/// Dispatch an event to this widget.
	internal override void DispatchEvent(ref XEvent xevent)
			{
				switch(xevent.type)
				{
					case EventType.Expose:
					{
						// Process exposures, and call "Paint" when ready.
						if(paint != null)
						{
							// Add the area to the expose region.
							if(exposeRegion == null)
							{
								// This is the first rectangle in an expose.
								exposeRegion = new Region
									(xevent.xexpose.x,
									 xevent.xexpose.y,
									 xevent.xexpose.width,
									 xevent.xexpose.height);

								// Queue this widget for later repainting.
								// We don't do it now or the system will be
								// very slow during opaque window drags.
								dpy.AddPendingExpose(this);
							}
							else
							{
								// This is an extra rectangle in an expose.
								exposeRegion.Union
									(xevent.xexpose.x,
									 xevent.xexpose.y,
									 xevent.xexpose.width,
									 xevent.xexpose.height);
							}
						}
					}
					break;
				}
			}

	// Remove this widget from the pending expose list.
	internal void RemovePendingExpose()
			{
				dpy.RemovePendingExpose(this);
				if(exposeRegion != null)
				{
					exposeRegion.Dispose();
					exposeRegion = null;
				}
			}

	// Process pending exposures on this widget.
	internal void Expose()
			{
				Region region = exposeRegion;
				if(region != null)
				{
					exposeRegion = null;
					if(paint != null)
					{
						Canvas canvas = new Canvas(this);
						canvas.SetClipRegion(region);
						region.Dispose();
						paint(this, canvas);
						canvas.Dispose();
					}
					else
					{
						region.Dispose();
					}
				}
			}

} // class InputOutputWidget

} // namespace XWindows
