/*
 * Pixmap.cs - Basic pixmap handling for X applications.
 *
 * This file is part of the X# library.
 * Copyright (C) 2002  Southern Storm Software, Pty Ltd.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace XWindows
{

using System;

/// <summary>
/// <para>The <see cref="T:XWindows.Pixmap"/> class manages off-screen
/// pixmaps on an X display screen.</para>
/// </summary>
public sealed class Pixmap : Drawable
{
	/// <summary>
	/// <para>Constructs a new <see cref="T:XWindows.Pixmap"/> instance
	/// that represents an off-screen pixmap.  The pixmap is created
	/// on the default screen of the primary display.</para>
	/// </summary>
	///
	/// <param name="width">
	/// <para>The width of the new pixmap.</para>
	/// </param>
	///
	/// <param name="height">
	/// <para>The height of the new pixmap.</para>
	/// </param>
	///
	/// <exception cref="T:XWindows.XException">
	/// <para>The <paramref name="width"/> or <paramref name="height"/>
	/// values are out of range.</para>
	/// </exception>
	public Pixmap(int width, int height)
			: base(XWindows.Display.Primary,
				   XWindows.Display.Primary.DefaultScreenOfDisplay,
				   DrawableKind.Pixmap)
			{
				if(width < 1 || width > 32767 ||
				   height < 1 || height > 32767)
				{
					throw new XException(S._("X_InvalidPixmapSize"));
				}
				try
				{
					IntPtr display = dpy.Lock();
					SetPixmapHandle(Xlib.XCreatePixmap
						(display, (Xlib.Drawable)
							Xlib.XRootWindowOfScreen(screen.screen),
						 (uint)width, (uint)height,
						 (uint)Xlib.XDefaultDepthOfScreen(screen.screen)));
				}
				finally
				{
					dpy.Unlock();
				}
			}

	/// <summary>
	/// <para>Constructs a new <see cref="T:XWindows.Pixmap"/> instance
	/// that represents an off-screen pixmap.  The pixmap is created
	/// on the specified <paramref name="screen"/>.</para>
	/// </summary>
	///
	/// <param name="screen">
	/// <para>The screen upon which to create the new pixmap.</para>
	/// </param>
	///
	/// <param name="width">
	/// <para>The width of the new pixmap.</para>
	/// </param>
	///
	/// <param name="height">
	/// <para>The height of the new pixmap.</para>
	/// </param>
	///
	/// <exception cref="T:System.ArgumentNullException">
	/// <para>The <paramref name="screen"/> value is <see langword="null"/>.
	/// </para>
	/// </exception>
	///
	/// <exception cref="T:XWindows.XException">
	/// <para>The <paramref name="width"/> or <paramref name="height"/>
	/// values are out of range.</para>
	/// </exception>
	public Pixmap(Screen screen, int width, int height)
			: base(GetDisplay(screen), screen, DrawableKind.Pixmap)
			{
				if(width < 1 || width > 32767 ||
				   height < 1 || height > 32767)
				{
					throw new XException(S._("X_InvalidPixmapSize"));
				}
				try
				{
					IntPtr display = dpy.Lock();
					SetPixmapHandle(Xlib.XCreatePixmap
						(display, (Xlib.Drawable)
							Xlib.XRootWindowOfScreen(screen.screen),
						 (uint)width, (uint)height,
						 (uint)Xlib.XDefaultDepthOfScreen(screen.screen)));
				}
				finally
				{
					dpy.Unlock();
				}
			}

	// Get the display value from a specified screen, and check for null.
	private static Display GetDisplay(Screen screen)
			{
				if(screen == null)
				{
					throw new ArgumentNullException("screen");
				}
				return screen.DisplayOfScreen;
			}

	/// <summary>
	/// <para>Destroy this pixmap if it is currently active.</para>
	/// </summary>
	public override void Destroy()
			{
				try
				{
					IntPtr d = dpy.Lock();
					if(handle != Xlib.Drawable.Zero)
					{
						Xlib.XFreePixmap(d, (Xlib.Pixmap)handle);
						handle = Xlib.Drawable.Zero;
					}
				}
				finally
				{
					dpy.Unlock();
				}
			}

} // class Pixmap

} // namespace XWindows
