/*
 * Widget.cs - Basic widget handling for X applications.
 *
 * This file is part of the X# library.
 * Copyright (C) 2002  Southern Storm Software, Pty Ltd.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace XWindows
{

using System;
using System.Collections;
using XWindows.Types;
using XWindows.Events;

/// <summary>
/// <para>The <see cref="T:XWindows.Widget"/> class manages widget
/// windows on an X display screen.</para>
///
/// <para>This is an abstract class.  Instantiate or inherit one of
/// the classes <see cref="T:XWindows.InputOutputWidget"/>,
/// <see cref="T:XWindows.InputOnlyWidget"/>, or
/// <see cref="T:XWindows.TopLevelWindow"/> in user applications.</para>
/// </summary>
public abstract class Widget : Drawable, ICollection, IEnumerable
{
	// Internal state.
	internal int x, y;
	internal bool mapped;
	internal int layer;
	private CursorType cursor;
	private bool autoMapChildren;
	private Widget parent;
	private Widget topChild;
	private Widget nextAbove;
	private Widget nextBelow;
	private EventMask eventMask;

	// Constructor.
	internal Widget(Display dpy, Screen screen,
					DrawableKind kind, Widget parent)
			: base(dpy, screen, kind)
			{
				// Set the initial widget properties.
				cursor = CursorType.XC_inherit_parent;
				autoMapChildren = true;

				// Insert this widget into the widget tree under its parent.
				this.parent = parent;
				this.topChild = null;
				this.nextAbove = null;
				if(parent != null)
				{
					nextBelow = parent.topChild;
					if(parent.topChild != null)
					{
						parent.topChild.nextAbove = this;
					}
					parent.topChild = this;
				}
				else
				{
					nextBelow = null;
				}
				this.eventMask = 0;
			}

	// Detach this widget from its position in the widget tree.
	internal void Detach()
			{
				// Detach ourselves from our siblings and parent.
				if(nextBelow != null)
				{
					nextBelow.nextAbove = nextAbove;
				}
				if(nextAbove != null)
				{
					nextAbove.nextBelow = nextBelow;
				}
				else if(parent != null)
				{
					parent.topChild = nextBelow;
				}

				// Detach ourselves from our children.
				Widget current, next;
				current = topChild;
				while(current != null)
				{
					next = current.nextBelow;
					current.parent = null;
					current.nextAbove = null;
					current.nextBelow = null;
					current = next;
				}

				// Clear all of our link fields.
				parent = null;
				topChild = null;
				nextAbove = null;
				nextBelow = null;
			}

	// Disassociate this widget instance and all of its children
	// from their X window handles, as the mirror copy in the X
	// server has been lost.
	internal void Disassociate()
			{
				if(handle != Xlib.Drawable.Zero)
				{
					dpy.handleMap.Remove((int)handle);
				}
				if(this is InputOutputWidget)
				{
					((InputOutputWidget)this).RemovePendingExpose();
				}
				handle = Xlib.Drawable.Zero;
				Widget child = topChild;
				while(child != null)
				{
					child.Disassociate();
					child = child.nextBelow;
				}
			}

	/// <summary>
	/// <para>Destroy this widget if it is currently active.</para>
	/// </summary>
	public override void Destroy()
			{
				try
				{
					IntPtr d = dpy.Lock();
					if(handle != Xlib.Drawable.Zero)
					{
						Xlib.XDestroyWindow(d, (Xlib.Window)handle);
						Disassociate();
					}
				}
				finally
				{
					dpy.Unlock();
				}
			}

	/// <summary>
	/// <para>Get the X position of this widget relative to its parent.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The X position of this widget in pixels.</para>
	/// </value>
	public int X
			{
				get
				{
					return x;
				}
			}

	/// <summary>
	/// <para>Get the Y position of this widget relative to its parent.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The Y position of this widget in pixels.</para>
	/// </value>
	public int Y
			{
				get
				{
					return y;
				}
			}

	/// <summary>
	/// <para>Determine if this widget is currently mapped.</para>
	/// </summary>
	///
	/// <value>
	/// <para>Returns <see langword="true"/> if the widget is mapped;
	/// <see langword="false"/> otherwise.</para>
	/// </value>
	///
	/// <remarks>
	/// <para>A mapped widget may still be invisible if it is mapped
	/// because its parent is unmapped, because it is covered by
	/// a sibling widget, or because its co-ordinates are outside
	/// the range of its parent widget.</para>
	///
	/// <para>Setting this property is equivalent to calling either
	/// <c>Map</c> or <c>Unmap</c>.</para>
	/// </remarks>
	public bool IsMapped
			{
				get
				{
					return mapped;
				}
				set
				{
					if(value)
					{
						Map();
					}
					else
					{
						Unmap();
					}
				}
			}

	/// <summary>
	/// <para>Determine if all of the ancestor widgets are mapped.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The value is <see langword="true"/> if all of the ancestors
	/// of this widget are mapped; <see langword="false"/> otherwise.</para>
	/// </value>
	public bool AncestorsMapped
			{
				get
				{
					Widget widget = parent;
					while(widget != null)
					{
						if(!(widget.mapped))
						{
							return false;
						}
						widget = widget.parent;
					}
					return true;
				}
			}

	/// <summary>
	/// <para>Get or set the cursor that is associated with this widget.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The cursor shape to set for the widget.  If the value is
	/// <c>CursorType.XC_inherit_parent</c>, then the widget inherits the
	/// cursor that is set on the parent widget.</para>
	/// </value>
	public virtual CursorType Cursor
			{
				get
				{
					return cursor;
				}
				set
				{
					try
					{
						IntPtr display = dpy.Lock();
						if(cursor != value)
						{
							cursor = value;
							if(value == CursorType.XC_inherit_parent)
							{
								// Revert to inheriting our parent's cursor.
								Xlib.XUndefineCursor
									(display, GetWidgetHandle());
							}
							else
							{
								// Change our cursor to a pre-defined shape.
								Xlib.XDefineCursor
									(display, GetWidgetHandle(),
									 dpy.GetCursor(value));
							}
						}
					}
					finally
					{
						dpy.Unlock();
					}
				}
			}

	/// <summary>
	/// <para>Get or set the flag that indicates if child widgets
	/// should be automatically mapped when they are created.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The value is <see langword="true"/> to automatically map
	/// children; <see langword="false"/> otherwise.</para>
	/// </value>
	///
	/// <remarks>
	/// <para>Normally, child widgets are automatically mapped when
	/// they are created, to avoid the need for the application to
	/// explicitly map each child as it is created.</para>
	///
	/// <para>By setting this flag to <see langword="false"/>, the
	/// program can control when children are mapped, which may be
	/// useful in certain circumstances (e.g. widgets that are
	/// hidden unless explicitly called for).</para>
	///
	/// <para>The root window has its <c>AutoMapChildren</c> flag set
	/// to <see langword="false"/> by default, and this cannot be
	/// changed.  This allows the application to fully create the
	/// widget tree before it is mapped to the screen.</para>
	/// </remarks>
	public bool AutoMapChildren
			{
				get
				{
					return autoMapChildren;
				}
				set
				{
					// Ignore the request if this is the root window.
					if(!(this is RootWindow))
					{
						autoMapChildren = value;
					}
				}
			}

	/// <summary>
	/// <para>Get the parent of this widget.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The <see cref="T:XWindows.Widget"/> instance that
	/// corresponds to this widget's parent, or <see langword="null"/>
	/// if this widget is an instance of <see cref="T:XWindows.RootWindow"/>.
	/// </para>
	/// </value>
	public Widget Parent
			{
				get
				{
					return parent;
				}
			}

	/// <summary>
	/// <para>Get the next widget above this one in stacking order.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The <see cref="T:XWindows.Widget"/> instance that
	/// corresponds to the next widget above this one in stacking order,
	/// or <see langword="null"/> if this is the top-most widget.</para>
	/// </value>
	///
	/// <remarks>
	/// <para>The actual ordering of top-level widgets may not match the
	/// value returned from this property because the window manager has
	/// changed the order itself based on user requests.</para>
	/// </remarks>
	public Widget NextAbove
			{
				get
				{
					return nextAbove;
				}
			}

	/// <summary>
	/// <para>Get the next widget below this one in stacking order.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The <see cref="T:XWindows.Widget"/> instance that
	/// corresponds to the next widget below this one in stacking order,
	/// or <see langword="null"/> if this is the bottom-most widget.</para>
	/// </value>
	///
	/// <remarks>
	/// <para>The actual ordering of top-level widgets may not match the
	/// value returned from this property because the window manager has
	/// changed the order itself based on user requests.</para>
	/// </remarks>
	public Widget NextBelow
			{
				get
				{
					return nextBelow;
				}
			}

	/// <summary>
	/// <para>Get the top-most child widget in stacking order that
	/// has this widget as a parent.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The <see cref="T:XWindows.Widget"/> instance that
	/// corresponds to the top-most child widget in stacking order,
	/// or <see langword="null"/> if there are no child widgets.</para>
	/// </value>
	///
	/// <remarks>
	/// <para>The actual ordering of top-level widgets may not match the
	/// value returned from this property because the window manager has
	/// changed the order itself based on user requests.</para>
	/// </remarks>
	public Widget TopChild
			{
				get
				{
					return topChild;
				}
			}

	/// <summary>
	/// <para>Get or set the stacking layer that this widget
	/// resides in.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The stacking layer value.</para>
	/// </value>
	///
	/// <remarks>
	/// <para>Widgets in higher layers always appear above widgets in
	/// lower layers.  This class will attempt to keep the widgets in
	/// the correct order with respect to each other.</para>
	///
	/// <para>The value zero corresponds to the "normal" widget layer.
	/// Negative layers appear below the normal layer and positive layers
	/// appear above the normal layer.</para>
	///
	/// <para>The actual ordering of top-level widgets may not match the
	/// value returned from this property because the window manager has
	/// changed the order itself based on user requests, or because the
	/// window manager does not support widget layering.</para>
	/// </remarks>
	public int Layer
			{
				get
				{
					return layer;
				}
				set
				{
					Widget child;
					int origLayer = layer;
					layer = value;
					if(parent != null)
					{
						if(origLayer < value)
						{
							// Push the child further down the parent's stack.
							if(nextBelow == null ||
							   nextBelow.layer <= value)
							{
								return;
							}
							child = nextBelow;
							while(child.nextBelow != null &&
							      child.nextBelow.layer > value)
							{
								child = child.nextBelow;
							}
							MoveToBelow(child);
							RepositionBelow(child);
						}
						else if(origLayer > value)
						{
							// Raise the child further up the parent's stack.
							if(nextAbove == null ||
							   nextAbove.layer >= value)
							{
								return;
							}
							child = nextAbove;
							while(child.nextAbove != null &&
							      child.nextAbove.layer < value)
							{
								child = child.nextAbove;
							}
							MoveToAbove(child);
							RepositionAbove(child);
						}
					}
				}
			}

	// Move this widget to below one of its siblings.
	private void MoveToBelow(Widget sibling)
			{
				// Detach ourselves from the widget tree.
				if(nextAbove != null)
				{
					nextAbove.nextBelow = nextBelow;
				}
				else
				{
					parent.topChild = nextBelow;
				}
				if(nextBelow != null)
				{
					nextBelow.nextAbove = nextAbove;
				}

				// Re-insert at the new position.
				nextAbove = sibling;
				nextBelow = sibling.nextBelow;
				if(nextBelow != null)
				{
					nextBelow.nextAbove = this;
				}
				sibling.nextBelow = this;
			}

	// Move this widget to above one of its siblings.
	private void MoveToAbove(Widget sibling)
			{
				// Detach ourselves from the widget tree.
				if(nextAbove != null)
				{
					nextAbove.nextBelow = nextBelow;
				}
				else
				{
					parent.topChild = nextBelow;
				}
				if(nextBelow != null)
				{
					nextBelow.nextAbove = nextAbove;
				}

				// Re-insert at the new position.
				nextAbove = sibling.nextAbove;
				nextBelow = sibling;
				if(nextAbove != null)
				{
					nextAbove.nextBelow = this;
				}
				else
				{
					parent.topChild = this;
				}
				sibling.nextAbove = this;
			}

	// Reposition this widget below one of its siblings.
	private void RepositionBelow(Widget child)
			{
				try
				{
					IntPtr display = dpy.Lock();
					/* eve include  = new XWindowChanges() */
					XWindowChanges changes = new XWindowChanges();
					changes.sibling = child.GetWidgetHandle();
					changes.stack_mode = 1;		/* Below */
					Xlib.XConfigureWindow
							(display, GetWidgetHandle(),
						     (uint)(ConfigureWindowMask.CWSibling |
							 	    ConfigureWindowMask.CWStackMode),
							 ref changes);
				}
				finally
				{
					dpy.Unlock();
				}
			}

	// Reposition this widget above one of its siblings.
	private void RepositionAbove(Widget child)
			{
				try
				{
					IntPtr display = dpy.Lock();
					/* eve include  = new XWindowChanges() */
					XWindowChanges changes = new XWindowChanges() ;
					changes.sibling = child.GetWidgetHandle();
					changes.stack_mode = 0;		/* Above */
					Xlib.XConfigureWindow
							(display, GetWidgetHandle(),
						     (uint)(ConfigureWindowMask.CWSibling |
							 	    ConfigureWindowMask.CWStackMode),
							 ref changes);
				}
				finally
				{
					dpy.Unlock();
				}
			}

	// Adjust the position and/or size of this widget.
	private void AdjustPositionAndSize(IntPtr display, int newX, int newY,
									   int newWidth, int newHeight)
			{
				// Make sure that the values are in range.
				if(newX < -32768 || newX > 32767 ||
				   newY < -32768 || newY > 32767)
				{
					throw new XException(S._("X_InvalidPosition"));
				}
				else if(newWidth > 32767 || newHeight > 32767)
				{
					throw new XException(S._("X_InvalidSize"));
				}

				// Send requests to the X server to update its state.
				if(newX != x || newY != y)
				{
					if(newWidth != width || newHeight != height)
					{
						Xlib.XMoveResizeWindow(display, GetWidgetHandle(),
										       newX, newY, (uint)newWidth,
											   (uint)newHeight);
					}
					else
					{
						Xlib.XMoveWindow(display, GetWidgetHandle(),
										 newX, newY);
					}
				}
				else if(newWidth != width || newHeight != height)
				{
					Xlib.XResizeWindow(display, GetWidgetHandle(),
									   (uint)newWidth, (uint)newHeight);
				}

				// Record the new widget information locally.
				x = newX;
				y = newY;
				width = newWidth;
				height = newHeight;
			}

	/// <summary>
	/// <para>Move this widget to a new location relative to its parent.</para>
	/// </summary>
	///
	/// <param name="x">
	/// <para>The X co-ordinate of the new top-left widget corner.</para>
	/// </param>
	///
	/// <param name="y">
	/// <para>The Y co-ordinate of the new top-left widget corner.</para>
	/// </param>
	///
	/// <exception cref="T:XWindows.XException">
	/// <para>Raised if <paramref name="x"/> or <paramref name="y"/>
	/// is out of range.</para>
	/// </exception>
	public virtual void Move(int x, int y)
			{
				try
				{
					IntPtr display = dpy.Lock();
					if(x != this.x || y != this.y)
					{
						AdjustPositionAndSize(display, x, y,
											  this.width, this.height);
						if(Moved != null)
						{
							Moved(this, x, y);
						}
					}
				}
				finally
				{
					dpy.Unlock();
				}
			}

	/// <summary>
	/// <para>Resize this widget to a new sie.</para>
	/// </summary>
	///
	/// <param name="width">
	/// <para>The new width for the widget.</para>
	/// </param>
	///
	/// <param name="height">
	/// <para>The new width for the widget.</para>
	/// </param>
	///
	/// <exception cref="T:XWindows.XException">
	/// <para>Raised if <paramref name="width"/> or <paramref name="height"/>
	/// is out of range.</para>
	/// </exception>
	public virtual void Resize(int width, int height)
			{
				if(width < 1 || height < 1 ||
				   !ValidateSize(width, height))
				{
					throw new XException(S._("X_InvalidSize"));
				}
				try
				{
					IntPtr display = dpy.Lock();
					if(width != this.width || height != this.height)
					{
						AdjustPositionAndSize(display, this.x, this.y,
											  width, height);
						if(Resized != null)
						{
							Resized(this, width, height);
						}
					}
				}
				finally
				{
					dpy.Unlock();
				}
			}

	/// <summary>
	/// <para>Validate a widget size change request.</para>
	/// </summary>
	///
	/// <param name="width">
	/// <para>The widget width to be validated.</para>
	/// </param>
	///
	/// <param name="height">
	/// <para>The widget height to be validated.</para>
	/// </param>
	///
	/// <returns>
	/// <para>Returns <see langword="true"/> if the size is acceptable;
	/// <see langword="false"/> otherwise.</para>
	/// </returns>
	///
	/// <remarks>
	/// <para>The implementation in the <see cref="T:XWindows.Widget"/>
	/// base class always returns <see langword="true"/>.</para>
	/// </remarks>
	protected virtual bool ValidateSize(int width, int height)
			{
				return true;
			}

	/// <summary>
	/// <para>Map this widget to the screen.</para>
	/// </summary>
	public virtual void Map()
			{
				try
				{
					IntPtr display = dpy.Lock();
					if(!mapped)
					{
						Xlib.XMapWindow(display, GetWidgetHandle());
						mapped = true;
						if(MapState != null)
						{
							MapState(this, true);
						}
					}
				}
				finally
				{
					dpy.Unlock();
				}
			}

	/// <summary>
	/// <para>Unmap this widget from the screen.</para>
	/// </summary>
	public virtual void Unmap()
			{
				try
				{
					IntPtr display = dpy.Lock();
					if(mapped)
					{
						Xlib.XUnmapWindow(display, GetWidgetHandle());
						mapped = false;
						if(MapState != null)
						{
							MapState(this, false);
						}
					}
				}
				finally
				{
					dpy.Unlock();
				}
			}

	// Record a change in map state from a subclass.
	internal void MapStateChanged()
			{
				if(MapState != null)
				{
					MapState(this, mapped);
				}
			}

	/// <summary>
	/// <para>Process a color theme change for this widget.</para>
	/// </summary>
	public virtual void ThemeChange()
			{
				Widget child = topChild;
				while(child != null)
				{
					child.ThemeChange();
					child = child.nextBelow;
				}
			}

	/// <summary>
	/// <para>Copy the children of this widget into an array.</para>
	/// </summary>
	///
	/// <param name="array">
	/// <para>The array to copy the children to.</para>
	/// </param>
	///
	/// <param name="index">
	/// <para>The index within the array to being copying.</para>
	/// </param>
	///
	/// <remarks>
	/// <para>This method implements the
	/// <see cref="T:System.Collections.ICollection"/> interface.</para>
	/// </remarks>
	public void CopyTo(Array array, int index)
			{
				Widget child = topChild;
				while(child != null)
				{
					array.SetValue(child, index++);
					child = child.nextBelow;
				}
			}

	/// <summary>
	/// <para>Get the number of children of this widget.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The number of children of this widget.</para>
	/// </value>
	///
	/// <remarks>
	/// <para>This property implements the
	/// <see cref="T:System.Collections.ICollection"/> interface.</para>
	/// </remarks>
	public int Count
			{
				get
				{
					Widget child = topChild;
					int count = 0;
					while(child != null)
					{
						++count;
						child = child.nextBelow;
					}
					return count;
				}
			}

	/// <summary>
	/// <para>Determine if this collection is synchronized.</para>
	/// </summary>
	///
	/// <value>
	/// <para>Always returns <see langword="false"/>.</para>
	/// </value>
	///
	/// <remarks>
	/// <para>This property implements the
	/// <see cref="T:System.Collections.ICollection"/> interface.</para>
	/// </remarks>
	public bool IsSynchronized
			{
				get
				{
					return false;
				}
			}

	/// <summary>
	/// <para>Get the synchronization root for this collection.</para>
	/// </summary>
	///
	/// <value>
	/// <para>Always returns <see langword="this"/>.</para>
	/// </value>
	///
	/// <remarks>
	/// <para>This property implements the
	/// <see cref="T:System.Collections.ICollection"/> interface.</para>
	/// </remarks>
	public Object SyncRoot
			{
				get
				{
					return this;
				}
			}

	/// <summary>
	/// <para>Get an enumerator for the children of this widget.</para>
	/// </summary>
	///
	/// <returns>
	/// <para>Returns the enumerator instance.</para>
	/// </returns>
	///
	/// <remarks>
	/// <para>This method implements the
	/// <see cref="T:System.Collections.IEnumerable"/> interface.</para>
	/// </remarks>
	public IEnumerator GetEnumerator()
			{
				return new WidgetEnumerator(this);
			}

	// Private enumerator class for "Widget.GetEnumerator()".
	private sealed class WidgetEnumerator : IEnumerator
	{
		// Internal state.
		private Widget parent;
		private Widget child;
		private bool atStart;

		// Constructor.
		public WidgetEnumerator(Widget parent)
				{
					this.parent = parent;
					this.child = null;
					this.atStart = true;
				}

		// Move to the next element in the enumeration order.
		public bool MoveNext()
				{
					if(atStart)
					{
						child = parent.topChild;
						atStart = false;
					}
					else
					{
						child = child.nextBelow;
					}
					return (child != null);
				}

		// Reset the enumeration order.
		public void Reset()
				{
					child = null;
					atStart = true;
				}

		// Get the current value in the enumeration.
		public Object Current
				{
					get
					{
						if(child == null)
						{
							throw new InvalidOperationException
								(S._("X_BadEnumeratorPosition"));
						}
						return child;
					}
				}

	} // class WidgetEnumerator

	/// Dispatch an event to this widget.
	internal virtual void DispatchEvent(ref XEvent xevent)
			{
				// Nothing to do here: overridden by subclasses.
			}

	// Select for particular events on this widget.
	//{eve}protected
	internal void SelectInput(EventMask mask)
			{
				try
				{
					IntPtr display = dpy.Lock();
					EventMask newMask = (eventMask | mask);
					Xlib.Window handle = GetWidgetHandle();
					if(newMask != eventMask)
					{
						eventMask = newMask;
						Xlib.XSelectInput(display, handle, (int)newMask);
					}
				}
				finally
				{
					dpy.Unlock();
				}
			}

	// Deselect particular events on this widget.
	//{eve}protected
	internal void DeselectInput(EventMask mask)
			{
				try
				{
					IntPtr display = dpy.Lock();
					EventMask newMask = (eventMask & ~mask);
					Xlib.Window handle = GetWidgetHandle();
					if(newMask != eventMask)
					{
						eventMask = newMask;
						Xlib.XSelectInput(display, handle, (int)newMask);
					}
				}
				finally
				{
					dpy.Unlock();
				}
			}

	/// <summary>
	/// <para>Event that is raised when the widget is moved to a
	/// new position.</para>
	/// </summary>
	public event MovedEventHandler Moved;

	/// <summary>
	/// <para>Event that is raised when the widget is resized to
	/// a new size.</para>
	/// </summary>
	public event ResizedEventHandler Resized;

	/// <summary>
	/// <para>Event that is raised when the widget's map state changes.</para>
	/// </summary>
	public event MapStateEventHandler MapState;

} // class Widget

} // namespace XWindows
