/*
	FreeRTOS V1.2.0 - Copyright (C) 2003 Richard Barry.

	Licensed under the Open Software License version 1.1

	The file list.c and related documentation is part of the FreeRTOS
	distribution.  Any and all data files, source code and documentation
	included in the FreeRTOS distribution are the exclusive property of
	Richard Barry.

	See www.FreeRTOS.org for documentation, license and warranty details.
*/

#include <stdlib.h>
#include "projdefs.h"
#include "portable.h"
#include "list.h"

#define listMAX_ITEM_VALUE		( ( unsigned portLONG ) 0xFFFFFFFFUL )

/*-----------------------------------------------------------
 * PUBLIC LIST API documented in list.h
 *----------------------------------------------------------*/

void vListInitialise( volatile xList *pxList )
{
	/* The list structure contains a list item which is used to mark the
	end of the list.  To initialise the list the list end is inserted
	as the only list entry. */
	pxList->pxHead = &( pxList->xListEnd );
	pxList->pxIndex = pxList->pxHead;

	/* The list end value is the highest possible value in the list to
	ensure it remains at the end of the list. */
	pxList->xListEnd.ulItemValue = listMAX_ITEM_VALUE;

	/* The list end next and previous pointers point to itself so we know
	when the list is empty. */
	pxList->xListEnd.pxNext = &( pxList->xListEnd );

	/* The list head will never get used and has no owner. */
	pxList->xListEnd.pvOwner = NULL;

	/* Make sure the marker items are not mistaken for being on a list. */
	vListInitialiseItem( &( pxList->xListEnd ) );
}
/*-----------------------------------------------------------*/

void vListInitialiseItem( volatile xListItem *pxItem )
{
	/* Make sure the list item is not recorded as being on a list. */
	pxItem->pvContainer = NULL;
}
/*-----------------------------------------------------------*/

void vListInsertEnd( volatile xList *pxList, volatile xListItem *pxNewListItem )
{
	/* Insert a new list item into pxList, but rather than sort the list,
	makes the new list item the last item to be removed by a call to
	pvListGetOwnerOfNextEntry.  This means it has to be the item pointed to by
	the pxIndex member. */
	pxNewListItem->pxNext = pxList->pxIndex->pxNext;
	pxList->pxIndex->pxNext = pxNewListItem;
	pxList->pxIndex = pxNewListItem;

	/* Remember which list the item is in. */
	pxNewListItem->pvContainer = ( void * ) pxList;
}
/*-----------------------------------------------------------*/

void vListInsert( volatile xList *pxList, volatile xListItem *pxNewListItem )
{
volatile xListItem *pxIterator;
register unsigned portLONG ulValueOfInsertion;

	/* Insert the new list item into the list, sorted in ulListItem order. */
	ulValueOfInsertion = pxNewListItem->ulItemValue;

	/* If the list already contains a list item with the same item value then
	the new list item should be placed after it.  This ensures that TCB's which
	are stored in ready lists (all of which have the same ulListItem member)
	get an equal share of the CPU.  However, if the ulItemValue is the same as
	the back marker the iteration loop below will not end.  This means we need
	to guard against this by checking the value first and modifying the
	algorithm slightly if necessary. */
	if( ulValueOfInsertion == listMAX_ITEM_VALUE )
	{
		for( pxIterator = pxList->pxHead; pxIterator->pxNext->ulItemValue < ulValueOfInsertion; pxIterator = pxIterator->pxNext )
		{
			/* There is nothing to do here, we are just iterating to the
			wanted insertion position. */
		}
	}
	else
	{
		for( pxIterator = pxList->pxHead; pxIterator->pxNext->ulItemValue <= ulValueOfInsertion; pxIterator = pxIterator->pxNext )
		{
			/* There is nothing to do here, we are just iterating to the
			wanted insertion position. */
		}
	}

	pxNewListItem->pxNext = pxIterator->pxNext;
	pxIterator->pxNext = pxNewListItem;

	/* Remember which list the item is in.  This allows fast removal of the
	item later. */
	pxNewListItem->pvContainer = ( void * ) pxList;
}
/*-----------------------------------------------------------*/

void vListRemove( volatile xListItem *pxItemToRemove )
{
volatile xListItem *pxIterator;
xList *pxList;

	/* The list item knows which list it is in.  Obtain the list from the list
	item. */
	pxList = ( xList * ) pxItemToRemove->pvContainer;
	if( pxList != NULL )
	{
		if( !listLIST_IS_EMPTY( pxList ) )
		{
			/* Find the list item in the list. */
			for( pxIterator = pxList->pxHead; pxIterator->pxNext != pxItemToRemove; pxIterator = pxIterator->pxNext )
			{
				/* Again, nothing to do here, we are just iterating to the wanted
				list item.  We can put a check here to ensure we don't loop forever
				if required.

				CURRENTLY THIS ASSUMES THE ITEM IS IN THE LIST - IF NOT, BIG
				TROUBLE! */
			}

			/* Remove the list item, ensuring the structure of the list is left
			in tact. */
			pxIterator->pxNext = pxIterator->pxNext->pxNext;

			if( pxList->pxIndex == pxItemToRemove )
			{
				pxList->pxIndex = pxIterator;
			}
		}

		pxItemToRemove->pvContainer = NULL;
	}
}
/*-----------------------------------------------------------*/

volatile void *pvListGetOwnerOfNextEntry( volatile xList *pxList )
{
	if( !listLIST_IS_EMPTY( pxList ) )
	{
		/* Increment the index to the next item and return the item, ensuring
		we don't return the marker used at the end of the list. */
		pxList->pxIndex = pxList->pxIndex->pxNext;
#ifdef __GNUrC__
		if( (pxList->pxIndex - pxList->pxHead) == NULL )
#else
		if( pxList->pxIndex == pxList->pxHead )
#endif
		{
			pxList->pxIndex = pxList->pxIndex->pxNext;
		}
		return pxList->pxIndex->pvOwner;
	}
	return NULL;
}
/*-----------------------------------------------------------*/

volatile void *pvListGetOwnerOfHeadEntry( const volatile xList * const pxList )
{
	/* The pxHead member actually points to the marker to the end of the list.
	To get the true head we therefore return the list item pointed to by the
	pxHead member. */
	if( !listLIST_IS_EMPTY( pxList ) )
	{
		return pxList->pxHead->pxNext->pvOwner;
	}
	return NULL;
}
/*-----------------------------------------------------------*/

