/*
	FreeRTOS V1.2.0 - Copyright (C) 2003 Richard Barry.

	Licensed under the Open Software License version 1.1

	The file port.c and related documentation is part of the FreeRTOS
	distribution.  Any and all data files, source code and documentation
	included in the FreeRTOS distribution are the exclusive property of
	Richard Barry.

	See www.FreeRTOS.org for documentation, license and warranty details.
*/
#include <stdlib.h>
#include <io.h>
#include <signal.h>

#include "portmacro.h"
#include "projdefs.h"
#include "portable.h"
#include "task.h"

/*-----------------------------------------------------------
 * Implementation of functions defined in portable.h for the AVR port.
 *----------------------------------------------------------*/

/* Start tasks with interrupts enables. */
#define portFLAGS_INT_ENABLED					( ( portSTACK_TYPE ) 0x80 )

/* Hardware constants for timer 1. */
#define portCLEAR_COUNTER_ON_MATCH				( ( unsigned portCHAR ) 0x08 )
#define portPRESCALE_256						( ( unsigned portCHAR ) 0x04 )
#define portCLOCK_PRESCALER						( ( unsigned portLONG ) 256 )
#define portCOMPARE_MATCH_A_INTERRUPT_ENABLE	( ( unsigned portCHAR ) 0x10 )

/*-----------------------------------------------------------*/

/* We require the address of the pxCurrentTCB variable, but don't want to know
any details of its type. */
// typedef void tskTCB;
extern volatile tskTCB * volatile pxCurrentTCB;

/*-----------------------------------------------------------*/

/*
 * Macro to save all the general purpose registers, the save the stack pointer
 * into the TCB.
 *
 * The first thing we do is save the flags then disable interrupts.  This is to
 * guard our stack against having a context switch interrupt after we have already
 * pushed the registers onto the stack - causing the 32 registers to be on the
 * stack twice.
 *
 * r1 is set to zero as the compiler expects it to be thus, however some
 * of the math routines make use of R1.
 *
 * The interrupts will have been disabled during the call to portSAVE_CONTEXT()
 * so we need not worry about reading/writing to the stack pointer.
 */


#define portSAVE_CONTEXT()									\
	asm ("push	r2"	);										\
	asm ("push	r4"	);										\
	asm ("push	r5"	);										\
	asm ("push	r6"	);										\
	asm ("push	r7"	);										\
	asm ("push	r8"	);										\
	asm ("push	r9"	);										\
	asm ("push	r10");										\
	asm ("push	r11");										\
	asm ("push	r12");										\
	asm ("push	r13");										\
	asm ("push	r14");										\
	asm ("push	r15");										\
	asm ("mov	%0,r15"::"m"(pxCurrentTCB))	;				\
	asm ("mov	r1,@r15");


/*
 * Opposite to portSAVE_CONTEXT().  Interrupts will have been disabled during
 * the context save so we can write to the stack pointer.
 */

/*
	In Abhaengigkeit von der CPU (HW-Multiplikation oder nicht, die
	Register des Multiplizierers retten
*/

#define portRESTORE_CONTEXT()		\
	asm ("mov	%0,r15"::"m"(pxCurrentTCB));	\
	asm ("mov	@r15,r1			");	\
	asm ("pop	r15				");	\
	asm ("pop	r14				");	\
	asm ("pop	r13				");	\
	asm ("pop	r12				");	\
	asm ("pop	r11				");	\
	asm ("pop	r10				");	\
	asm ("pop	r9				");	\
	asm ("pop	r8				");	\
	asm ("pop	r7				");	\
	asm ("pop	r6				");	\
	asm ("pop	r5				");	\
	asm ("pop	r4				"); \
	asm ("pop   r2				");


/*-----------------------------------------------------------*/

/*
 * Perform hardware setup to enable ticks from timer 1, compare match A.
 */
static void prvSetupTimerInterrupt( void );
/*-----------------------------------------------------------*/

/*
 * See header file for description.
 */
portSTACK_TYPE *pxPortInitialiseStack( portSTACK_TYPE *pxTopOfStack, pdTASK_CODE pxCode, void *pvParameters )
{

	/* The start of the task code will be popped off the stack last, so place
	it on first. */

	pxTopOfStack += portSTACK_GROWTH;
	*pxTopOfStack = ( portSTACK_TYPE ) ( pxCode);

	pxTopOfStack += portSTACK_GROWTH;
	*pxTopOfStack = ( portSTACK_TYPE ) 0x00; /* R2 GIE = 1	  */


	/* Now the remaining registers. */

/*	Der Konstantengenerator laesst sich nicht poppen. Der Stackpointer bleibt
	dabei unveraendert.

	pxTopOfStack += portSTACK_GROWTH;
	*pxTopOfStack = ( portSTACK_TYPE ) 0x03;	// R3 Konstantengenerator
*/
	pxTopOfStack += portSTACK_GROWTH;
	*pxTopOfStack = ( portSTACK_TYPE ) 0x04;	/* R4 */
	pxTopOfStack += portSTACK_GROWTH;
	*pxTopOfStack = ( portSTACK_TYPE ) 0x05;	/* R5 */
	pxTopOfStack += portSTACK_GROWTH;
	*pxTopOfStack = ( portSTACK_TYPE ) 0x06;	/* R6 */
	pxTopOfStack += portSTACK_GROWTH;
	*pxTopOfStack = ( portSTACK_TYPE ) 0x07;	/* R7 */
	pxTopOfStack += portSTACK_GROWTH;
	*pxTopOfStack = ( portSTACK_TYPE ) 0x08;	/* R8 */
	pxTopOfStack += portSTACK_GROWTH;
	*pxTopOfStack = ( portSTACK_TYPE ) 0x09;	/* R9 */
	pxTopOfStack += portSTACK_GROWTH;
	*pxTopOfStack = ( portSTACK_TYPE ) 0x10;	/* R10 */
	pxTopOfStack += portSTACK_GROWTH;
	*pxTopOfStack = ( portSTACK_TYPE ) 0x11;	/* R11 */
	pxTopOfStack += portSTACK_GROWTH;
	*pxTopOfStack = ( portSTACK_TYPE ) 0x12;	/* R12 */
	pxTopOfStack += portSTACK_GROWTH;
	*pxTopOfStack = ( portSTACK_TYPE ) 0x13;	/* R13 */
	pxTopOfStack += portSTACK_GROWTH;
	*pxTopOfStack = ( portSTACK_TYPE ) 0x14;	/* R14 */

/*  R15 zeigt auf den Parametersatz der Task
	pxTopOfStack += portSTACK_GROWTH;
	*pxTopOfStack = ( portSTACK_TYPE ) 0x15;	// R15
*/
	/* Place the parameter on the stack in the expected location. */

	pxTopOfStack += portSTACK_GROWTH;
	*pxTopOfStack = ( portSTACK_TYPE ) ( pvParameters );


	return pxTopOfStack;
}
/*-----------------------------------------------------------*/

portSHORT sPortStartScheduler( portSHORT sUsePreemption )
{
	/* In this port we ignore the parameter and use the portUSE_PREEMPTION
	definition instead. */

	/* Setup the hardware to generate the tick. */
	prvSetupTimerInterrupt();

	/* Restore the context of the first task that is going to run. */
	portRESTORE_CONTEXT();

	/* Simulate a function call end as generated by the compiler.  We will now
	jump to the start of the task the context of which we have just restored. */
	asm volatile ( "ret" );

	/* Should not get here. */
	return pdTRUE;
}
/*-----------------------------------------------------------*/

void vPortEndScheduler( void )
{
	/* It is unlikely that the AVR port will get stopped.  If required simply
	disable the tick interrupt here. */
}
/*-----------------------------------------------------------*/

/*
 * Manual context switch.  The first thing we do is save the registers so we
 * can use a naked attribute.
 */
// void __attribute__ ( ( naked ) ) vPortYield( void ) ;
void __attribute__ ( ( naked ) ) vPortYield( void )
{

	portSAVE_CONTEXT();

	vTaskSwitchContext();
	portRESTORE_CONTEXT();

	asm volatile ( "ret" );
}
/*-----------------------------------------------------------*/

/*
 * Context switch function used by the tick.  This must be identical to
 * vPortYield() from the call to vTaskSwitchContext() onwards.  The only
 * difference from vPortYield() is the tick count is incremented as the
 * call comes from the tick ISR.
 *
 *
 * Im Falle des MSP430 liegt auf dem Stack zusaetzlich das Statusregister.
 * Der MSP legt das naemlich automatisch drauf (nur bei Interrupts)
 */
void __attribute__ ( ( naked ) ) vPortYieldFromTick( void ) ;
void __attribute__ ( ( naked ) ) vPortYieldFromTick( void )
{
	portSAVE_CONTEXT();
	vTaskIncrementTick();
	vTaskSwitchContext();
	portRESTORE_CONTEXT();

	asm volatile ( "ret" );
}
/*-----------------------------------------------------------*/

/*
 * Setup timer 1 compare match A to generate a tick interrupt.
 */
static void prvSetupTimerInterrupt( void )
{
	BTCTL = BTIP2 + BTIP1 + BTIP0;        // 7.8125ms Interrupt
	BTCNT2 = 0;							  // Ruecksetzen des Zaehlerregisters

	IE2 |= BTIE;                          // Enable BT interrupt
}
/*-----------------------------------------------------------*/

#if portUSE_PREEMPTION == 1

	/*
	 * Tick ISR for preemptive scheduler.  We can use a naked attribute as
	 * the context is saved at the start of vPortYieldFromTick().  The tick
	 * count is incremented after the context is saved.
	 */
	interrupt (BASICTIMER_VECTOR) __attribute__ ((naked)) BasicTimerISR(void);
	interrupt (BASICTIMER_VECTOR) __attribute__ ((naked)) BasicTimerISR(void)
	{
		vPortYieldFromTick();

		asm volatile ( "reti" ); // noetig, weil der Compiler hier keinen
								 // Ruecksprung einbaut

	}
#else

	/*
	 * Tick ISR for the cooperative scheduler.  All this does is increment the
	 * tick count.  We don't need to switch context, this can only be done by
	 * manual calls to portYIELD();
	 */
	interrupt (BASICTIMER_VECTOR)  BasicTimerISR(void);
	interrupt (BASICTIMER_VECTOR)  BasicTimerISR(void)
	{
		vTaskIncrementTick();
	}
#endif



