/*
	FreeRTOS V1.2.0 - Copyright (C) 2003 Richard Barry.

	Licensed under the Open Software License version 1.1

	The file queue.c and related documentation is part of the FreeRTOS
	distribution.  Any and all data files, source code and documentation
	included in the FreeRTOS distribution are the exclusive property of
	Richard Barry.

	See www.FreeRTOS.org for documentation, license and warranty details.
*/

/*
Changes from V1.01:

	+ More use of 8bit data types.
	+ Function name prefixes changed where the data type returned has changed.
*/

#include <stdlib.h>
#include <string.h>
#include "projdefs.h"
#include "portable.h"
#include "errors.h"
#include "task.h"
#include "list.h"

/*-----------------------------------------------------------
 * PUBLIC LIST API documented in list.h
 *----------------------------------------------------------*/


/*
 * Definition of the queue used by the scheduler.
 * Items are queued by copy, not reference.
 */
typedef struct QueueDefinition
{
	portCHAR *pcHead;						/*< Points to the beginning of the queue storage area. */
	portCHAR *pcTail;						/*< Points to the byte at the end of the queue storage area.  Once more byte is allocated than necessary to store the queue items, this is used as a marker. */

	portCHAR *pcWriteTo;					/*< Points to the free next place in the storage area. */
	portCHAR *pcReadFrom;					/*< Points to the last place that a queued item was read from. */

	xList xTasksWaitingToSend;				/*< List of tasks that are blocked waiting to post onto this queue.  Stored in priority order. */
	xList xTasksWaitingToReceive;			/*< List of tasks that are blocked waiting to read from this queue.  Stored in priority order. */

	unsigned portCHAR ucMessagesWaiting;	/*< The number of items currently in the queue. */
	unsigned portCHAR ucLength;				/*< The length of the queue defined as the number of items it will hold, not the number of bytes. */
	unsigned portCHAR ucItemSize;			/*< The size of each items that the queue will hold. */
} xQUEUE;
/*-----------------------------------------------------------*/

/*
 * Macro that copies an item into the queue.  This is done by copying the item
 * byte for byte, not by reference.  Updates the queue state to ensure it's
 * integrity after the copy.
 */
#define prvCopyQueueData( pxQueue, pvItemToQueue )												\
{																								\
	memcpy( ( void * ) pxQueue->pcWriteTo, pvItemToQueue, ( unsigned ) pxQueue->ucItemSize );	\
	++( pxQueue->ucMessagesWaiting );															\
	pxQueue->pcWriteTo += pxQueue->ucItemSize;													\
	if( pxQueue->pcWriteTo >= pxQueue->pcTail )													\
	{																							\
		pxQueue->pcWriteTo = pxQueue->pcHead;													\
	}																							\
}
/*-----------------------------------------------------------*/

/*
 * Inside this function xQueueHandle is a pointer to a xQUEUE structure.
 * To keep the definition private the API header file defines it as a
 *pointer to void.
 */
typedef xQUEUE * xQueueHandle;

/*
 * Prototypes for public functions are included here so we don't have to
 * include the API header file (as it defines xQueueHandle differently).  These
 * functions are documented in the API header file.
 */
xQueueHandle xQueueCreate( unsigned portCHAR ucQueueLength, unsigned portCHAR ucItemSize );
portCHAR cQueueSend( xQueueHandle xQueue, const void * pvItemToQueue, unsigned portLONG ulTicksToWait );
unsigned portCHAR ucQueueMessagesWaiting( xQueueHandle pxQueue );
void vQueueDelete( xQueueHandle xQueue );
portCHAR cQueueSendFromISR( xQueueHandle pxQueue, const void *pvItemToQueue, portCHAR cTaskPreviouslyWoken );
portCHAR cQueueReceive( xQueueHandle pxQueue, void *pcBuffer, unsigned portLONG ulTicksToWait );


extern unsigned portSHORT usSaveStatusRegister;
/*-----------------------------------------------------------
 * PUBLIC QUEUE MANAGEMENT API documented in queue.h
 *----------------------------------------------------------*/

xQueueHandle xQueueCreate( unsigned portCHAR ucQueueLength, unsigned portCHAR ucItemSize )
{
xQUEUE *pxNewQueue;
unsigned portSHORT usQueueSizeInBytes;

	/* Allocate the new queue structure. */
	if( ucQueueLength > ( unsigned portCHAR ) 0 )
	{
		pxNewQueue = ( xQUEUE * ) pvPortMalloc( sizeof( xQUEUE ) );
		if( pxNewQueue != NULL )
		{
			/* Create the list of pointers to queue items.  The queue is one byte longer
			than asked for to make wrap checking easier/faster. */
			usQueueSizeInBytes = ( unsigned portSHORT ) ( ucQueueLength * ucItemSize ) + ( unsigned portSHORT ) 1;

			pxNewQueue->pcHead = ( portCHAR * ) pvPortMalloc( usQueueSizeInBytes );
			if( pxNewQueue->pcHead != NULL )
			{
				/* Initialise the queue members as described above where the queue
				type is defined. */
				pxNewQueue->pcTail = pxNewQueue->pcHead + ( ucQueueLength * ucItemSize );
				pxNewQueue->ucMessagesWaiting = ( unsigned portCHAR ) 0;
				pxNewQueue->pcWriteTo = pxNewQueue->pcHead;
				pxNewQueue->pcReadFrom = pxNewQueue->pcHead + ( ( ucQueueLength - ( unsigned portCHAR ) 1 ) * ucItemSize );
				pxNewQueue->ucLength = ucQueueLength;
				pxNewQueue->ucItemSize = ucItemSize;

				/* Likewise ensure the event queues start with the correct state. */
				vListInitialise( &( pxNewQueue->xTasksWaitingToSend ) );
				vListInitialise( &( pxNewQueue->xTasksWaitingToReceive ) );

				return  pxNewQueue;
			}
			else
			{
				vPortFree( pxNewQueue );
			}
		}
	}

	/* Will only reach here if we could not allocate enough memory or no memory was required. */
	return NULL;
}
/*-----------------------------------------------------------*/

portCHAR cQueueSend( xQueueHandle pxQueue, const void *pvItemToQueue, unsigned portLONG ulTicksToWait )
{
portCHAR cReturn;

	taskENTER_CRITICAL();
	{
		/* If the queue is already full we may have to block. */
		if( pxQueue->ucMessagesWaiting == pxQueue->ucLength )
		{
			/* The queue is full - do we want to block or just leave without
			posting? */
			if( ulTicksToWait > ( unsigned portLONG ) 0 )
			{
				/* This task will automatically be woken by the end of the delay
				period.  It can also be woken by a space becoming available on
				the queue. */
				vTaskPlaceOnEventList( &( pxQueue->xTasksWaitingToSend ), ulTicksToWait );

				/* Force a context switch now as we are blocked.  We can do
				this from within a critical section as the task we are
				switching to has its own context.  When we return here (i.e. we
				unblock) we will leave the critical section as normal. */
				taskYIELD();
			}
		}

		if( pxQueue->ucMessagesWaiting < pxQueue->ucLength )
		{
			/* There is room in the queue, copy the data into the queue. */
			prvCopyQueueData( pxQueue, pvItemToQueue );
			cReturn = ( portCHAR ) pdPASS;

			/* Do we need to wake a task waiting for a message? */
			if( !listLIST_IS_EMPTY( &( pxQueue->xTasksWaitingToReceive ) ) )
			{
				if( cTaskRemoveFromEventList( &( pxQueue->xTasksWaitingToReceive ) ) != ( portCHAR ) pdFALSE )
				{
					/* The task waiting has a higher priority than us so
					force a context switch.  As described above, it is okay to
					do this from within a critical section. */
					taskYIELD();
				}
			}
		}
		else
		{
			cReturn = errQUEUE_FULL;
		}
	}
	taskEXIT_CRITICAL();

	return cReturn;
}
/*-----------------------------------------------------------*/

portCHAR cQueueSendFromISR( xQueueHandle pxQueue, const void *pvItemToQueue, portCHAR cTaskPreviouslyWoken )
{
	/* Similar to cQueueSend, except we don't block if there is no room in the
	queue.  Also we don't directly wake a task that was blocked on a queue
	read, instead we return a flag to say whether a context switch is required
	or not (i.e. has a task with a higher priority than us been woken by this
	post). */
	if( pxQueue->ucMessagesWaiting < pxQueue->ucLength )
	{
		prvCopyQueueData( pxQueue, pvItemToQueue );

		/* Do we need to wake a task waiting for a message? */
		if( !cTaskPreviouslyWoken )
		{
			if( !listLIST_IS_EMPTY( &( pxQueue->xTasksWaitingToReceive ) ) )
			{
				if( cTaskRemoveFromEventList( &( pxQueue->xTasksWaitingToReceive ) ) != ( portCHAR ) pdFALSE )
				{
					/* The task waiting has a higher priority so record that a
					context	switch is required. */
					return pdTRUE;
				}
			}
		}
	}

	return cTaskPreviouslyWoken;
}
/*-----------------------------------------------------------*/

portCHAR cQueueReceive( xQueueHandle pxQueue, void *pcBuffer, unsigned portLONG ulTicksToWait )
{
portCHAR cReturn;

	taskENTER_CRITICAL();
	{
		/* If there are no messages in the queue we may have to block. */
		if( pxQueue->ucMessagesWaiting == ( unsigned portCHAR ) 0 )
		{
			/* There are no messages in the queue, do we want to block or just
			leave with nothing? */
			if( ulTicksToWait > ( unsigned portLONG ) 0 )
			{
				/* This task will automatically be woken by the end of the delay
				period.  It can also be woken by a message becoming available. */
				vTaskPlaceOnEventList( &( pxQueue->xTasksWaitingToReceive ), ulTicksToWait );

				/* Force a context switch as we just put ourselves to sleep. It
				is okay to do this from within a critical section as the task
				we switch to will have it's own context.  When we unblock we
				will resume from here and leave the critical section below. */
				taskYIELD();
			}
		}

		if( pxQueue->ucMessagesWaiting > ( unsigned portCHAR ) 0 )
		{
			/* There is an item in the queue.  Read it out and update the queue
			state variables. */
			pxQueue->pcReadFrom += pxQueue->ucItemSize;
			if( pxQueue->pcReadFrom >= pxQueue->pcTail )
			{
				pxQueue->pcReadFrom = pxQueue->pcHead;
			}
			--( pxQueue->ucMessagesWaiting );

			memcpy( pcBuffer, ( void * ) pxQueue->pcReadFrom, ( unsigned ) pxQueue->ucItemSize );

			/* Do we need to wake a task waiting for space to post a message? */
			if( !listLIST_IS_EMPTY( &( pxQueue->xTasksWaitingToSend ) ) )
			{
				if( cTaskRemoveFromEventList( &( pxQueue->xTasksWaitingToSend ) ) != ( portCHAR ) pdFALSE )
				{
					/* The task waiting has a higher priority than us so
					force a context switch. */
					taskYIELD();
				}
			}

			cReturn = ( portCHAR ) pdPASS;
		}
		else
		{
			cReturn = ( portCHAR ) pdFAIL;
		}
	}
	taskEXIT_CRITICAL();

	return cReturn;
}
/*-----------------------------------------------------------*/

unsigned portCHAR ucQueueMessagesWaiting( xQueueHandle pxQueue )
{
unsigned portCHAR ucReturn;

	taskENTER_CRITICAL();
		ucReturn = pxQueue->ucMessagesWaiting;
	taskEXIT_CRITICAL();

	return ucReturn;
}
/*-----------------------------------------------------------*/

void vQueueDelete( xQueueHandle pxQueue )
{
	vPortFree( pxQueue->pcHead );
	vPortFree( pxQueue );
}
/*-----------------------------------------------------------*/






