/*
	FreeRTOS V1.2.0 - Copyright (C) 2003 Richard Barry.

	Licensed under the Open Software License version 1.1

	The file queue.h and related documentation is part of the FreeRTOS
	distribution.  Any and all data files, source code and documentation 
	included in the FreeRTOS distribution are the exclusive property of 
	Richard Barry.

	See www.FreeRTOS.org for documentation, license and warranty details.
*/

#ifndef QUEUE_H
#define QUEUE_H

typedef void * xQueueHandle;

/**
 * Creates a new queue instance.  This allocates the storage required by the
 * new queue and returns a handle for the queue.
 *
 * @param ucQueueLength The maximum number of items that the queue can contain.
 *
 * @param ucItemSize The number of bytes each item in the queue will require.  
 * Items are queued by copy, not by reference, so this is the number of bytes
 * that will be copied for each posted item.  Each item on the queue must be
 * the same size.
 *
 * @return If the queue is successfully create then a handle to the newly 
 * created queue is returned.  If the queue cannot be created then 0 is
 * returned.
 * 
 * \page xQueueCreate xQueueCreate
 * \ingroup QueueManagement
 * <HR>
 */
xQueueHandle xQueueCreate( unsigned portCHAR ucQueueLength, unsigned portCHAR ucItemSize );

/**
 * Post an item on a queue.  The item is queued by copy, not by reference.
 * This function must not be called from an interrupt service routine.
 * See cQueueSendFromISR () for an alternative which may be used in an ISR.
 *
 * @param xQueue The handle to the queue on which the item is to be posted.
 * 
 * @param pvItemToQueue A pointer to the item that is to be placed on the 
 * queue.  The size of the items the queue will hold was defined when the
 * queue was created, so this many bytes will be copied from pvItemToQueue
 * into the queue storage area.
 *
 * @param ulTicksToWait The maximum amount of time the task should block
 * waiting for space to become available on the queue, should it already
 * be full.  The call will return immediately if this is set to 0.  The
 * time is defined in tick periods so the constant portTICKS_PER_MS 
 * should be used to convert to real time if this is required.
 *
 * @return pdTRUE if the item was successfully posted, otherwise errQUEUE_FULL.
 *
 * \page cQueueSend cQueueSend
 * \ingroup QueueManagement
 * <HR>
 */
portCHAR cQueueSend( xQueueHandle xQueue, const void * pvItemToQueue, unsigned portLONG ulTicksToWait );

/**
 * Receive an item from a queue.  The item is received by copy so a buffer of 
 * adequate size must be provided.  The number of bytes copied into the buffer
 * was defined when the queue was created.
 *
 * @param pxQueue The handle to the queue from which the item is to be
 * received.
 *
 * @param pcBuffer Pointer to the buffer into which the received item will
 * be copied.
 * 
 * @param ulTicksToWait The maximum amount of time the task should block
 * waiting for an item to receive should the queue be empty at the time
 * of the call.    The time is defined in tick periods so the constant 
 * portTICKS_PER_MS should be used to convert to real time if this is required.
 *
 * @return pdTRUE if an item was successfully received from the queue,
 * otherwise pdFALSE.
 *
 * \page cQueueReceive cQueueReceive
 * \ingroup QueueManagement
 * <HR>
 */
portCHAR cQueueReceive( xQueueHandle xQueue, void *pcBuffer, unsigned portLONG ulTicksToWait );

/**
 * Return the number of messages stored in a queue.
 *
 * @param xQueue A handle to the queue being queried.
 * 
 * @return The number of messages available in the queue.
 *
 * \page ucQueueMessagesWaiting ucQueueMessagesWaiting
 * \ingroup QueueManagement
 * <HR>
 */
unsigned portCHAR ucQueueMessagesWaiting( xQueueHandle xQueue );

/**
 * Delete a queue - freeing all the memory allocated for storing of items
 * placed on the queue.
 * 
 * @param xQueue A handle to the queue to be deleted.
 *
 * \page vQueueDelete vQueueDelete
 * \ingroup QueueManagement
 * <HR>
 */
void vQueueDelete( xQueueHandle xQueue );

/**
 * Post an item on a queue.  It is safe to use this function from within an
 * interrupt service routine.
 *
 * Items are queued by copy not reference so it is preferable to only
 * queue small items, especially when called from an ISR.  In most cases
 * it would be preferable to store a pointer to the item being queued.
 *
 * @param xQueue The handle to the queue on which the item is to be posted.
 * 
 * @param pvItemToQueue A pointer to the item that is to be placed on the 
 * queue.  The size of the items the queue will hold was defined when the
 * queue was created, so this many bytes will be copied from pvItemToQueue
 * into the queue storage area.
 *
 * @param cTaskPreviouslyWoken This is included so an ISR can post onto
 * the same queue multiple times from a single interrupt.  The first call
 * should always pass in pdFALSE.  Subsequent calls should pass in
 * the value returned from the previous call.  See the file serial .c in the
 * PC port for a good example of this mechanism.
 *
 * @return pdTRUE if a task was woken by posting onto the queue.  This is 
 * used by the ISR to determine if a context switch may be required following
 * the ISR.
 *
 * \page cQueueSendFromISR cQueueSendFromISR
 * \ingroup QueueManagement
 * <HR>
 */
portCHAR cQueueSendFromISR( xQueueHandle pxQueue, const void *pvItemToQueue, portCHAR cTaskPreviouslyWoken );





#endif

