/*
	FreeRTOS V1.2.0 - Copyright (C) 2003 Richard Barry.

	Licensed under the Open Software License version 1.1

	The file semphr.h and related documentation is part of the FreeRTOS
	distribution.  Any and all data files, source code and documentation 
	included in the FreeRTOS distribution are the exclusive property of 
	Richard Barry.

	See www.FreeRTOS.org for documentation, license and warranty details.
*/

#include "queue.h"

#ifndef SEMAPHORE_H
#define SEMAPHORE_H

typedef xQueueHandle xSemaphoreHandle;

#define semBINARY_SEMAPHORE_QUEUE_LENGTH	( ( unsigned portCHAR ) 1 )
#define semSEMAPHORE_QUEUE_ITEM_LENGTH		( ( unsigned portCHAR ) 0 )
#define semGIVE_BLOCK_TIME					( ( unsigned portLONG ) 0 )


/**
 * Macro that implements a semaphore by using the existing queue mechanism.
 * The queue length is 1 as this is a binary semaphore.  The data size is 0
 * as we don't want to actually store any data - we just want to know if the
 * queue is empty or full.
 *
 * @param xSemaphore Handle to the created semaphore.  Should be of type xSemaphoreHandle.
 *
 * \page vSemaphoreCreateBinary vSemaphoreCreateBinary
 * \ingroup Semaphores
 * <HR>
 */
#define vSemaphoreCreateBinary( xSemaphore )		{																							\
														xSemaphore = xQueueCreate( ( unsigned portCHAR ) 1, semSEMAPHORE_QUEUE_ITEM_LENGTH );	\
														if( xSemaphore != NULL )																\
														{																						\
															cSemaphoreGive( xSemaphore );														\
														}																						\
													}

/**
 * Obtain a semaphore.  The semaphore must of been created using 
 * vSemaphoreCreateBinary ().
 *
 * @param xSemaphore A handle to the semaphore being obtained.  This is the
 * handle returned by vSemaphoreCreateBinary ();
 *
 * @param ulBlockTime The time in ticks to wait for the semaphore to become
 * available.  The macro portTICKS_PER_MS can be used to convert this to a
 * real time.  A block time of zero can be used to poll the semaphore.
 *
 * @return pdTRUE if the semaphore was obtained.  pdFALSE if ulBlockTime
 * expired without the semaphore becoming available.
 *
 * \page cSemaphoreTake cSemaphoreTake
 * \ingroup Semaphores
 * <HR>
 */
#define cSemaphoreTake( xSemaphore, ulBlockTime )	cQueueReceive( ( xQueueHandle ) xSemaphore, NULL, ulBlockTime )

/**
 * Release a semaphore.  The semaphore must of been created using 
 * vSemaphoreCreateBinary (), and obtained using sSemaphoreTask ().
 *
 * This must not be used from an ISR.  See cSemaphoreGiveFromISR () for
 * an alternative which can be used from an ISR.
 *
 * @param xSemaphore A handle to the semaphore being released.  This is the
 * handle returned by vSemaphoreCreateBinary ();
 *
 * @return pdTRUE if the semaphore was released.  pdFALSE if an error occurred.
 * Semaphores are implemented using queues.  An error can occur if there is
 * no space on the queue to post a message - indicating that the 
 * semaphore was not first obtained correctly.
 * \page cSemaphoreGive cSemaphoreGive
 * \ingroup Semaphores
 * <HR>
 */
#define cSemaphoreGive( xSemaphore )				cQueueSend( ( xQueueHandle ) xSemaphore, NULL, semGIVE_BLOCK_TIME )

/**
 * Release a semaphore.  The semaphore must of been created using 
 * vSemaphoreCreateBinary (), and obtained using sSemaphoreTask ().
 *
 * This macro can be used from an ISR.
 *
 * @param xSemaphore A handle to the semaphore being released.  This is the
 * handle returned by vSemaphoreCreateBinary ();
 *
 * @param sTaskPreviouslyWoken This is included so an ISR can make multiple calls
 * to cSemaphoreGiveFromISR () from a single interrupt.  The first call
 * should always pass in pdFALSE.  Subsequent calls should pass in
 * the value returned from the previous call.  See the file serial .c in the
 * PC port for a good example of using xSemaphoreGiveFromISR ().
 *
 * @return pdTRUE if a task was woken by releasing the semaphore.  This is 
 * used by the ISR to determine if a context switch may be required following
 * the ISR.
 *
 * \page cSemaphoreGiveISR cSemaphoreGiveFromISR
 * \ingroup Semaphores
 */
#define cSemaphoreGiveFromISR( xSemaphore, sTaskPreviouslyWoken )			cQueueSendFromISR( ( xQueueHandle ) xSemaphore, NULL, sTaskPreviouslyWoken )


#endif

