/*
	FreeRTOS V1.2.0 - Copyright (C) 2003 Richard Barry.

	Licensed under the Open Software License version 1.1

	The file task.h and related documentation is part of the FreeRTOS
	distribution.  Any and all data files, source code and documentation
	included in the FreeRTOS distribution are the exclusive property of
	Richard Barry.

	See www.FreeRTOS.org for documentation, license and warranty details.
*/

#ifndef TASK_H
#define TASK_H

#include "portable.h"
#include "projdefs.h"
#include "list.h"

/*-----------------------------------------------------------
 * MACROS AND DEFINITIONS
 *----------------------------------------------------------*/

/**
 * Type by which tasks are referenced.  For example, a call to sTaskCreate
 * returns (via a pointer parameter) an xTaskHandle variable that can then
 * be used as a parameter to vTaskDelete to delete the task.
 *
 * \page xTaskHandle xTaskHandle
 * \ingroup Tasks
 * <HR>
 */
typedef void * xTaskHandle;

/**
 * Defines the priority used by the idle task.  This must not be modified.
 *
 * \ingroup TaskUtils
 * <HR>
 */
#define tskIDLE_PRIORITY			( ( unsigned portCHAR ) 0 )

/**
 * The maximum number of characters a task name can take.  This includes
 * the terminating null as a character.
 *
 * \ingroup TaskUtils
 * <HR>
 */
#define tskMAX_TASK_NAME_LEN		( ( unsigned portSHORT ) 8 )

/**
 * Macro for forcing a context switch.
 *
 * \page taskYEILD taskYIELD
 * \ingroup SchedulerControl
 * <HR>
 */
#define taskYIELD()					portYIELD()

/**
 * Macro to mark the start of a critical code region.  Preemptive context
 * switches cannot occur when in a critical region.
 *
 * NOTE: This may alter the stack (depending on the portable implementation)
 * so must be used with care!
 *
 * \page taskENTER_CRITICAL taskENTER_CRITICAL
 * \ingroup SchedulerControl
 * <HR>
 */
#define taskENTER_CRITICAL()		portENTER_CRITICAL()

/**
 * Macro to mark the end of a critical code region.  Preemptive context
 * switches cannot occur when in a critical region.
 *
 * NOTE: This may alter the stack (depending on the portable implementation)
 * so must be used with care!
 *
 * \page taskEXIT_CRITICAL taskEXIT_CRITICAL
 * \ingroup SchedulerControl
 * <HR>
 */
#define taskEXIT_CRITICAL()			portEXIT_CRITICAL()

/**
 * Macro to disable all maskable interrupts.
 *
 * \page taskDISABLE_INTERRUPTS taskDISABLE_INTERRUPTS
 * \ingroup SchedulerControl
 * <HR>
 */
#define taskDISABLE_INTERRUPTS()	portDISABLE_INTERRUPTS()

/**
 * Macro to enable processor interrupts.
 *
 * \page taskENABLE_INTERRUPTS taskENABLE_INTERRUPTS
 * \ingroup SchedulerControl
 * <HR>
 */
#define taskENABLE_INTERRUPTS()		portENABLE_INTERRUPTS()



/**
 * Task control block.  A task control block (TCB) is allocated to each task,
 * and stores the context of the task.
 * \page TCB TCB
 * \ingroup Tasks
 * <HR>
 */


typedef struct tskTaskControlBlock
{
	portSTACK_TYPE		*pxTopOfStack;						/*< Points to the location of the last item placed on the tasks stack.  THIS MUST BE THE FIRST MEMBER OF THE STRUCT. */
	portSTACK_TYPE		*pxStack;							/*< Points to the start of the stack. */
	unsigned portSHORT	usStackDepth;						/*< Total depth of the stack (when empty).  This is defined as the number of variables the stack can hold, not the number of bytes. */
	portCHAR			pcTaskName[ tskMAX_TASK_NAME_LEN ];	/*< Descriptive name given to the task when created.  Facilitates debugging only. */
	unsigned portCHAR	ucPriority;							/*< The priority of the task where 0 is the lowest priority. */

	xListItem			xGenericListItem;					/*< List item used to place the TCB in ready and blocked queues. */
	xListItem			xEventListItem;						/*< List item used to place the TCB in event lists. */

	unsigned portCHAR	ucTCBNumber;						/*< This is used for tracing the scheduler only. */
} tskTCB;


/*-----------------------------------------------------------
 * TASK MANAGEMENT API
 *----------------------------------------------------------*/

/**
 * Create a new task and add it to the list of tasks that are ready to run.
 *
 * @param pvTaskCode Pointer to the task entry function.  Tasks must be
 * implemented to never return (i.e. continuous loop).
 *
 * @param pcName A descriptive name for the task.  This is mainly used to
 * facilitate debugging.
 *
 * @param usStackDepth The size of the task stack specified as the number of
 * variables the stack can hold - not the number of bytes.  For example, if
 * the stack is 16 bits wide and usStackDepth is defined as 100, 200 bytes
 * will be allocated for stack storage.
 *
 * @param pvParameters Pointer that will be used as the parameter for the task
 * being created.
 *
 * @param usPriority The priority at which the task should run.
 *
 * @param pvCreatedTask Used to pass back a handle by which the created task
 * can be referenced.
 *
 * @return pdPASS if the task was successfully created and added to a ready
 * list, otherwise an error code defined in the file errors. h
 *
 * \page sTaskCreate sTaskCreate
 * \ingroup Tasks
 * <HR>
 */
portSHORT sTaskCreate( pdTASK_CODE pvTaskCode, const portCHAR * const pcName, unsigned portSHORT usStackDepth, void *pvParameters, unsigned portCHAR ucPriority, xTaskHandle *pvCreatedTask );

/**
 * Remove a task from the scheduler.  The task being deleted will be removed
 * from all ready, blocked and event lists.
 *
 * NOTE:  The idle task is responsible for freeing the scheduler allocated
 * memory from tasks that have been deleted.  It is therefore important that
 * the idle task is not starved of processor time if your application makes
 * any calls to vTaskDelete ().  Memory allocated by the task code is not
 * automatically freed, and should be freed before the task is deleted.
 *
 * See the demo application file death. c for sample code that utilises
 * vTaskDelete ().
 *
 * @param pxTask The handle of the task to be deleted.  Passing NULL will
 * cause the calling task to be deleted.
 *
 * \page vTaskDelete vTaskDelete
 * \ingroup Tasks
 * <HR>
 */
void vTaskDelete( xTaskHandle pxTask );

/**
 * Delay a task for a given number of ticks.  The actual time that the
 * task remains blocked depends on the tick rate.  The constant
 * portTICKS_PER_MS can be used to calculate real time from the tick
 * rate - with the resolution of one tick period.
 *
 * @param ulTicksToDelay The amount of time, in tick periods, that
 * the calling task should block.
 *
 * \page vTaskDelay vTaskDelay
 * \ingroup Tasks
 * <HR>
 */
void vTaskDelay( unsigned portLONG ulTicksToDelay );



/*-----------------------------------------------------------
 * SCHEDULER CONTROL
 *----------------------------------------------------------*/

/**
 * Starts the scheduler tick processing.  After calling the scheduler
 * has control over which tasks are executed and when.  This function
 * does not return until an executing task calls vTaskEndScheduler ().
 *
 * At least one task should be created via a call to sTaskCreate ()
 * before calling vTaskStartScheduler ().  The idle task is created
 * automatically when the first application task is created.
 *
 * See the demo application file main. c for an example of creating
 * tasks and starting the scheduler.
 *
 * @param sUsePreemption Set to pdTRUE for the preemptive scheduling
 * policy to be used, or pdFALSE for the cooperative scheduling
 * policy.
 *
 * \page vTaskStartScheduler vTaskStartScheduler
 * \ingroup SchedulerControl
 * <HR>
 */
void vTaskStartScheduler( portSHORT sUsePreemption );

/**
 * Stops the scheduler tick.  All created tasks will be automatically
 * deleted and multitasking (either preemptive or cooperative) will
 * stop.  Execution then resumes from the point where vTaskStartScheduler ()
 * was called, as if vTaskStartScheduler () had just returned.
 *
 * See the demo application file main. c for an example that uses
 * vTaskEndScheduler ().
 *
 * \page vTaskEndScheduler vTaskEndScheduler
 * \ingroup SchedulerControl
 * <HR>
 */
void vTaskEndScheduler( void );


/*-----------------------------------------------------------
 * TASK UTILITIES
 *----------------------------------------------------------*/

/**
 * @return The count of ticks since vTaskStartScheduler was called.
 *
 * \page ulTaskGetTickCount ulTaskGetTickCount
 * \ingroup TaskUtils
 * <HR>
 */
volatile unsigned portLONG ulTaskGetTickCount( void );

/**
 * @return The number of tasks that the scheduler is currently managing.
 * This includes all ready and blocked tasks.
 *
 * \page usTaskGetNumberOfTasks usTaskGetNumberOfTasks
 * \ingroup TaskUtils
 * <HR>
 */
unsigned portSHORT usTaskGetNumberOfTasks( void );

/**
 * NOTE: This function will disable interrupts for its duration.  It is
 * not intended for normal application runtime use but as a debug aid.
 *
 * Lists all the current tasks, along with their current state and stack
 * usage high water mark.
 *
 * Tasks are reported as blocked ('B'), ready ('R') or deleted ('D').
 *
 * @param pcWriteBuffer A buffer into which the above mentioned details
 * will be written, in ascii form.  This buffer is assumed to be large
 * enough to contain the generated report.  Approximately 40 bytes per
 * task should be sufficient.
 *
 * \page vTaskList vTaskList
 * \ingroup TaskUtils
 * <HR>
 */
void vTaskList( portCHAR *pcWriteBuffer );

/**
 * Starts a scheduler activity trace.  The trace logs the identity of which
 * task is running when.  To use the trace function the macro
 * USE_TRACE_FACILITY must be defined within task. c.
 *
 * The trace file is stored in binary format.  A separate DOS utility called
 * convtrce.exe is used to convert this into a tab delimited text file which
 * can be viewed and plotted in a spread sheet.
 *
 * @param pcBuffer The buffer into which the trace will be written.
 *
 * @param usBufferSize The size of pcBuffer in bytes.  The trace will continue
 * until either the buffer in full, or usTaskEndTrace () is called.
 *
 * \page vTaskStartTrace vTaskStartTrace
 * \ingroup TaskUtils
 * <HR>
 */
void vTaskStartTrace( portCHAR * pcBuffer, unsigned portSHORT usBufferSize );

/**
 * Stops a scheduler activity trace.  See vTaskStartTrace ().
 *
 * @return The number of bytes that have been written into the trace buffer.
 *
 * \page usTaskEndTrace usTaskEndTrace
 * \ingroup TaskUtils
 * <HR>
 */
unsigned portSHORT usTaskEndTrace( void );


/*-----------------------------------------------------------
 * SCHEDULER INTERNALS AVAILABLE FOR PORTING PURPOSES
 *----------------------------------------------------------*/

/*
 * THIS FUNCTION MUST NOT BE USED FROM APPLICATION CODE.  IT IS ONLY
 * INTENDED FOR USE WHEN IMPLEMENTING A PORT OF THE SCHEDULER AND IS
 * AN INTERFACE WHICH IS FOR THE EXCLUSIVE USE OF THE SCHEDULER.
 *
 * Called from the scheduler tick (either preemptive or cooperative),
 * this increments the tick count and checks if any tasks that are blocked
 * for a finite period required removing from a blocked list and placing on
 * a ready list.
 */
void vTaskIncrementTick( void );

/*
 * THIS FUNCTION MUST NOT BE USED FROM APPLICATION CODE.  IT IS AN
 * INTERFACE WHICH IS FOR THE EXCLUSIVE USE OF THE SCHEDULER.
 *
 * THIS FUNCTION MUST BE CALLED WITH INTERRUPTS DISABLED.
 *
 * Removes the calling task from the ready list and places it both
 * on the list of tasks waiting for a particular event, and the
 * list of delayed tasks.  The task will be removed from both lists
 * and replaced on the ready list should either the event occur (and
 * there be no higher priority tasks waiting on the same event) or
 * the delay period expires.
 *
 * @param pxEventList The list containing tasks that are blocked waiting
 * for the event to occur.
 *
 * @param ulTicksToWait The maximum amount of time that the task should wait
 * for the event to occur.  This is specified in scheduler ticks,the constant
 * portTICKS_PER_MS can be used to convert scheduler ticks into a real time
 * period.
 */
void vTaskPlaceOnEventList( xList *pxEventList, unsigned portLONG ulTicksToWait );

/*
 * THIS FUNCTION MUST NOT BE USED FROM APPLICATION CODE.  IT IS AN
 * INTERFACE WHICH IS FOR THE EXCLUSIVE USE OF THE SCHEDULER.
 *
 * THIS FUNCTION MUST BE CALLED WITH INTERRUPTS DISABLED.
 *
 * Removes a task from both the specified event list and the list of blocked
 * tasks, and places it on a ready queue.
 *
 * cTaskRemoveFromEventList () will be called if either an event occurs to
 * unblock a task, or the block timeout period expires.
 *
 * @return pdTRUE if the task being removed has a higher priority than the task
 * making the call, otherwise pdFALSE.
 */
portCHAR cTaskRemoveFromEventList( const xList *pxEventList );

/*
 * THIS FUNCTION MUST NOT BE USED FROM APPLICATION CODE.  IT IS AN
 * INTERFACE WHICH IS FOR THE EXCLUSIVE USE OF THE SCHEDULER.
 *
 * Empties the ready and delayed queues of task control blocks, freeing the
 * memory allocated for the task control block and task stacks as it goes.
 */
void vTaskCleanUpResources( void );

/*
 * THIS FUNCTION MUST NOT BE USED FROM APPLICATION CODE.  IT IS ONLY
 * INTENDED FOR USE WHEN IMPLEMENTING A PORT OF THE SCHEDULER AND IS
 * AN INTERFACE WHICH IS FOR THE EXCLUSIVE USE OF THE SCHEDULER.
 *
 * Sets the pointer to the current TCB to the TCB of the highest priority task
 * that is ready to run.
 */
void vTaskSwitchContext( void );

#endif /* TASK_H */



