/*
	FreeRTOS V1.2.0 - Copyright (C) 2003 Richard Barry.

	Licensed under the Open Software License version 1.1

	The file tasks.c and related documentation is part of the FreeRTOS
	distribution.  Any and all data files, source code and documentation
	included in the FreeRTOS distribution are the exclusive property of
	Richard Barry.

	See www.FreeRTOS.org for documentation, license and warranty details.
*/

/*
Changes from V1.00:

	+ Call to portRESTORE_CONTEXT has been removed.  The first context
	  switch is now performed within sPortStartScheduler().

Changes from V1.01:

	+ More use of 8bit data types.
	+ Function name prefixes changed where the data type returned has changed.
	+ USE_TRACE_FACILITY is no longer defined by default.

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "projdefs.h"
#include "portable.h"
#include "task.h"
#include "errors.h"

/*
 * Macro to define the amount of stack available to the idle task.
 */
#define tskIDLE_STACK_SIZE	portMINIMAL_STACK_SIZE

/* Constant definition for binary file identification. */
const portCHAR * const pcVersionNumber = "FreeRTOS V1.2.0 - Copyright (C) 2003 Richard Barry";


unsigned portSHORT usSaveStatusRegister;
/*
 * Macro that looks at the list of tasks that are currently delayed to see if
 * any require waking.
 *
 * Tasks are stored in the queue in the order of their wake time - meaning
 * once one tasks has been found whose timer has not expired we need not look
 * any further down the list.
 */
#define prvCheckDelayedTasks()																	\
{																								\
tskTCB *pxTCB;																					\
																								\
	while( ( pxTCB = ( tskTCB * ) pvListGetOwnerOfHeadEntry( pxDelayedTaskList ) ) != NULL )	\
	{																							\
		if( ulTickCount < listGET_LIST_ITEM_VALUE( &( pxTCB->xGenericListItem ) ) )				\
		{																						\
			break;																				\
		}																						\
		vListRemove( &( pxTCB->xGenericListItem ) );											\
		/* Is the task waiting on an event also? */												\
		vListRemove( &( pxTCB->xEventListItem ) );												\
		prvAddTaskToReadyQueue( pxTCB );														\
	}																							\
}

/*lint -e956 */

volatile tskTCB * volatile pxCurrentTCB = NULL;

/* Lists for ready and blocked tasks. --------------------*/

volatile xList pxReadyTasksLists[ portMAX_PRIORITIES ];	/* Ready lists can be modified by normal tick and other ISR's. */
volatile xList xDelayedTaskList1;						/* Delayed list can be modified by call to delay, and by tick ISR. */
volatile xList xDelayedTaskList2;						/* Two lists are used as one will hold tasks whose delay has overflowed the tick count. */
volatile xList xTasksWaitingTermination;					/* Termination list can be modified by call to vTaskDelete, and idle task. */
volatile xList *pxDelayedTaskList;						/* Points to the delayed task list currently in used (either xDelayedTaskList1 or 2). */
volatile xList *pxOverflowDelayedTaskList;				/* Points to the delayed task list that is not being pointed to by pxDelayedTaskList. */
portSHORT sUsingPreemption = pdFALSE;

/* File private variables. --------------------------------*/
static unsigned portSHORT usCurrentNumberOfTasks = 0;
static volatile unsigned portLONG ulTickCount = 0;				/* Modified by tick ISR. */
static unsigned portCHAR ucTopUsedPriority = tskIDLE_PRIORITY;
static portSHORT sSchedulerRunning = pdFALSE;

/* Debugging and trace facilities private variables and macros. ------------*/

/*
 * The value used to fill the stack of a task when the task is created.  This
 * is used purely for checking the high water mark for tasks.
 */
#define tskSTACK_FILL_BYTE	( ( unsigned portCHAR ) 0xa5 )

/*
 * Macros used by vListTask to indicate which state a task is in.
 */
#define tskBLOCKED_CHAR		( ( portCHAR ) 'B' )
#define tskREADY_CHAR		( ( portCHAR ) 'R' )
#define tskDELETED_CHAR		( ( portCHAR ) 'D' )

/*
 * Macros and private variables used by the trace facility.
 */
#define tskSIZE_OF_EACH_TRACE_LINE			( sizeof( unsigned portLONG ) + sizeof( unsigned portCHAR ) )
static volatile portCHAR * volatile pcTraceBuffer;
static portCHAR *pcTraceBufferStart;
static portCHAR *pcTraceBufferEnd;
static portCHAR cTracing = ( portCHAR ) pdFALSE;

/*
 * Macro that writes a trace of scheduler activity to a buffer.  This trace
 * shows which task is running when and is very useful as a debugging tool.
 * As this macro is called each context switch it is a good idea to undefined
 * it if not using the facility.
 */
#ifdef USE_TRACE_FACILITY
	#define vWriteTraceToBuffer()														\
	{																					\
		if( cTracing )																	\
		{																				\
			static unsigned portCHAR ucPreviousTask = 255;								\
																						\
			if( ucPreviousTask != pxCurrentTCB->ucTCBNumber )							\
			{																			\
				if( ( pcTraceBuffer + tskSIZE_OF_EACH_TRACE_LINE ) < pcTraceBufferEnd )	\
				{																		\
					ucPreviousTask = pxCurrentTCB->ucTCBNumber;							\
					*( unsigned portLONG * ) pcTraceBuffer = ulTickCount;				\
					pcTraceBuffer += sizeof( unsigned portLONG );						\
					*( unsigned portCHAR * ) pcTraceBuffer = ucPreviousTask;			\
					pcTraceBuffer += sizeof( unsigned portCHAR );						\
				}																		\
				else																	\
				{																		\
					cTracing = ( portCHAR ) pdFALSE;									\
				}																		\
			}																			\
		}																				\
	}
#else
	#define vWriteTraceToBuffer()
#endif


/* File private functions. --------------------------------*/

/*
 * Utility to ready a TCB for a given task.  Mainly just copies the parameters
 * into the TCB structure.
 */
static void prvInitialiseTCBVariables( tskTCB *pxTCB, unsigned portSHORT usStackDepth, const portCHAR * const pcName, unsigned portCHAR ucPriority );

/*
 * Utility to ready all the lists used by the scheduler.  This is called
 * automatically upon the creation of the first task.
 */
static portSHORT prvInitialiseTaskLists( void );

/*
 * Place the task represented by pxTCB into the appropriate ready queue for
 * the task.
 */
static void prvAddTaskToReadyQueue( tskTCB *pxTCB );

/*
 * The idle task, which as all tasks is implemented as a never ending loop.
 * The idle task is automatically created and added to the ready lists upon
 * creation of the first user task.
 */
static void prvIdleTask( void *pvParameters );

/*
 * Utility to free all memory allocated by the scheduler to hold a TCB,
 * including the stack pointed to by the TCB.
 *
 * This does not free memory allocated by the task itself (i.e. memory
 * allocated by calls to pvPortMalloc from within the tasks application code).
 */
static void prvDeleteTCB( tskTCB *pxTCB );

/*
 * Used only by the idle task.  This checks to see if anything has been placed
 * in the list of tasks waiting to be deleted.  If so the task is cleaned up
 * and its TCB deleted.
 */
static void prvCheckTasksWaitingTermination( void );

/*
 * Allocates memory from the heap for a TCB and associated stack.  Checks the
 * allocation was successful.
 */
static tskTCB *prvAllocateTCBAndStack( unsigned portSHORT usStackDepth );

/*
 * Called from vTaskList.  vListTasks details all the tasks currently under
 * control of the scheduler.  The tasks may be in one of a number of lists.
 * prvListTaskWithinSingleList accepts a list and details the tasks from
 * within just that list.
 *
 * THIS FUNCTION IS INTENDED FOR DEBUGGING ONLY, AND SHOULD NOT BE CALLED FROM
 * NORMAL APPLICATION CODE.
 */
static void prvListTaskWithinSingleList( portCHAR *pcWriteBuffer, volatile xList *pxList, portCHAR cStatus );


/*lint +e956 */

/*-----------------------------------------------------------
 * PUBLIC TASK MANAGEMENT API documented in task.h
 *----------------------------------------------------------*/

portSHORT sTaskCreate( pdTASK_CODE pvTaskCode, const portCHAR * const pcName, unsigned portSHORT usStackDepth, void *pvParameters, unsigned portCHAR ucPriority, xTaskHandle *pxCreatedTask )
{
portSHORT sReturn;
tskTCB *pxNewTCB;
static unsigned portCHAR ucTaskNumber = 0; /*lint !e956 Static is deliberate - this is guarded before use. */

	/* Allocate the memory required by the TCB and stack for the new task.
	checking that the allocation was successful. */
	pxNewTCB = prvAllocateTCBAndStack( usStackDepth );
	if( pxNewTCB != NULL )
	{
		portLONG lStackSize;
		portSTACK_TYPE *pxTopOfStack;

		/* Setup the newly allocated TCB with the initial state of the task. */
		prvInitialiseTCBVariables( pxNewTCB, usStackDepth, pcName, ucPriority );

		/* Calculate the top of stack address.  This depends on whether the
		stack grows from high memory to low (as per the 80x86) or visa versa.
		portSTACK_GROWTH is used to make the result positive or negative as
		required by the port. */
		lStackSize = ( portLONG ) ( pxNewTCB->usStackDepth); // - ( unsigned portSHORT ) 1 ) * ( portLONG ) portSTACK_GROWTH;
		pxTopOfStack = pxNewTCB->pxStack + lStackSize;

		/* Initialise the TCB stack to look as if the task was already running,
		but had been interrupted by the scheduler.  The return address is set
		to the start of the task function. Once the stack has been initialised
		the	top of stack variable is updated. */
		pxNewTCB->pxTopOfStack = pxPortInitialiseStack( pxTopOfStack, pvTaskCode, pvParameters );

		/* We are going to manipulate the task queues to add this task to a
		ready list, so must make sure no interrupts occur. */
		portENTER_CRITICAL();
		{
			usCurrentNumberOfTasks++;
			if( usCurrentNumberOfTasks == ( unsigned portSHORT ) 1 )
			{
				/* This is the first task to be created so do the preliminary
				initialisation required.  We will not recover if this call
				fails, but we will report the failure. */
				sReturn = prvInitialiseTaskLists();

				/* As this is the first task it must also be the current task. */
				pxCurrentTCB = pxNewTCB;
			}
			else
			{
				sReturn = pdPASS;

				/* If the scheduler is not already running, make this task the
				current task if it is the highest priority task to be created
				so far. */
				if( sSchedulerRunning == pdFALSE )
				{
					if( pxCurrentTCB->ucPriority <= ucPriority )
					{
						pxCurrentTCB = pxNewTCB;
					}
				}
			}

			/* Remember the top priority to make context switching faster.  Use
			the priority in pxNewTCB as this has been capped to a valid value. */
			if( pxNewTCB->ucPriority > ucTopUsedPriority )
			{
				ucTopUsedPriority = pxNewTCB->ucPriority;
			}

			/* Add a counter into the TCB for tracing only. */
			pxNewTCB->ucTCBNumber = ucTaskNumber;
			ucTaskNumber++;

			prvAddTaskToReadyQueue( pxNewTCB );
		}
		portEXIT_CRITICAL();
	}
	else
	{
		sReturn = errCOULD_NOT_ALLOCATE_REQUIRED_MEMORY;
	}

	if( pxCreatedTask != NULL )
	{
		if( sReturn == pdPASS )
		{
			/* Pass the TCB out - in an anonymous way.  The calling function/
			task can use this as a handle to delete the task later if
			required.*/
			*pxCreatedTask = ( xTaskHandle ) pxNewTCB;
		}
	}

	return sReturn;
}
/*-----------------------------------------------------------*/

void vTaskDelete( xTaskHandle pxTaskToDelete )
{
tskTCB *pxTCB;

	taskENTER_CRITICAL();
	{
		/* If null is passed in here then we are deleting ourselves. */
		if( pxTaskToDelete == NULL )
		{
			pxTCB = ( tskTCB * ) pxCurrentTCB;
		}
		else
		{
			pxTCB = ( tskTCB * ) pxTaskToDelete;
		}

		/* Remove task from the ready list and place in the	termination list.
		This will stop the task from be scheduled.  The idle task will check
		the termination list and free up any memory allocated by the
		scheduler for the TCB and stack. */
		vListRemove( &( pxTCB->xGenericListItem ) );
		vListInsert( &xTasksWaitingTermination, &( pxTCB->xGenericListItem ) );
	}
	taskEXIT_CRITICAL();

	/* Force a reschedule - we may have just deleted the current task!. */
	taskYIELD();
}
/*-----------------------------------------------------------*/

void vTaskDelay( unsigned portLONG ulTicksToDelay )
{
unsigned portLONG ulTimeToWake;

	/* A delay time of zero just forces a reschedule. */
	if( ulTicksToDelay > ( unsigned portLONG ) 0 )
	{
		taskENTER_CRITICAL();
		{
			/* Calculate the time to wake - this may overflow but this is not a
			problem. */
			ulTimeToWake = ulTickCount + ulTicksToDelay;

			/* We must remove ourselves from the ready list before adding
			ourselves to the blocked list as the same list item is used for
			both lists. */
			vListRemove( &( pxCurrentTCB->xGenericListItem ) );

			/* The list item will be inserted in wake time order. */
			listSET_LIST_ITEM_VALUE( &( pxCurrentTCB->xGenericListItem ), ulTimeToWake );

			if( ulTimeToWake < ulTickCount )
			{
				/* Wake time has overflowed.  Place this item in the overflow list. */
				vListInsert( pxOverflowDelayedTaskList, &( pxCurrentTCB->xGenericListItem ) );
			}
			else
			{
				/* The wake time has not overflowed, so we can use the current block list. */
				vListInsert( pxDelayedTaskList, &( pxCurrentTCB->xGenericListItem ) );
			}
		}
		taskEXIT_CRITICAL();
	}

	/* Force a reschedule, we may have put ourselves to sleep. */
	taskYIELD();
}

/*-----------------------------------------------------------
 * PUBLIC SCHEDULER CONTROL documented in task.h
 *----------------------------------------------------------*/

void vTaskStartScheduler( portSHORT sUsePreemption )
{
	/* Check a task has been created. */
	if( pxCurrentTCB != NULL )
	{
		sUsingPreemption = sUsePreemption;

		/* Interrupts are turned off here, to ensure a tick does not occur
		before or during the call to sPortStartScheduler().  The stacks of
		the created tasks contain a status word with interrupts switched on
		so interrupts will automatically get re-enabled when the first task
		starts to run.

		STEPPING THROUGH HERE USING A DEBUGGER CAN CAUSE BIG PROBLEMS IF THE
		DEBUGGER ALLOWS INTERRUPTS TO BE PROCESSED. */
		portDISABLE_INTERRUPTS();
		sSchedulerRunning = pdTRUE;
		ulTickCount = ( unsigned portLONG ) 0;

		/* Setting up the timer tick is hardware specific and thus in the
		portable interface. */
		if( sPortStartScheduler( sUsePreemption ) )
		{
			/* Should not reach here as if the scheduler is running the
			function will not return. */
		}
		else
		{
			/* Should only reach here if a task calls sTaskEndScheduler(). */
		}
	}
}
/*-----------------------------------------------------------*/

void vTaskEndScheduler( void )
{
	/* Stop the scheduler interrupts and call the portable scheduler end
	routine so the original ISRs can be restored if necessary.  The port
	layer must ensure interrupts enable	bit is left in the correct state. */
	portDISABLE_INTERRUPTS();
	sSchedulerRunning = pdFALSE;
	vPortEndScheduler();
}

/*-----------------------------------------------------------
 * PUBLIC TASK UTILITIES documented in task.h
 *----------------------------------------------------------*/

volatile unsigned portLONG ulTaskGetTickCount( void )
{
unsigned portLONG ulTicks;

	/* Critical section required if running on a 16 bit processor. */
	taskENTER_CRITICAL();
	{
		ulTicks = ulTickCount;
	}
	taskEXIT_CRITICAL();

	return ulTicks;
}
/*-----------------------------------------------------------*/

unsigned portSHORT usTaskGetNumberOfTasks( void )
{
	return usCurrentNumberOfTasks;
}
/*-----------------------------------------------------------*/

void vTaskList( portCHAR *pcWriteBuffer )
{
unsigned portSHORT usQueue;

	/* This is a VERY costly function that should be used for debug only.
	It leaves interrupts disabled for a LONG time. */

	taskENTER_CRITICAL();
	{
		/* Run through all the lists that could potentially contain a TCB and
		report the task name, state and stack high water mark. */

		pcWriteBuffer[ 0 ] = 0x00;
		strcat( pcWriteBuffer, "\r\n" );

		usQueue = ( unsigned portSHORT ) ucTopUsedPriority + ( unsigned portSHORT ) 1;

		do
		{
			usQueue--;

			if( !listLIST_IS_EMPTY( &( pxReadyTasksLists[ usQueue ] ) ) )
			{
				prvListTaskWithinSingleList( pcWriteBuffer, &( pxReadyTasksLists[ usQueue ] ), tskREADY_CHAR );
			}
		}while( usQueue > ( unsigned portSHORT ) tskIDLE_PRIORITY );

		if( !listLIST_IS_EMPTY( pxDelayedTaskList ) )
		{
			prvListTaskWithinSingleList( pcWriteBuffer, pxDelayedTaskList, tskBLOCKED_CHAR );
		}

		if( !listLIST_IS_EMPTY( pxOverflowDelayedTaskList ) )
		{
			prvListTaskWithinSingleList( pcWriteBuffer, pxOverflowDelayedTaskList, tskBLOCKED_CHAR );
		}

		if( !listLIST_IS_EMPTY( &xTasksWaitingTermination ) )
		{
			prvListTaskWithinSingleList( pcWriteBuffer, &xTasksWaitingTermination, tskDELETED_CHAR );
		}
	}
	taskEXIT_CRITICAL();
}
/*----------------------------------------------------------*/

void vTaskStartTrace( portCHAR * pcBuffer, unsigned portSHORT usBufferSize )
{
	portENTER_CRITICAL();
	{
		pcTraceBuffer = pcBuffer;
		pcTraceBufferStart = pcBuffer;
		pcTraceBufferEnd = pcBuffer + usBufferSize - tskSIZE_OF_EACH_TRACE_LINE;
		cTracing = ( portCHAR ) pdTRUE;
	}
	portEXIT_CRITICAL();
}
/*----------------------------------------------------------*/

unsigned portSHORT usTaskEndTrace( void )
{
unsigned portSHORT usBufferLength;

	portENTER_CRITICAL();
		cTracing = ( portCHAR ) pdFALSE;
	portEXIT_CRITICAL();

	usBufferLength = pcTraceBuffer - pcTraceBufferStart;

	return usBufferLength;
}

/*-----------------------------------------------------------
 * SCHEDULER INTERNALS AVAILABLE FOR PORTING PURPOSES
 * documented in task.h
 *----------------------------------------------------------*/

void vTaskIncrementTick( void )
{
	/* Called by the portable layer each time a tick interrupt occurs.
	Increments the tick then checks to see if the new tick value will cause any
	tasks to be unblocked. */
	++ulTickCount;
	if( ulTickCount == ( unsigned portLONG ) 0 )
	{
		volatile xList *pxTemp;

		/* Tick count has overflowed so we need to swap the delay lists.  If there are
		any items in pxdelayedTaskList here then there is an error! */
		pxTemp = pxDelayedTaskList;
		pxDelayedTaskList = pxOverflowDelayedTaskList;
		pxOverflowDelayedTaskList = pxTemp;
	}

	/* See if this tick has made a timeout expire. */
	prvCheckDelayedTasks();
}
/*-----------------------------------------------------------*/

void vTaskCleanUpResources( void )
{
unsigned portSHORT usQueue;
tskTCB *pxTCB;

	usQueue = ( unsigned portSHORT ) ucTopUsedPriority + ( unsigned portSHORT ) 1;

	/* Remove any TCB's from the ready queues. */
	do
	{
		usQueue--;

		while( !listLIST_IS_EMPTY( &( pxReadyTasksLists[ usQueue ] ) ) )
		{
			pxTCB = ( tskTCB * ) pvListGetOwnerOfNextEntry( &( pxReadyTasksLists[ usQueue ] ) );
			vListRemove( &( pxTCB->xGenericListItem ) );

			prvDeleteTCB( pxTCB );
		}
	}while( usQueue > tskIDLE_PRIORITY );

	/* Remove any TCB's from the delayed queue. */
	while( !listLIST_IS_EMPTY( &xDelayedTaskList1 ) )
	{
		pxTCB = ( tskTCB * ) pvListGetOwnerOfNextEntry( &xDelayedTaskList1 );
		vListRemove( &( pxTCB->xGenericListItem ) );

		prvDeleteTCB( pxTCB );
	}

	/* Remove any TCB's from the overflow delayed queue. */
	while( !listLIST_IS_EMPTY( &xDelayedTaskList2 ) )
	{
		pxTCB = ( tskTCB * ) pvListGetOwnerOfNextEntry( &xDelayedTaskList2 );
		vListRemove( &( pxTCB->xGenericListItem ) );

		prvDeleteTCB( pxTCB );
	}
}
/*-----------------------------------------------------------*/

void vTaskSwitchContext( void )
{
register unsigned portCHAR ucPriority;

	/* Find the highest priority queue that contains ready tasks. */
	ucPriority = ucTopUsedPriority;
	while( listLIST_IS_EMPTY( &( pxReadyTasksLists[ ucPriority ] ) ) )
	{
		--ucPriority;
	}

	/* pvListGetOwnerOfNextEntry walks through the list, so the tasks of the
	same priority get an equal share of the processor time. */
	pxCurrentTCB = ( tskTCB * ) pvListGetOwnerOfNextEntry( &( pxReadyTasksLists[ ucPriority ] ) );
	vWriteTraceToBuffer();
}
/*-----------------------------------------------------------*/

void vTaskPlaceOnEventList( xList *pxEventList, unsigned portLONG ulTicksToWait )
{
unsigned portLONG ulTimeToWake;

	/* THIS FUNCTION MUST BE CALLED WITH INTERRUPTS DISABLED. */

	/* Place the event list item of the TCB in the appropriate event list.
	This is placed in the list in priority order so the highest priority task
	is the first to be woken by the event. */
	vListInsert( pxEventList, &( pxCurrentTCB->xEventListItem ) );

	/* Calculate the time at which the task should be woken if the event does
	not occur.  This may overflow but this doesn't matter. */
	ulTimeToWake = ulTickCount + ulTicksToWait;

	/* We must remove ourselves from the ready list before adding ourselves
	to the blocked list as the same list item is used for both lists. */
	vListRemove( &( pxCurrentTCB->xGenericListItem ) );
	listSET_LIST_ITEM_VALUE( &( pxCurrentTCB->xGenericListItem ), ulTimeToWake );

	if( ulTimeToWake < ulTickCount )
	{
		/* Wake time has overflowed.  Place this item in the overflow list. */
		vListInsert( pxOverflowDelayedTaskList, &( pxCurrentTCB->xGenericListItem ) );
	}
	else
	{
		/* The wake time has not overflowed, so we can use the current block list. */
		vListInsert( pxDelayedTaskList, &( pxCurrentTCB->xGenericListItem ) );
	}
}
/*-----------------------------------------------------------*/

portCHAR cTaskRemoveFromEventList( const xList *pxEventList )
{
tskTCB *pxUnblockedTCB;

	/* THIS FUNCTION MUST BE CALLED WITH INTERRUPTS DISABLED.

	The event list is sorted in priority order, so we can remove the
	first in the list, remove the TCB from the delayed list, and add
	it to the ready list. */
	pxUnblockedTCB = ( tskTCB * ) pvListGetOwnerOfHeadEntry( pxEventList );
	vListRemove( &( pxUnblockedTCB->xEventListItem ) );
	vListRemove( &( pxUnblockedTCB->xGenericListItem ) );
	prvAddTaskToReadyQueue( pxUnblockedTCB );

	if( pxUnblockedTCB->ucPriority > pxCurrentTCB->ucPriority )
	{
		/* Return true if the task removed from the event list has
		a higher priority than the calling task.  This allows
		the calling task to know if it should force a context
		switch now. */
		return ( portCHAR ) pdTRUE;
	}
	else
	{
		return ( portCHAR ) pdFALSE;
	}
}

/*-----------------------------------------------------------
 * File private functions documented at the top of the file.
 *----------------------------------------------------------*/

static void prvInitialiseTCBVariables( tskTCB *pxTCB, unsigned portSHORT usStackDepth, const portCHAR * const pcName, unsigned portCHAR ucPriority )
{
	pxTCB->usStackDepth = usStackDepth;

	/* Make sure the name is not too long before copying it into the TCB. */
	if( strlen( pcName ) < tskMAX_TASK_NAME_LEN )
	{
		portENTER_CRITICAL();
			strcpy( pxTCB->pcTaskName, pcName );
		portEXIT_CRITICAL();
	}
	else
	{
		portENTER_CRITICAL();
			strncpy( pxTCB->pcTaskName, pcName, tskMAX_TASK_NAME_LEN );
		portEXIT_CRITICAL();
		pxTCB->pcTaskName[ tskMAX_TASK_NAME_LEN - ( unsigned portSHORT ) 1 ] = '\0';
	}

	/* This is used as an array index so must ensure it's not too large. */
	if( ucPriority >= ( unsigned portCHAR ) portMAX_PRIORITIES )
	{
		ucPriority = ( unsigned portCHAR ) portMAX_PRIORITIES - ( unsigned portCHAR ) 1;
	}
	pxTCB->ucPriority = ucPriority;

	vListInitialiseItem( &( pxTCB->xGenericListItem ) );
	vListInitialiseItem( &( pxTCB->xEventListItem ) );

	/* Set the pxTCB as a link back from the xListItem.  This is so we can get
	back to	the containing TCB from a generic item in a list. */
	listSET_LIST_ITEM_OWNER( &( pxTCB->xGenericListItem ), pxTCB );

	/* Event lists are always in priority order. */
	listSET_LIST_ITEM_VALUE( &( pxTCB->xEventListItem ), ( unsigned portLONG ) ucPriority );
	listSET_LIST_ITEM_OWNER( &( pxTCB->xEventListItem ), pxTCB );
}
/*-----------------------------------------------------------*/

static portSHORT prvInitialiseTaskLists( void )
{
register unsigned portCHAR ucPriority;
portSHORT sReturn;

	for( ucPriority = ( unsigned portCHAR) 0; ucPriority < ( unsigned portCHAR ) portMAX_PRIORITIES; ucPriority++ )
	{
		vListInitialise( &( pxReadyTasksLists[ ucPriority ] ) );
	}

	vListInitialise( &xDelayedTaskList1 );
	vListInitialise( &xDelayedTaskList2 );
	vListInitialise( &xTasksWaitingTermination );

	/* Start with pxDelayedTaskList using list1 and the pxOverflowDelayedTaskList
	using list2. */
	pxDelayedTaskList = &xDelayedTaskList1;
	pxOverflowDelayedTaskList = &xDelayedTaskList2;

	/* Add the idle task at the lowest priority. */
	sReturn = sTaskCreate( prvIdleTask, "IDLE", tskIDLE_STACK_SIZE, ( void * ) NULL, tskIDLE_PRIORITY, NULL );

	return sReturn;
}
/*-----------------------------------------------------------*/

static void prvAddTaskToReadyQueue( tskTCB *pxTCB )
{
	/* All TCB's in a ready queue have the same priority, so the item value is
	not important.  The item is placed at the back of the list which is
	effectively unsorted. */
	listSET_LIST_ITEM_VALUE( &( pxTCB->xGenericListItem ), 0 );
	vListInsertEnd( &( pxReadyTasksLists[ pxTCB->ucPriority ] ), &( pxTCB->xGenericListItem ) );
}
/*-----------------------------------------------------------*/

static void prvIdleTask( void *pvParameters )
{
	for( ;; )
	{
		/* See if any tasks have been deleted. */
		prvCheckTasksWaitingTermination();
		if( sUsingPreemption == pdFALSE )
		{
			/* If we are not using preemption we keep forcing a task switch to
			see if any other task has become available.  If we are using
			preemption we don't need to do this as any task becoming available
			will automatically get the processor anyway. */
			taskYIELD();
		}
	}
} /*lint !e715 pvParameters is not accessed but all task functions require the same prototype. */
/*-----------------------------------------------------------*/

static void prvCheckTasksWaitingTermination( void )
{
	if( !listLIST_IS_EMPTY( &xTasksWaitingTermination ) )
	{
		tskTCB *pxTCB;

		portENTER_CRITICAL();
		{
			pxTCB = ( tskTCB * ) pvListGetOwnerOfHeadEntry( &xTasksWaitingTermination );
			vListRemove( &( pxTCB->xGenericListItem ) );
		}
		portEXIT_CRITICAL();
		prvDeleteTCB( pxTCB );
		--usCurrentNumberOfTasks;
	}
}
/*-----------------------------------------------------------*/

static tskTCB *prvAllocateTCBAndStack( unsigned portSHORT usStackDepth )
{
tskTCB *pxNewTCB;

	/* Allocate space for the TCB.  Where the memory comes from depends on
	the implementation of the port malloc function. */
	pxNewTCB = ( tskTCB * ) pvPortMalloc( sizeof( tskTCB ) );

	if( pxNewTCB != NULL )
	{
		/* Allocate space for the stack used by the task being created.
		The base of the stack memory stored in the TCB so the task can
		be deleted later if required. */
		pxNewTCB->pxStack = ( portSTACK_TYPE * ) pvPortMalloc(usStackDepth);

		if( pxNewTCB->pxStack == NULL )
		{
			/* Could not allocate the stack.  Delete the allocated TCB. */
			vPortFree( pxNewTCB );
			pxNewTCB = NULL;
		}
		else
		{
			/* Just to help debugging. */
			memset( pxNewTCB->pxStack, tskSTACK_FILL_BYTE, usStackDepth * sizeof( portSTACK_TYPE ) );
		}
	}

	return pxNewTCB;
}
/*-----------------------------------------------------------*/

static void prvListTaskWithinSingleList( portCHAR *pcWriteBuffer, volatile xList *pxList, portCHAR cStatus )
{
#ifdef USE_STDIO
volatile tskTCB *pxNextTCB, *pxFirstTCB;
static portCHAR pcStatusString[ 50 ];
unsigned portSHORT usStackRemaining;

	/* Write the details of all the TCB's in pxList into the buffer. */
	pxFirstTCB = pvListGetOwnerOfNextEntry( pxList );
	do
	{
		pxNextTCB = pvListGetOwnerOfNextEntry( pxList );
		usStackRemaining = usPortCheckFreeStackSpace( ( unsigned portCHAR * ) pxNextTCB->pxStack, tskSTACK_FILL_BYTE );
		sprintf( pcStatusString, "%s\t\t%c\t%u\t%u\t%d\r\n", pxNextTCB->pcTaskName, cStatus, pxNextTCB->ucPriority, usStackRemaining, pxNextTCB->ucTCBNumber );
		strcat( pcWriteBuffer, pcStatusString );

	} while( pxNextTCB != pxFirstTCB );
#endif /*USE_STDIO*/
}
/*-----------------------------------------------------------*/

static void prvDeleteTCB( tskTCB *pxTCB )
{
	/* Free up the memory allocated by the scheduler for the task.  It is up to
	the task to free any memory allocated at the application level. */
	vPortFree( pxTCB->pxStack );
	vPortFree( pxTCB );
}









