#region GNU General Public License
//
// NAntContrib
// Copyright (C) 2001-2003 Gerry Shaw
//
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
//

// Jason Reimer, Diversant Inc. (jason.reimer@diversant.net)
// William E. Caputo, ThoughtWorks Inc. (billc@thoughtworks.com)
#endregion

using System;
using System.Diagnostics;
using System.Text;

using NAnt.Core;
using NAnt.Core.Attributes;

namespace NAnt.Contrib.Tasks.Vault {

    /// <summary>
    /// Merge action to be taken for modified files
    /// in local folder.
    /// </summary>
    public enum Merge {
        Automatic,
        Later,
        Overwrite
    }

    /// <summary>
    /// Method used to set modification time of files
    /// retrieved from the repository.
    /// </summary>
    public enum FileTime {
        Checkin,
        Current,
        Modification
    }

    /// <summary>
    /// Generic encapsulation of a task carried out on a
    /// SourceGear Vault repository.
    /// Author: Bart Read
    /// </summary>
    public abstract class VaultTask : Task {

        string		_exeName		= @"C:\Program Files\SourceGear\Vault Client\vault.exe";
        string		_host			= "localhost";
        bool		_ssl			= true;
        string		_username;
        string		_password;
        string		_repository;
        string		_repositoryPath;

        /// <summary>
        /// Name of Vault executable.  Default is
        /// 'C:\Program Files\SourceGear\Vault Client\vault.exe'.
        /// </summary>
        [TaskAttribute("exename")]
        public string ExeName {
            get { return _exeName; }
            set { _exeName = value; }
        }

        /// <summary>
        /// Hostname of the Vault server to connect to.
        /// Default is 'localhost'.
        /// </summary>
        [TaskAttribute("host")]
        public string Host {
            get { return _host; }
            set { _host = value; }
        }

        /// <summary>
        /// Enable SSL for server connection.  Default is true.
        /// </summary>
        [TaskAttribute("ssl")]
        public bool SSL {
            get { return _ssl; }
            set { _ssl = value; }
        }

        /// <summary>
        /// Username to use when connecting to server.
        /// </summary>
        [TaskAttribute("username", Required=true)]
        public string Username {
            get { return _username; }
            set { _username = value; }
        }

        /// <summary>
        /// Password to use when connecting to server.
        /// </summary>
        [TaskAttribute("password")]
        public string Password {
            get { return _password; }
            set { _password = value; }
        }

        /// <summary>
        /// Repository to which to connect.
        /// </summary>
        [TaskAttribute("repository", Required=true)]
        public string Repository {
            get { return _repository; }
            set { _repository = value; }
        }

        /// <summary>
        /// Path of files / folders in repository.  Usage
        /// depends on action.
        /// </summary>
        [TaskAttribute("repositorypath", Required=true)]
        public string RepositoryPath {
            get { return _repositoryPath; }
            set { _repositoryPath = value; }
        }

        protected string CreateCommonArguments() {
            StringBuilder arguments = new StringBuilder();

            arguments.Append(" -host ");
            arguments.Append(Host);

            if (SSL) {
                arguments.Append(" -ssl");
            }

            arguments.Append(" -username ");
            arguments.Append("\"" + Username + "\"");

            if (null != Password) {
                arguments.Append(" -password ");
                arguments.Append("\"" + Password + "\"");
            }

            arguments.Append(" -repository ");
            arguments.Append("\"" + Repository + "\"");

            return arguments.ToString();
        }

        protected abstract string CreateArguments();

        protected override void ExecuteTask() {
            try {
                Process p = new Process();
                p.StartInfo.FileName	= ExeName;
                p.StartInfo.Arguments	= CreateArguments();
                p.StartInfo.RedirectStandardOutput = true;
                p.StartInfo.UseShellExecute = false;
                p.Start();
                string output = p.StandardOutput.ReadToEnd();
                Log(Level.Info, "Output from '" + ExeName + "'");
                Log(Level.Info, output);
                p.WaitForExit();
                if (p.ExitCode != 0) {
                    throw new BuildException(
                        "Error executing " + Name + " task. '" + ExeName + "' returned a non-zero exit code: " + p.ExitCode);
                }
            } catch ( Exception e ) {
                throw new BuildException("Error executing " + Name + " task: " + e.Message, e);
            }
        }

    }

}