#include <net-snmp/net-snmp-config.h>
#include <net-snmp/net-snmp-includes.h>
#include <net-snmp/library/snmp_api.h>
#include <net-snmp/library/snmp.h>


/*
	This demo is a simplified version of the demo at the net-snmp project web page.
	
	This demo expects two parameters:
		1st: the IP address of the SNMP agent
		2nd: the community name to use
	
	Note that we are using SNMPv1.
	For an SNMPv3 example check the original example at the net-snmp project web page.
	*/
	

int main(int argc, char ** argv) {

	

	
	/*
      The first variables we need to declare:

         1. struct snmp_session: A structure that holds information
            about who we're going to be talking to. We need to declare 2
            of these, one to fill with information, and second which is
            a pointer returned by the library.
         2. struct snmp_pdu: This structure will hold all of the
            information that we're going to send to the remote host.
            We'll declare a second for the information they they are
            going to send back.
         3. oid: An OID is going to hold the location of the information
            which we want to retrieve. It'll need a size as well.
         4. struct variable_list: This will hold the variables that we
            want to manipulate via SNMP.
         */

    struct snmp_session session, *ss;
    struct snmp_pdu *pdu;
    struct snmp_pdu *response;
	
    oid anOID[MAX_OID_LEN];
    size_t anOID_len = MAX_OID_LEN;

    struct variable_list *vars;
    int status;


/* Then, the first thing we must do is to inialize the snmp library: */

    /*
     * Initialize the SNMP library
     */
    init_snmp("snmpapp");


/*Next, we'll inialize a session that describes who we want to talk to,
what version of SNMP we want to use, how to authenticate to it, etc. A
full definition of this session can be found in the net-snmp/snmp_api.h
header file.
*/


     /*
     * Initialize a "session" that defines who we're going to talk to
     */
    snmp_sess_init( &session );                   /* set up defaults */
    session.peername = argv[1];

    	 printf("Session Peername:%s\n",session.peername);

    /* set up the authentication parameters for talking to the server */


    /* set the SNMP version number */
    session.version = SNMP_VERSION_1;

    /* set the SNMPv1 community name used for authentication */
    session.community = argv[2];
    session.community_len = strlen(session.community);
    session.timeout=1000;

/*After we have established the session, we then need to open it. Opening
it returns a pointer to another session that we should use for all our
future calls:
*/
/*Enabling the SNMP STDERR Log*/
	snmp_enable_stderrlog();
       /*
     * Open the session
     */
    ss = snmp_open(&session);                     /* establish the session */

/*If the session failed to open properly, print an error message and exit: */

    if (!ss) {
        snmp_perror("ack");
        snmp_log(LOG_ERR, "something horrible happened!!!\n");
        exit(2);
    }

/*Next we create the PDU the we are going to send to the remote host when
we request information. In this case, we're going to create a SNMP-GET
pdu, which is what the snmpget <../../commands/snmpget.html> program
uses, for instance. It retrieves a value for each oid that you initalize
it with.*/

    /*
     * Create the PDU for the data for our request.
     *   1) We're going to GET the system.sysDescr.0 node.
     */
    pdu = snmp_pdu_create(SNMP_MSG_GET);

/*So, let's fill it with our requested oid. Let's get the
system.sysDescr.0 variable for this example. There are numerious ways
you could create the oid in question. We recommend "get_node",
as it is the most powerful and
accepts more types of OIDs. */


//    get_node("sysDescr.0", anOID, &anOID_len);
      read_objid(".1.3.6.1.2.1.1.1.0", anOID, &anOID_len);

/*So we add this oid, with a NULL value to the PDU using the following
statement: (all oids should be paired with a NULL value for outgoing
requests for information. For an SNMP-SET pdu, we'd put in the value we
wanted to set the oid to).
*/

    snmp_add_null_var(pdu, anOID, anOID_len);


/*Finally, we can send out the request using the session pointer and the
pdu and we get back a status and the response, which is stored in the
newly malloced response pdu pointer.
*/

    /*
     * Send the Request out.
     */
//	printf("Community for outging:%s\n",(char*)pdu->community);
    status = snmp_synch_response(ss, pdu, &response);

	/*Analyzing the data returned aftr sending the PDU .
	  Printing the data from response*/
	printf("Status Value:%d\n",status);
	

/*Now we can examine the information, assuming our request was fulfilled
properly. If not, we'll print the error that caused the problem:
*/

    /*
     * Process the response.
     */
    if (status == STAT_SUCCESS && response->errstat == SNMP_ERR_NOERROR) {
      /*
       * SUCCESS:
       */
		printf("Processing the PDU returned\n");
      for(vars = response->variables; vars; vars = vars->next_variable) {
        int count=1;
        if (vars->type == ASN_OCTET_STR) {
	  /* sysDescr.0 is a string, so we are expecting a string*/
	  char *sp = (char *)malloc(1 + vars->val_len);
	  memcpy(sp, vars->val.string, vars->val_len);
	  sp[vars->val_len] = '\0';
          printf("value #%d is a string: %s\n", count++, sp);
	  free(sp);
	}
        else
          printf("value #%d is NOT a string! Ack!\n", count++);
      }
	

      /*Finally, print a description of the error in case there was one:*/

    } else {
      /*
       * FAILURE: print what went wrong!
       */

      if (status == STAT_SUCCESS)
        fprintf(stderr, "Error in packet\nReason: %s\n",
                snmp_errstring(response->errstat));
      else
        snmp_sess_perror("snmpget", ss);

    }
	

/*Last, but not least, free the response, as it is now our job to do so.
Exit cleanly by calling snmp_close() as well:
*/

    /*
     * Clean up:
     *  1) free the response.
     *  2) close the session.
     */
    if (response)
      snmp_free_pdu(response);
    snmp_close(ss);


}


