#include <sys/socket.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <linux/sctp.h>
#include <arpa/inet.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <netdb.h>
#include <alloca.h>

char *key_text = NULL;

int sctp_set_auth_key(int fd, sctp_assoc_t assoc_id, uint16_t keyid,
                      uint16_t keylen, uint8_t *keytext) {
        socklen_t len;
        struct sctp_authkey *akey;
        int result;

        len = sizeof(*akey) + keylen;
        akey = (struct sctp_authkey *)alloca(len);
        if (akey == NULL) {
                printf("could not get memory for akey\n");
                return (-1);
        }
        akey->sca_assoc_id = assoc_id;
        akey->sca_keynumber = keyid;
        akey->sca_keylength = keylen;
        bcopy(keytext, akey->sca_key, keylen);
        result = setsockopt(fd, IPPROTO_SCTP, SCTP_AUTH_KEY, akey, len);
        return (result);
}

int sctp_set_active_key(int fd, sctp_assoc_t assoc_id, uint16_t keyid) {
        socklen_t len;
        struct sctp_authkeyid akey;
        int result;

        len = sizeof(akey);
        akey.scact_assoc_id = assoc_id;
        akey.scact_keynumber = keyid;
        result = setsockopt(fd, IPPROTO_SCTP, SCTP_AUTH_ACTIVE_KEY,
                            &akey, len);
        return (result);
}

int sctp_set_auth_chunk_id(int fd, uint8_t chk)
{
        int result;
        socklen_t len;
        struct sctp_authchunk ch;

        len = sizeof(ch);
        ch.sauth_chunk = chk;
        result = setsockopt(fd, IPPROTO_SCTP, SCTP_AUTH_CHUNK,
                            &ch, len);
        return(result);

}

int handle_notification(int fd,char *notify_buf)
{
	union sctp_notification *snp;
	struct sctp_assoc_change *sac;
	struct sctp_paddr_change *spc;
	struct sctp_remote_error *sre;
	struct sctp_send_failed *ssf;
	struct sctp_shutdown_event *sse;
	struct sctp_authkey_event *auth;
#if defined(__BSD_SCTP_STACK__)
	struct sctp_stream_reset_event *strrst;
#endif
	int asocDown;
	char *str;
	char buf[256];
	struct sockaddr_in *sin;
	struct sockaddr_in6 *sin6;

	asocDown = 0;
	snp = (union sctp_notification *)notify_buf;
	switch(snp->sn_header.sn_type) {
	case SCTP_ASSOC_CHANGE:
		sac = &snp->sn_assoc_change;
		switch(sac->sac_state) {

		case SCTP_COMM_UP:
			str = "COMMUNICATION UP";
			break;
		case SCTP_COMM_LOST:
			str = "COMMUNICATION LOST";
			asocDown = 1;
			break;
		case SCTP_RESTART:
			str = "RESTART";
			break;
		case SCTP_SHUTDOWN_COMP:
			str = "SHUTDOWN COMPLETE";
			asocDown = 1;
			break;
		case SCTP_CANT_STR_ASSOC:
			str = "CANT START ASSOC";
			asocDown = 1;
			break;
		default:
			str = "UNKNOWN";
		} /* end switch(sac->sac_state) */
		printf("SCTP_ASSOC_CHANGE: %s, sac_error=0x%x assoc=0x%x\n",
		       str,
		       (uint32_t)sac->sac_error,
		       (uint32_t)sac->sac_assoc_id);
		break;
	case SCTP_PEER_ADDR_CHANGE:
		spc = &snp->sn_paddr_change;
		switch(spc->spc_state) {
		case SCTP_ADDR_AVAILABLE:
			str = "ADDRESS AVAILABLE";
			break;
		case SCTP_ADDR_UNREACHABLE:
			str = "ADDRESS UNAVAILABLE";
			break;
		case SCTP_ADDR_REMOVED:
			str = "ADDRESS REMOVED";
			break;
		case SCTP_ADDR_ADDED:
			str = "ADDRESS ADDED";
			break;
		case SCTP_ADDR_MADE_PRIM:
			str = "ADDRESS MADE PRIMARY";
			break;
#if defined(__BSD_SCTP_STACK__)
		case SCTP_ADDR_CONFIRMED:
			str = "ADDRESS CONFIRMED";
			break;
#endif
		default:
			str = "UNKNOWN";
		} /* end switch */
		sin6 = (struct sockaddr_in6 *)&spc->spc_aaddr;
		if (sin6->sin6_family == AF_INET6) {
			char scope_str[16];
			snprintf(scope_str, sizeof(scope_str)-1, " scope %u",
				 sin6->sin6_scope_id);
			inet_ntop(AF_INET6, (char*)&sin6->sin6_addr, buf, sizeof(buf));
			strcat(buf, scope_str);
		} else {
			sin = (struct sockaddr_in *)&spc->spc_aaddr;
			inet_ntop(AF_INET, (char*)&sin->sin_addr, buf, sizeof(buf));
		}
		printf("SCTP_PEER_ADDR_CHANGE: %s, addr=%s, assoc=0x%x\n",
		       str, buf, (uint32_t)spc->spc_assoc_id);
		break;
	case SCTP_REMOTE_ERROR:
		sre = &snp->sn_remote_error;
		printf("SCTP_REMOTE_ERROR: assoc=0x%x\n",
		       (uint32_t)sre->sre_assoc_id);
		break;
	// Support SCTP_AUTH_NEWKEY only on Linux currently
	case SCTP_AUTHENTICATION_INDICATION:
	{
		auth = (struct sctp_authkey_event *)&snp->sn_authkey_event;
		printf("SCTP_AUTHKEY_EVENT: assoc=0x%x - ",
		       (uint32_t)auth->auth_assoc_id);
		switch(auth->auth_indication) {
		case SCTP_AUTH_NEWKEY:
			printf("AUTH_NEWKEY");
			break;
#if defined(__BSD_SCTP_STACK__)
		case SCTP_AUTH_NO_AUTH:
			printf("AUTH_NO_AUTH");
			break;
		case SCTP_AUTH_FREE_KEY:
			printf("AUTH_FREE_KEY");
			break;
#endif
		default:
			printf("Indication 0x%x", auth->auth_indication);
		}
		printf(" key %u, alt_key %u\n", auth->auth_keynumber,
		       auth->auth_altkeynumber);
		break;
	}
#if defined(__BSD_SCTP_STACK__)
	case SCTP_SENDER_DRY_EVENT:
	  break;
	case SCTP_STREAM_RESET_EVENT:
	{
#if defined(UPDATED_SRESET)
		int len;
		char *strscope="unknown";
#endif
		strrst = (struct sctp_stream_reset_event *)&snp->sn_strreset_event;
		printf("SCTP_STREAM_RESET_EVENT: assoc=0x%x\n",
		       (uint32_t)strrst->strreset_assoc_id);
#if defined(UPDATED_SRESET)
		if(strrst->strreset_flags & SCTP_STRRESET_FAILED) {
			printf("Failed\n");
			break;
		}
		if (strrst->strreset_flags & SCTP_STRRESET_INBOUND_STR) {
			strscope = "inbound";
		} else if (strrst->strreset_flags & SCTP_STRRESET_OUTBOUND_STR) {
			strscope = "outbound";
		}
		if (strrst->strreset_flags & SCTP_STRRESET_ADD_STREAM) {
		  printf("Added streams %s new stream total is:%d\n",
				 strscope,
				 strrst->strreset_list[0]
				 );
		  break;
		}
		if(strrst->strreset_flags & SCTP_STRRESET_ALL_STREAMS) {
			printf("All %s streams have been reset\n",
			       strscope);
		} else {
			int i,cnt=0;
			len = ((strrst->strreset_length - sizeof(struct sctp_stream_reset_event))/sizeof(uint16_t));
			printf("Streams ");
			for ( i=0; i<len; i++){
				cnt++;
				printf("%d",strrst->strreset_list[i]);
				if ((cnt % 16) == 0) {
					printf("\n");
				} else {
					printf(",");
				}
			}
			if((cnt % 16) == 0) {
				/* just put out a cr */
				printf("Have been reset %s\n",strscope);
			} else {
				printf(" have been reset %s\n",strscope);
			}
		}
#endif
	}
	break;
#endif
	case SCTP_SEND_FAILED:
	{
		char *msg;
		static char msgbuf[200];
		ssf = &snp->sn_send_failed;
		if(ssf->ssf_flags == SCTP_DATA_UNSENT)
			msg = "data unsent";
		else if(ssf->ssf_flags == SCTP_DATA_SENT)
			msg = "data sent";
		else{
			sprintf(msgbuf,"unknown flags:%d",ssf->ssf_flags);
			msg = msgbuf;
		}
		printf("SCTP_SEND_FAILED: assoc=0x%x flag indicate:%s\n",
		       (uint32_t)ssf->ssf_assoc_id,msg);

	}

		break;
	case SCTP_ADAPTATION_INDICATION:
	  {
	    struct sctp_adaptation_event *ae;
	    ae = &snp->sn_adaptation_event;
	    printf("SCTP_ADAPTION_INDICATION: assoc=0x%x - indication:0x%x\n",
		   (uint32_t)ae->sai_assoc_id, (uint32_t)ae->sai_adaptation_ind);
	  }
	  break;
	case SCTP_PARTIAL_DELIVERY_EVENT:
	  {
	    struct sctp_pdapi_event *pdapi;

	    pdapi = &snp->sn_pdapi_event;
	    printf("SCTP_PD-API event:%u\n",
		   pdapi->pdapi_indication);
	    if(pdapi->pdapi_indication == SCTP_PARTIAL_DELIVERY_ABORTED){
	      printf("PDI - Aborted\n");
	    }
	  }
	  break;

	case SCTP_SHUTDOWN_EVENT:
		sse = &snp->sn_shutdown_event;
		printf("SCTP_SHUTDOWN_EVENT: assoc=0x%x\n",
		       (uint32_t)sse->sse_assoc_id);
		break;
	default:
		printf("Unknown notification event type=0x%x\n",
		       snp->sn_header.sn_type);
	} /* end switch(snp->sn_header.sn_type) */
	return asocDown;
}

static int handle_input(int fd,int maxread)
{
	struct iovec iov[2];
	unsigned char from[200];
	char read_buf[65535];	
	int sz=0;
	char ctrl_vector[65535];
	struct msghdr msg;

	memset(&msg, 0, sizeof(msg));
	memset(ctrl_vector, 0, sizeof(ctrl_vector));
	memset(read_buf, 0, sizeof(read_buf));

	iov[0].iov_base = read_buf;
	iov[0].iov_len = maxread;
	iov[1].iov_base = NULL;
	iov[1].iov_len = 0;
	msg.msg_name = (void *)from;
	msg.msg_namelen = sizeof(from);
	msg.msg_iov = iov;
	msg.msg_iovlen = 1;
	msg.msg_control = (void *)ctrl_vector;
	msg.msg_controllen = sizeof(ctrl_vector);
	errno = 0;

	sz = recvmsg(fd, &msg, MSG_WAITALL);
	printf("Read fd:%d returns %d errno:%d control len is %zu msgflg:%x\n",
	       fd,
	       sz,errno,
	       msg.msg_controllen,
	       msg.msg_flags);

	if (msg.msg_flags & MSG_NOTIFICATION) {
		printf("Got a notification\n");
		return(handle_notification(fd, read_buf));
	}else{
		printf("Got data\n");
		return 0;
	}
}

static int sn_do_setup_frag_interleave(int fd)
{
	int param, param_len;

	param = 1;
	param_len = sizeof(param);
	if (setsockopt(fd, IPPROTO_SCTP, SCTP_FRAGMENT_INTERLEAVE,
		       &param, sizeof(param))) {
		perror("setsockopt");
		exit(-1);
	}

	return 0;
}

static int sn_do_setup_strm_interleave(int fd)
{
	struct sctp_assoc_value param;

	param.assoc_id = 0;
	param.assoc_value = sizeof(param);

	if (setsockopt(fd, IPPROTO_SCTP, SCTP_INTERLEAVING_SUPPORTED,
		       &param, sizeof(param))) {
		perror("setsockopt");
	}

	return 0;
}
int main(int argc,char *argv[])  
{
	struct sctp_event_subscribe subscribe;
	struct addrinfo *res;
	int fd, error, chk = 0x40;
	uint16_t keyid, keylen;

        if (argc != 3) {
                printf("Usage: %s remote_ip_addr remote_port\n", argv[0]);
                return 1;
        }

	if (getaddrinfo(argv[1], argv[2], NULL, &res) != 0) {
		perror("getaddrinfo error");
		return 1;
	}

	if((fd = socket(res->ai_family, SOCK_SEQPACKET, IPPROTO_SCTP)) < 0)
	{
		perror("socket error");
		return(1);
	}

	sn_do_setup_frag_interleave(fd);
	sn_do_setup_strm_interleave(fd);

	memset(&subscribe, 0, sizeof(subscribe));
//	subscribe.sctp_data_io_event = 1;
//	subscribe.sctp_association_event = 1;
	subscribe.sctp_authentication_event = 1;
	error = setsockopt(fd, IPPROTO_SCTP, SCTP_EVENTS,
		(char *)&subscribe, sizeof(subscribe));
	if (error) {
		printf("SCTP_EVENTS: error: %d\n", error);
		return(1);
	}

	key_text = "AUTHKEY1";
	keylen = strlen(key_text);
	keyid = 0;
	error = sctp_set_auth_key(fd, 0, keyid, keylen, (uint8_t *)key_text);
	if (error != 0) {
		printf("Set Auth Key:%s error\n", key_text);
		return(1);
	}

	key_text = "AUTHKEY2";
	keylen = strlen(key_text);
	keyid = 1;
	error = sctp_set_auth_key(fd, 0, keyid, keylen, (uint8_t *)key_text);
	if (error != 0) {
		printf("Set Auth Key:%s error\n", key_text);
		return(1);
	}

	/* Set keyid(1) to be active key
	 * Peer set keyid(0) to be active key
	 */
	error = sctp_set_active_key(fd, 0, 1);
	if (error != 0) {
		printf("set active key error\n");
		return(1);
	}

	error = sctp_set_auth_chunk_id(fd, chk);
	if (error != 0) {
		printf("Set Auth Chunk:%u error\n", chk);
		return(1);
	}


	if(bind(fd, res->ai_addr, res->ai_addrlen) < 0)
	{
		perror("bind error");
		return(1);
	}

	freeaddrinfo(res);

	if (listen(fd, 10) == -1) {
                perror("listen error");
                return 1;
	}

	while(1)
	{
		error = handle_input(fd, 65535);
		if (error != 0) {
			printf("handle_input error");
		}
		fflush(stdout);
	}
	close(fd);
	return 0;  
}
