/*
 * 	packet.c
 * 
 * 2006 Copyright (c) Evgeniy Polyakov <johnpol@2ka.mipt.ru>
 * All rights reserved.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/poll.h>
#include <sys/time.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <signal.h>
#include <netdb.h>
#include <time.h>

#include <arpa/inet.h>
#include <netpacket/packet.h>

#include <linux/if_ether.h>

#include <netinet/ip.h>
#include <netinet/tcp.h>

#include "sys.h"
#include "stat.h"

static int need_exit;
static int alarm_timeout = 1;

extern unsigned char packet_edst[];
extern int packet_index;

static void term_signal(int signo)
{
	need_exit = signo;
}

static void alarm_signal(int signo __attribute__ ((unused)))
{
	print_stat();
	alarm(alarm_timeout);
}

int transmit_data(struct nc_buff *ncb)
{
	int err;
#if defined UDEBUG
	if (ncb->dst->proto == IPPROTO_TCP) {
		struct iphdr *iph = ncb->nh.iph;
		struct tcphdr *th = ncb->h.th;

		ulog("S %u.%u.%u.%u:%u <-> %u.%u.%u.%u:%u : seq: %u, ack: %u, win: %u, doff: %u, "
			"s: %u, a: %u, p: %u, r: %u, f: %u: tlen: %u.\n",
			NIPQUAD(iph->saddr), ntohs(th->source),
			NIPQUAD(iph->daddr), ntohs(th->dest),
			ntohl(th->seq), ntohl(th->ack_seq), ntohs(th->window), th->doff,
			th->syn, th->ack, th->psh, th->rst, th->fin,
			ntohs(iph->tot_len));
	}
#endif
	err = netchannel_send_raw(ncb);
	if (err)
		return err;

	ncb_put(ncb);
	return 0;
}

static unsigned int packet_convert_addr(char *addr_str, unsigned int *addr)
{
	struct hostent *h;

	h = gethostbyname(addr_str);
	if (!h) {
		ulog_err("%s: Failed to get address of %s", __func__, addr_str);
		return -1;
	}
	
	memcpy(addr, h->h_addr_list[0], 4);
	return 0;
}

static int packet_parse_addr(char *str, unsigned char addr[])
{
	int i;
	char *p = str;

	/*
	 * 00:11:22:33:44:55
	 */
	if (strlen(str) != 2*ETH_ALEN + ETH_ALEN-1) {
		ulog("Wrong ethernet address string '%s', it has to be in the wollowing form: 00:11:22:33:44:55.\n",
				str);
		return -EINVAL;
	}

	for (i=0; i<ETH_ALEN; ++i) {
		p = str+2;
		*p = 0;

		addr[i] = strtoul(str, NULL, 16);

		str = p+1;
	}

	return 0;
}

static void usage(const char *p)
{
	ulog_info("Usage: %s -s saddr -d daddr -S sport -D dport -p proto -l <listen> -o mem_order -h -b size -e eth_dst_addr -i eth_out_index\n", p);
}

int main(int argc, char *argv[])
{
	int err, ch, sent, recv;
	struct unetchannel unc;
	struct netchannel *nc;
	char *saddr, *daddr;
	__u32 src, dst;
	__u16 sport, dport;
	__u8 proto;
	struct nc_route rt;
	char str[4096];
	unsigned int state, order, size;

	srand(time(NULL));

	saddr = "192.168.4.78";
	daddr = "192.168.0.48";
	sport = rand();
	dport = 1025;
	proto = IPPROTO_TCP;
	state = NETCHANNEL_ATCP_CONNECT;
	order = 20;
	size = sizeof(str);

	while ((ch = getopt(argc, argv, "e:i:s:d:S:D:hp:lb:")) != -1) {
		switch (ch) {
			case 'i':
				packet_index = atoi(optarg);
				break;
			case 'e':
				err = packet_parse_addr(optarg, packet_edst);
				if (err)
					return err;
				break;
			case 'b':
				size = atoi(optarg);
				if (size > sizeof(str))
					size = sizeof(str);
				break;
			case 'o':
				order = atoi(optarg);
				break;
			case 'l':
				state = NETCHANNEL_ATCP_LISTEN;
				break;
			case 'p':
				proto = atoi(optarg);
				break;
			case 'D':
				dport = atoi(optarg);
				break;
			case 'S':
				sport = atoi(optarg);
				break;
			case 'd':
				daddr = optarg;
				break;
			case 's':
				saddr = optarg;
				break;
			default:
				usage(argv[0]);
				return 0;
		}
	}

	if (packet_convert_addr(saddr, &src) || packet_convert_addr(daddr, &dst)) {
		usage(argv[0]);
		return -1;
	}

	err = route_init();
	if (err)
		return err;

	rt.header_size = MAX_HEADER_SIZE;
	rt.src = src;
	rt.dst = dst;
	rt.proto = proto;

	err = route_add(&rt);
	if (err)
		return err;

	netchannel_setup_unc(&unc, src, htons(sport), dst, htons(dport), proto, order);
	nc = netchannel_create(&unc, state);
	if (!nc)
		return -EINVAL;

	signal(SIGTERM, term_signal);
	signal(SIGINT, term_signal);
	signal(SIGALRM, alarm_signal);
	init_stat();
	alarm(alarm_timeout);

	sent = recv = 0;

	printf("size: %u.\n", size);

	while (!need_exit) {
		err = netchannel_recv(nc, str, size);
		ulog("%s: recv: err: %d.\n", __func__, err);
		if (err > 0) {
			stat_written += err;
			stat_written_msg++;
			last_fd = nc->hit;
		} else if (err < 0) {
			if (err != -EAGAIN)
				need_exit = 1;
		}
	}

	netchannel_remove(nc);

	return err;
}
