/* apcbzbr.h - APC BZ-BR models (Brazil)

   Copyright (C) 2014  Carlos Guidugli  <guidugli@gmail.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

   2014/09/22 - Version 0.01 - Initial release
 
*/
#include "main.h"
#include "serial.h"
#include "apcbzbr.h" 
#include "math.h"

#include <ctype.h>
#include <stdio.h>

#include "timehead.h"

/* driver description structure */
upsdrv_info_t upsdrv_info = {
	DRIVER_NAME,
	DRIVER_VERSION,
	DRIVER_AUTHOR,
	DRV_STABLE,
	{ NULL }
};

bool_t prefix(const char *pre, const char *str)
{
    return strncmp(pre, str, strlen(pre)) == 0;
}

/*
 * Based on the dataset retrieved by apc_readserial,
   parse results, apply formulas and set variables
   to be used by nut clients
   Dataset:
                0: Header
                1: Out Voltage
                2: In Voltage
                3: Battery Voltage
                4: UPS Temperature
                5: Out Current
                6: Relay Configuration
                7,8: Real power
                9: UPS Time: seconds
                10: UPS Time: minutes
                11: UPS Time: hour
                12: Not used
                13: Time to switch ups on: hour
                14: Time to switch ups on: minute
                15: Time to switch ups off: hour
                16: Time to switch ups off: minute
                17: Weekdays to turn ups on/off
                18: UPS Date: day / Week Day
                19: UPS Date: month / year
                20: Status bits
                21,22: In Frequency
                23: Checksum
                24: End of dataset (FE)
ba 61 4b ae 00 00 cc 14 00 0a 10 14 02 00 00
   1.401667	 00 01 00 17 90 09 60 60 95 fe
*/
static void apc_update_ups_info(const unsigned char *dataset)
{
	upsdebugx(3,MSG_ENTERFUNC, __func__);

	/* Set Model */
	switch(dataset[0])
	{
		case 0xB8:
                        ups.Model = PS350_CII;
			break;
		case 0xB9:
			ups.Model = PS800;
			break;
		case 0xBA:
			ups.Model = STAY1200;
			break;
		case 0xBB:
			ups.Model = PS2200;
                        break;
                case 0xBC:
			ups.Model = PS2200_22;
                        break;
		case 0xBD:
                        ups.Model = STAY700;
                        break;
		case 0xBE:
                        ups.Model = BZ1500;
                        break;
		default:
			ups.Model = UNKNOWN;	
	}	

	setStatus(dataset[20]);	
	ups.RelayMode = ((dataset[6] & 0x38) >> 3);

	/* Temperature - Simple assign - dont need a function */
	ups.Temperature = dataset[4];

	/* UPS Time/Date */
	ups.Localtime.Seconds = dataset[9];
	ups.Localtime.Minutes = dataset[10];
	ups.Localtime.Hours = dataset[11];
	ups.Localtime.Day = dataset[18] & 0x1F;
	ups.Localtime.WeekDay = ((dataset[18] & 0xE0) >> 5);
	ups.Localtime.Month = (dataset[19] & 0xF0) >> 4;
	ups.Localtime.Year = ((dataset[19] & 0x0F) + (N_YEARS * 16) + BASE_YEAR);

	/* UPS schedule turn on/off */
	ups.Schedule.HourOn = dataset[13];
	ups.Schedule.MinuteOn = dataset[14];
	ups.Schedule.HourOff = dataset[15];
	ups.Schedule.MinuteOff = dataset[16];

	/* Convert from UPS to internal representation */
	ups.Schedule.DaysOfWeek = 0x7F & rotateright(dataset[17] & 0x7F, 1+ups.Localtime.WeekDay, 7) ;

	setBatteryVoltage(dataset[3]);
	setAutonomy(dataset[3]);
	setBatteryPercentCharged();

	setInputVoltage(dataset[2]);
	setOutputCurrent(dataset[5]);
	setOutputVoltage(dataset[1]);

/*
	FILE *fpo;
        char tmpBuffer[1024];
        fpo = popen("sudo /usr/bin/sigrok-cli -O csv --channels P1 --samples 1 -d brymen-bm86x -l 0 | grep -v \\; | tr -d '\n'", "r");

        FILE *fp;

        fp = fopen("/tmp/nobreak.txt", "a+");

        while (fgets(tmpBuffer, sizeof(tmpBuffer)-1, fpo) != NULL) {
            upsdebugx(1,"Buffer: %s\n", tmpBuffer);
            if (! prefix(tmpBuffer, ";")){
                fprintf(fp, "Data: %d, a: %f, BatVolt: %f, X: %f, Y: %f, Curr: %f, Outvoltage: %f, Multimeter: %s\n", dataset[1], sqrt(dataset[1]/64.0), ups.Battery.Voltage, OutVoltage_X_Bat[ups.Model][ups.RelayMode], OutVoltage_Y_Bat[ups.Model][ups.RelayMode], ups.Out.Current, ups.Out.Voltage, tmpBuffer);
            }
        }
        fclose(fp);
        pclose(fpo);
*/

	setApparentPower();
	setRealPower(dataset[7] + dataset[8] * 256);
	setPowerFactor();
	setPowerLoad();
	setInputCurrent();
	setInputFrequency((long int)(dataset[21] + dataset[22] * 256));
	setOutputFrequency();
	setEfficiency();

	if (nut_debug_level > 0) printDBGInformation();

	upsdebugx(3,MSG_EXITFUNC, __func__);

}

static void apc_readserial(void)
{
	unsigned char c=0;
        int i=0;
        int j=0;
	int size=0; /* buffer size */
        unsigned char dataset[DS_SIZE];
        unsigned char checksum; /* data set check sum */

	upsdebugx(3,MSG_ENTERFUNC, __func__);

	upsdebugx(1, MSG_SEPREADSER);

        upsdebugx(1, "Flushing IO result: %i", ser_flush_io(upsfd));

	/* this is a loop do get the information. The first part syncs
           the communication and the second part gets the package. If
           the package is not ok, try again.
	*/
	do{
		j++;
		/* loop until find char 0xFE (end of dataset).
		   The idea is to get buffer ready for next record */
		i=0;
		while ((ser_get_char(upsfd, &c, 1, 0) == 1) && (c != 0xFE) && (i<1000))
		{
			i++;
			upsdebugx(3,"Discarding package: %02x", c);
		}
		upsdebugx(3,"Discarding package: %02x", c);

		size = ser_get_buf_len(upsfd, dataset, DS_SIZE, 3, 0);

		upsdebug_hex(1, MSG_BUFFER, (char *)dataset, size);
	} while ( (((dataset[0] & 0xF0) != 0xB0) || (dataset[24] != 0xFE)) && (j<RETRIES) );

	/* Check package size, it should be the DS_SIZE */
	if (size != DS_SIZE){
		upslogx(LOG_WARNING, MSG_ERRPKGSIZE, size, DS_SIZE);
		upsdebugx(3,MSG_EXITFUNC, __func__);
		return;
	}

	if (nut_debug_level > 1) printDBGDataSet(dataset);

	/* check begin and end of dataset */
	if ( ((dataset[0] & 0xF0) != 0xB0) || (dataset[24] != 0xFE) ){
		upslogx(LOG_WARNING,MSG_UNKDS,dataset[0] & 0xF0,dataset[24]);
		upsdebugx(3,MSG_EXITFUNC, __func__);
		return;
	} 

	/* Calculate checksum */
	checksum=0;

	for ( i=0 ; i < (DS_SIZE-2) ; ++i )
		checksum = checksum + dataset[i];

	checksum = checksum % 256;

	if (dataset[23] != checksum){
		upslogx(LOG_WARNING, MSG_CHECKSUM_UNMATCH, checksum, dataset[23]);
		upsdebugx(3,MSG_EXITFUNC, __func__);
		return;
	}

	upsdebugx(1, MSG_SEPARATOR);
	upsdebugx(1, " ");

	/* parse data set contents and store on local variables */
	apc_update_ups_info((unsigned char *)dataset);

	upsdebugx(3,MSG_EXITFUNC, __func__);
}


void upsdrv_initinfo(void)
{
	upsdebugx(3,MSG_ENTERFUNC, __func__);

	apc_readserial();

	if (ups.Model == UNKNOWN)
		fatalx(EXIT_FAILURE,  MSG_UNKNOWN_MODEL );

	printf(MSG_APCMODEL, ModelName[ups.Model], device_path);
	upslogx(LOG_INFO, MSG_APCMODEL, ModelName[ups.Model], device_path);

	/* Read paremeters */
	readParameters();

	setUPSClock();

	setRWDStates();
	setDStates();
	dstate_addcmd("shutdown.stayoff"); 
	dstate_addcmd("load.on"); 
	dstate_addcmd("load.off"); 
	dstate_addcmd("test.failure.start"); 
	dstate_addcmd("test.failure.stop"); 

	upsh.setvar = setvar;
	upsh.instcmd = instcmd; 
	upsdebugx(3,MSG_EXITFUNC, __func__);
}

void upsdrv_updateinfo(void)
{
	upsdebugx(3,MSG_ENTERFUNC, __func__);

	apc_readserial();
	if (ups.Model == UNKNOWN){
		printf("%s", "EOROROROROOR");
		return;
	}

	setUPSClock();

	setDStates();

        status_init();

        if (ups.Status.BatteryCharging)
                status_set("CHRG");

        if (ups.Status.CriticalBattery)
                status_set("LB");

        if (!ups.Status.Output)
                status_set("OFF");

        if (ups.Status.BatteryMode){
                status_set("OB");
                status_set("DISCHRG");
        }

        if (ups.Status.NetworkMode)
                status_set("OL");

        if (ups.Status.Overload)
                status_set("OVER");

        if (ups.Battery.PercentCharged > 90)
                status_set("HB");

	if (scheduleShutdownImminent()){
		upsdebugx(1,"+-+-+-+-+-+ IMMINENT SHUTDOWN! +-+-+-+-+-+");
		status_set("FSD");
	}

        status_commit();


	dstate_dataok();
	upsdebugx(3,MSG_EXITFUNC, __func__);

}

void upsdrv_shutdown(void)
{
	upsdebugx(3,MSG_ENTERFUNC, __func__);

	ser_send_buf(upsfd, CMD_SHUTDOWN, 4);
	upslogx(LOG_NOTICE, "Ups shutdown command sent");
	upsdebugx(3,MSG_EXITFUNC, __func__);
}

/* This routine exists only for debugging.
   The main idea is to detect other package formats.
   Theoretically, the UPS can receive a list of events,
   but this was not detect so far in the testings.
*/
static void detectPackages(void)
{
	unsigned char c;
	int i,x;
	int f=0;
	unsigned char tmp[100];

	upsdebugx(3,MSG_ENTERFUNC, __func__);
	for(x=0; x<= 100; x++) tmp[x]=0;
	upsdebugx(1, "Flushing IO result: %i", ser_flush_io(upsfd));
	/* loop until find char 0xFE (end of dataset).
	   The idea is to get buffer ready for next record */
	i=0;
	while ((ser_get_char(upsfd, &c, 1, 0) == 1) && (c != 0xFE) && (i<1000))
	{
		i++;
		upsdebugx(2,"Discarding package: %02x", c);
	}
	upsdebugx(2,"Discarding package: %02x", c);
	i=0;
	while (1){
		if (ser_get_char(upsfd, &c, 1, 0) != 1) continue;
		upsdebugx(3,"CHAR READ: %02x", c);
		if (f>0 && (c == 0xFE)){
			upsdebugx(3,"HERE1");
			f = 0;
			tmp[i]=c;
			upsdebug_hex(1, MSG_STDPKG, (char *)tmp, sizeof(tmp));
			for(x=0; x< 100; x++) tmp[x]=0;
			i=0;
		} else if ((f < 0) && ((c & 0xF0) == 0xB0)){
			upsdebugx(3,"HERE2");
			f =1;
			upsdebug_hex(1, MSG_NOTSTDPKG, (char *)tmp, sizeof(tmp));
			for(x=0; x< 100; x++) tmp[x]=0;
			i=1;
			tmp[0]=c;
		} else if (f == 0){
			upsdebugx(3,"HERE3");
			if ((c & 0xF0) == 0xB0){
				f = 1;
			} else {
				f = -1;
			}

			tmp[i]=c;
			i++;
		} else {
			upsdebugx(3,"HERE4");
			tmp[i]=c;
			i++;
		}
	}


}

static int setvar(const char *varname, const char *val)
{
	upsdebugx(3,MSG_ENTERFUNC, __func__);
	upsdebugx(2,MSG_SETVAR, varname, val);

	if (!strcasecmp(varname, "ups.mfr")) {
		dstate_setinfo("ups.mfr", "%s", val);
		return STAT_SET_HANDLED;
	}

	if (!strcasecmp(varname, "ups.model")) {
		dstate_setinfo("ups.model", "%s", val);
		return STAT_SET_HANDLED;
	}

	if (!strcasecmp(varname, "ups.power.nominal")) {
		dstate_setinfo("ups.power.nominal", "%d", atoi(val));
		return STAT_SET_HANDLED;
	}

	if (!strcasecmp(varname, "ups.realpower.nominal")) {
		dstate_setinfo("ups.realpower.nominal", "%f", atof(val));
		return STAT_SET_HANDLED;
	}

	if (!strcasecmp(varname, "input.voltage.nominal")) {
		dstate_setinfo("input.voltage.nominal", "%d", atoi(val));
		return STAT_SET_HANDLED;
	}
	
	if (!strcasecmp(varname, "input.current.nominal")) {
		dstate_setinfo("input.current.nominal", "%.1f", atof(val));
		return STAT_SET_HANDLED;
	}

	if (!strcasecmp(varname, "input.frequency.nominal")) {
		dstate_setinfo("input.frequency.nominal", "%d", atoi(val));
		return STAT_SET_HANDLED;
	}

	if (!strcasecmp(varname, "output.voltage.nominal")) {
		dstate_setinfo("output.voltage.nominal", "%f", atof(val));
		return STAT_SET_HANDLED;
	}

	if (!strcasecmp(varname, "battery.voltage.nominal")) {
		dstate_setinfo("battery.voltage.nominal", "%d", atoi(val));
		return STAT_SET_HANDLED;
	}

	if (!strcasecmp(varname, "battery.voltage.low")) {
		dstate_setinfo("battery.voltage.low", "%f", atof(val));
		BatteryVoltage_Min[ups.Model] = atof(val);
		return STAT_SET_HANDLED;
	}

	if (!strcasecmp(varname, "battery.voltage.high")) {
		dstate_setinfo("battery.voltage.high", "%f", atof(val));
		BatteryVoltage_Max[ups.Model] = atof(val);
		return STAT_SET_HANDLED;
	}

	if (!strcasecmp(varname, "battery.capacity")) {
		dstate_setinfo("battery.capacity", "%d", atoi(val));
		return STAT_SET_HANDLED;
	}

	if (!strcasecmp(varname, "battery.packs")) {
		dstate_setinfo("battery.packs", "%d", atoi(val));
		return STAT_SET_HANDLED;
	}

	if (!strcasecmp(varname, "battery.type")) {
		dstate_setinfo("battery.type", "%s", val);
		return STAT_SET_HANDLED;
	} 

	if (!strcasecmp(varname, "battery.charge.low")) {
		dstate_setinfo("battery.charge.low", "%f", atof(val));
		return STAT_SET_HANDLED;
	} 

	if (!strcasecmp(varname, "battery.charge.warning")) {
		dstate_setinfo("battery.charge.warning", "%d", atoi(val));
		return STAT_SET_HANDLED;
	}

	if (!strcasecmp(varname, "battery.runtime.low")) {
		dstate_setinfo("battery.runtime.low", "%d", atoi(val));
		return STAT_SET_HANDLED;
	}

	upslogx(LOG_NOTICE, "setvar: unknown variable [%s]", varname);
	return STAT_SET_UNKNOWN;
}

void upsdrv_help(void)
{
	upsdebugx(3,MSG_ENTERFUNC, __func__);

	printf("\n-------------------------------------\n");
	printf("APC Back-UPS BR Family Driver Options\n");
	printf("-------------------------------------\n\n");
	printf("This driver was tested with model BZ1200BR but should for other\n");
	printf("models of the same familty.\n");
	printf("This driver can schedule UPS power on/off. Check parameters listed above.\n\n");

	upsdebugx(3,MSG_EXITFUNC, __func__);
}

/* list flags and values that you want to receive via -x */
void upsdrv_makevartable(void)
{
	upsdebugx(3,MSG_ENTERFUNC, __func__);

	addvar(VAR_VALUE, "battery_extension", "battery extension capacity in Ah");
	addvar(VAR_VALUE, "daysofweek", "Days of the week to power up/down the UPS in 7 bit format 1111111 - Sun to Sat");
	addvar(VAR_VALUE, "time_on", "Time to power on UPS (HH:MM)");
	addvar(VAR_VALUE, "time_off", "Time to power off UPS (HH:MM)");
	addvar(VAR_VALUE, "clockupdate", "Number of seconds between each clock update");
	addvar(VAR_FLAG, "sniffer", "Enable input sniffer mode - do not use it - for dev purposes only");

	upsdebugx(3,MSG_EXITFUNC, __func__);

}

void upsdrv_initups(void)
{

	upsdebugx(3,MSG_ENTERFUNC, __func__);

	upsfd = ser_open(device_path); 
	ser_set_speed(upsfd, device_path, B9600); 

        ser_set_dtr(upsfd, 1);
        ser_set_rts(upsfd, 0);

	/* Initialize Ups record */
	ups.Model = UNKNOWN;
	ups.Temperature = 0;
	ups.Efficiency = 0;
	ups.RelayMode = 0;
	ups.In.Voltage = 0;
	ups.In.Current = 0;
	ups.In.Frequency = 0;
	ups.In.VoltageMax = 0;
	ups.In.VoltageMin = 999;
	ups.Out.Voltage = 0;
        ups.Out.Current = 0;
        ups.Out.Frequency = 0;
        ups.Out.ApparentPower = 0;
        ups.Out.RealPower = 0;
        ups.Out.PowerFactor = 0;
        ups.Out.Load = 0;
	ups.Battery.Voltage = 0;
        ups.Battery.Current = 0;
        ups.Battery.Extension = 0;
        ups.Battery.PercentCharged = 0;
	ups.Status.VIn220 = false;
	ups.Status.VOut220 = false;
	ups.Status.Output = false;
	ups.Status.CriticalBattery = false;
	ups.Status.NetworkMode = false;
	ups.Status.BatteryMode = false;
	ups.Status.BatteryCharging = false;
	ups.Status.Overload = false;
	ups.Status.Overheat = false;
	ups.Localtime.Day = 0;
	ups.Localtime.Year = 0;
	ups.Localtime.Month = 0;
	ups.Localtime.Hours = 0;
	ups.Localtime.Minutes = 0;
	ups.Localtime.Seconds = 0;
	ups.Localtime.WeekDay = 0;
	ups.Schedule.HourOn = 0;
	ups.Schedule.HourOff = 0;
	ups.Schedule.MinuteOn = 0;
	ups.Schedule.MinuteOff = 0;
	ups.Schedule.DaysOfWeek = 0;
	nextupdate = -1;

	upsdebugx(3,MSG_EXITFUNC, __func__);

}

void upsdrv_cleanup(void)
{
	upsdebugx(3,MSG_ENTERFUNC, __func__);
	ser_close(upsfd, device_path); 
	upsdebugx(3,MSG_EXITFUNC, __func__);
}


/************************************************
 * Miscellaneous functions                      *
 ************************************************/

/* Checks if it is near ups scheduled shutdown time.
   The UPS have a single time in the day to perform 
   shutdown, so it is only relevant to check the 
   current day and tomorrow (because it may be 
   scheduled for 0:00, for example).
*/
static bool_t scheduleShutdownImminent(void)
{
	time_t nowsec;
	time_t sdsec;
	struct tm now;
	struct tm shutdate;
	bool_t sdtoday;
	bool_t sdtomorrow;

	upsdebugx(3,MSG_ENTERFUNC, __func__);

	if (ups.Schedule.DaysOfWeek == 0){
		/* Nothing set, return */
		upsdebugx(3,MSG_EXITFUNC, __func__);
		return false;
	}

	sdtoday = ((ups.Schedule.DaysOfWeek & DoWFilter[ups.Localtime.WeekDay]) == 0)?false:true;
	sdtomorrow = ((ups.Schedule.DaysOfWeek * DoWFilter[(ups.Localtime.WeekDay+1)%7]) == 0)?false:true;
	upsdebugx(3, MSG_SSDIMM1, BoolName[sdtoday], BoolName[sdtomorrow]);

	if (!sdtoday && !sdtomorrow){
		/* Nothing set for today or tomorrow, return */
		upsdebugx(3,MSG_EXITFUNC, __func__);
		return false;
	}

	shutdate.tm_sec = 0;
	shutdate.tm_min = ups.Schedule.MinuteOff;
	shutdate.tm_hour = ups.Schedule.HourOff;
	shutdate.tm_wday = ups.Localtime.WeekDay;
	shutdate.tm_mday = ups.Localtime.Day;
	shutdate.tm_mon = ups.Localtime.Month-1;
	shutdate.tm_year = ups.Localtime.Year-1900;

	now.tm_sec = ups.Localtime.Seconds;
	now.tm_min = ups.Localtime.Minutes;
	now.tm_hour = ups.Localtime.Hours;
	now.tm_wday = ups.Localtime.WeekDay;
	now.tm_mday = ups.Localtime.Day;
	now.tm_mon = ups.Localtime.Month-1;
	now.tm_year = ups.Localtime.Year-1900;

	/* get time in seconds */
	sdsec = mktime(&shutdate);
	nowsec = mktime(&now);

	upsdebugx(3, MSG_SSDIMM2, (int)sdsec, (int)nowsec);

	if (sdtoday & (sdsec > nowsec)){
		upsdebugx(3,MSG_EXITFUNC, __func__);
		return (difftime(sdsec,nowsec) <= SCHEDULED_FSD_SECONDS)?true:false;
	}

	sdsec += 86400;
	if (sdtomorrow){
		upsdebugx(3,MSG_EXITFUNC, __func__);
		return (difftime(sdsec,nowsec) <= SCHEDULED_FSD_SECONDS)?true:false;
	}

	upsdebugx(3,MSG_EXITFUNC, __func__);
	return false;
		

}

static void printDBGDataSet(const unsigned char *dataset)
{
	upsdebugx(3,MSG_ENTERFUNC, __func__);

	upsdebugx(2,MSG_SEPDSPACK);

	upsdebugx(2,MSG_DS);
	upsdebugx(2,MSG_DSHEADER,dataset[0]);
	upsdebugx(2,MSG_DSOUTVOL,dataset[1]);
	upsdebugx(2,MSG_DSINVOL,dataset[2]);
	upsdebugx(2,MSG_DSBATVOL,dataset[3]);
	upsdebugx(2,MSG_DSTEMP,dataset[4]);
	upsdebugx(2,MSG_DSOUTCUR,dataset[5]);
	upsdebugx(2,MSG_DSRELAY,dataset[6]);
	upsdebugx(2,MSG_DSREALPWR1,dataset[7]);
	upsdebugx(2,MSG_DSREALPWR2,dataset[8]);
	upsdebugx(2,MSG_DSTIMESEC,dataset[9]);
	upsdebugx(2,MSG_DSTIMEMIN,dataset[10]);
	upsdebugx(2,MSG_DSTIMEHOUR,dataset[11]);
	upsdebugx(2,MSG_DSNOTUSED,dataset[12]);
	upsdebugx(2,MSG_DSHON,dataset[13]);
	upsdebugx(2,MSG_DSMON,dataset[14]);
	upsdebugx(2,MSG_DSHOFF,dataset[15]);
	upsdebugx(2,MSG_DSMOFF,dataset[16]);
	upsdebugx(2,MSG_DSWEEKDAYS,dataset[17]);
	upsdebugx(2,MSG_DSDATE1,dataset[18]);
	upsdebugx(2,MSG_DSDATE2,dataset[19]);
	upsdebugx(2,MSG_DSSTATUS,dataset[20]);
	upsdebugx(2,MSG_DSFREQ1,dataset[21]);
	upsdebugx(2,MSG_DSFREQ2,dataset[22]);
	upsdebugx(2,MSG_DSCHKSUM,dataset[23]);
	upsdebugx(2,MSG_DSEODS,dataset[24]);

	upsdebugx(2,MSG_SEPARATOR);

	upsdebugx(3,MSG_EXITFUNC, __func__);
}

static unsigned char rotateleft(unsigned char value, unsigned char nshifts, unsigned char nbits)
{

	nshifts %= nbits;
	if (nshifts == 0){
		return value;
	} else {
		return (value << nshifts) | (value >> (sizeof(value) * nbits - nshifts));
	}
}

static unsigned char rotateright(unsigned char value, unsigned char nshifts, unsigned char nbits)
{       
	nshifts %= nbits;
	upsdebugx(1, "VALUE: %d, shifts: %d, Bits: %d, Result: %d", value,nshifts,nbits,(value >> nshifts) | (value << (sizeof(value) * nbits - nshifts)));
	if (nshifts == 0){
		return value;
	} else {
		return (value >> nshifts) | (value << (sizeof(value) * nbits - nshifts));
	}
}

static unsigned char BinStr2Byte(char *str)
{
	return (unsigned char) strtol(str, NULL, 2);
}

const char *byte_to_binary(int x)
{
	static char b[10];
	int z;

	b[0] = '\0';

	for (z = 128; z > 0; z >>= 1)
		snprintfcat(b,sizeof(b), ((x & z) == z) ? "1" : "0");

	return b;
}

/* print information read from serial port */
static void printDBGInformation(void)
{
	upsdebugx(3,MSG_ENTERFUNC, __func__);

	upsdebugx(1,MSG_SEPINFO);

	upsdebugx(1,MSG_UPSINFO);
	upsdebugx(1,MSG_RELAYMODE,ups.RelayMode);
	upsdebugx(1,MSG_EFFICIENCY,ups.Efficiency);
	upsdebugx(1,MSG_TEMPERATURE,ups.Temperature);
	upsdebugx(1,MSG_DATETIME, ups.Localtime.Day, ups.Localtime.Month, ups.Localtime.Year, ups.Localtime.Hours, ups.Localtime.Minutes, ups.Localtime.Seconds, ups.Localtime.WeekDay);
	upsdebugx(1,MSG_SCHEDULE, ups.Schedule.HourOn, ups.Schedule.MinuteOn, ups.Schedule.HourOff, ups.Schedule.MinuteOff, ups.Schedule.DaysOfWeek); 
	upsdebugx(1," ");
        upsdebugx(1,MSG_UPSINPUT);
	upsdebugx(1,MSG_VOLTAGEMAX, ups.In.VoltageMax);
	upsdebugx(1,MSG_VOLTAGEMIN, ups.In.VoltageMin);
	upsdebugx(1,MSG_VOLTAGE, ups.In.Voltage);
	upsdebugx(1,MSG_CURRENT, ups.In.Current);
	upsdebugx(1,MSG_FREQUENCY, ups.In.Frequency);
	upsdebugx(1," ");
	upsdebugx(1,MSG_UPSOUTPUT);
	upsdebugx(1,MSG_VOLTAGE, ups.Out.Voltage);
	upsdebugx(1,MSG_CURRENT, ups.Out.Current);
	upsdebugx(1,MSG_FREQUENCY, ups.Out.Frequency);
	upsdebugx(1,MSG_APP_POWER, ups.Out.ApparentPower);
	upsdebugx(1,MSG_REAL_POWER, ups.Out.RealPower);
	upsdebugx(1,MSG_PWR_FACTOR, ups.Out.PowerFactor);
	upsdebugx(1,MSG_LOAD, ups.Out.Load);
	upsdebugx(1," ");
	upsdebugx(1,MSG_UPSBATTERY);
	upsdebugx(1,MSG_VOLTAGE, ups.Battery.Voltage);
	upsdebugx(1,MSG_CURRENT, ups.Battery.Current);
	upsdebugx(1,MSG_PERC_CHARGED, ups.Battery.PercentCharged);
	upsdebugx(1,MSG_EXTENSION, ups.Battery.Extension);
	upsdebugx(1,MSG_AUTONOMY, ups.Battery.Autonomy);
	upsdebugx(1," ");
	upsdebugx(1,MSG_UPSSTATUS);
	upsdebugx(1,MSG_NETMODE,BoolName[ups.Status.NetworkMode]);
	upsdebugx(1,MSG_VIN220,BoolName[ups.Status.VIn220]);
	upsdebugx(1,MSG_OVERLOAD,BoolName[ups.Status.Overload]);
	upsdebugx(1,MSG_VOUT220,BoolName[ups.Status.VOut220]);
	upsdebugx(1,MSG_OUTPUT,BoolName[ups.Status.Output]);
	upsdebugx(1,MSG_BATMODE,BoolName[ups.Status.BatteryMode]);
	upsdebugx(1,MSG_CRITBAT,BoolName[ups.Status.CriticalBattery]);
	upsdebugx(1,MSG_BATCHARGE,BoolName[ups.Status.BatteryCharging]);
	upsdebugx(1,MSG_OVERHEAT,BoolName[ups.Status.Overheat]);

	upsdebugx(1,MSG_SEPARATOR);

	upsdebugx(3,MSG_EXITFUNC, __func__);
}

/* Set dstate writeable variables. Basically they are set only 
   once at startup then left alone. If we change it every cycle
   it will overwride any user input */
static void setRWDStates(void)
{
	upsdebugx(3,MSG_ENTERFUNC, __func__);

	dstate_setinfo("ups.mfr", "%s", MANUFACTURER);
	dstate_setflags("ups.mfr", ST_FLAG_STRING | ST_FLAG_RW);
	dstate_setaux("ups.mfr", 32);

	dstate_setinfo("ups.model", "%s", ModelName[ups.Model]);
	dstate_setflags("ups.model", ST_FLAG_STRING | ST_FLAG_RW);
	dstate_setaux("ups.model", 32);

	/* Nominal value of apparent power (Volt-Amps) */
	dstate_setinfo("ups.power.nominal", "%d", NominalApparentPower[ups.Model]);
	dstate_setflags("ups.power.nominal", ST_FLAG_STRING | ST_FLAG_RW);
	dstate_setaux("ups.power.nominal", 5);

	/* Nominal value of real power (Watts) */
	dstate_setinfo("ups.realpower.nominal", "%d", NominalRealPower[ups.Model]);
	dstate_setflags("ups.realpower.nominal", ST_FLAG_STRING | ST_FLAG_RW);
	dstate_setaux("ups.realpower.nominal", 5);

	dstate_setinfo("input.voltage.nominal", "%f", InVoltageNominal[ups.Status.VIn220][ups.Model]);
	dstate_setflags("input.voltage.nominal", ST_FLAG_STRING | ST_FLAG_RW);
	dstate_setaux("input.voltage.nominal", 5);

	dstate_setinfo("input.current.nominal", "%.1f", InCurrentNominal[ups.Status.VIn220][ups.Model]);
	dstate_setflags("input.current.nominal", ST_FLAG_STRING | ST_FLAG_RW);
	dstate_setaux("input.current.nominal", 5);

	dstate_setinfo("input.frequency.nominal", "%d", NominalFrequency[ups.Model]);
	dstate_setflags("input.frequency.nominal", ST_FLAG_STRING | ST_FLAG_RW);
	dstate_setaux("input.frequency.nominal", 3);

	dstate_setinfo("output.voltage.nominal", "%f", OutVoltageNominal[ups.Status.VOut220][ups.Model]);
	dstate_setflags("output.voltage.nominal", ST_FLAG_STRING | ST_FLAG_RW);
	dstate_setaux("output.voltage.nominal", 3);

	dstate_setinfo("battery.voltage.nominal", "%d", BatteryVoltageNominal[ups.Model]);
	dstate_setflags("battery.voltage.nominal", ST_FLAG_STRING | ST_FLAG_RW);
	dstate_setaux("battery.voltage.nominal", 3);

	dstate_setinfo("battery.voltage.low", "%f", BatteryVoltage_Min[ups.Model]);
	dstate_setflags("battery.voltage.low", ST_FLAG_STRING | ST_FLAG_RW);
	dstate_setaux("battery.voltage.low", 6);

	dstate_setinfo("battery.voltage.high", "%f", BatteryVoltage_Max[ups.Model]);
	dstate_setflags("battery.voltage.high", ST_FLAG_STRING | ST_FLAG_RW);
	dstate_setaux("battery.voltage.high", 6);

	dstate_setinfo("battery.capacity", "%d", BatteryCapacity[ups.Model]);
	dstate_setflags("battery.capacity", ST_FLAG_STRING | ST_FLAG_RW);
	dstate_setaux("battery.capacity", 4);

	dstate_setinfo("battery.packs", "%d", BatteryPacks[ups.Model]);
	dstate_setflags("battery.packs", ST_FLAG_STRING | ST_FLAG_RW);
	dstate_setaux("battery.packs", 3);

	dstate_setinfo("battery.type", "%s", BatteryType[ups.Model]);
	dstate_setflags("battery.type", ST_FLAG_STRING | ST_FLAG_RW);
	dstate_setaux("battery.type", 32);

	/* Remaining battery level when UPS switches to LB (percent) */
	dstate_setinfo("battery.charge.low", "%d", LOW_BATT_CHARGE);
	dstate_setflags("battery.charge.low", ST_FLAG_STRING | ST_FLAG_RW);
	dstate_setaux("battery.charge.low", 2);

	dstate_setinfo("battery.charge.warning", "%d", WARN_BATT_CHARGE);
	dstate_setflags("battery.charge.warning", ST_FLAG_STRING | ST_FLAG_RW);
	dstate_setaux("battery.charge.warning", 2);

	dstate_setinfo("battery.runtime.low", "%d", BATT_RUNTIME_LOW);
	dstate_setflags("battery.runtime.low", ST_FLAG_STRING | ST_FLAG_RW);
	dstate_setaux("battery.runtime.low", 5);

}


/* Set dstate based on internal variables */
static void setDStates(void)
{
	upsdebugx(3,MSG_ENTERFUNC, __func__);

	dstate_setinfo("ups.date", "%02d-%02d-%04d", ups.Localtime.Day, ups.Localtime.Month, ups.Localtime.Year);
	dstate_setinfo("ups.time", "%02d:%02d:%02d", ups.Localtime.Hours, ups.Localtime.Minutes, ups.Localtime.Seconds);
	dstate_setinfo("ups.temperature", "%f", ups.Temperature);
	dstate_setinfo("ups.load", "%.1f", ups.Out.Load);
	dstate_setinfo("ups.load.high", "100");
	dstate_setinfo("ups.efficiency", "%d", ups.Efficiency);

	/* Current value of apparent power (Volt-Amps) */
	dstate_setinfo("ups.power", "%d", (int)ups.Out.ApparentPower);

	/* Current value of real power (Watts) */
	dstate_setinfo("ups.realpower", "%d", (int)ups.Out.RealPower);
	
	dstate_setinfo("input.voltage", "%.1f", ups.In.Voltage);
	
	/* Maximum incoming voltage seen */
	dstate_setinfo("input.voltage.maximum", "%.1f", ups.In.VoltageMax);
	/* Minimum incoming voltage seen */
	dstate_setinfo("input.voltage.minimum", "%.1f", ups.In.VoltageMin);
	dstate_setinfo("input.current", "%.2f", ups.In.Current);
	dstate_setinfo("input.frequency", "%d", (int)ups.In.Frequency);


	dstate_setinfo("output.voltage", "%.1f", ups.Out.Voltage);
	dstate_setinfo("output.frequency", "%d", (int)ups.Out.Frequency);
	dstate_setinfo("output.current", "%.2f", ups.Out.Current);

	/* Battery charge (percent) */
	dstate_setinfo("battery.charge", "%d", ups.Battery.PercentCharged);

	dstate_setinfo("battery.voltage", "%.1f", ups.Battery.Voltage);
	dstate_setinfo("battery.current", "%.2f", ups.Battery.Current);
	dstate_setinfo("battery.runtime", "%d", ups.Battery.Autonomy);

	upsdebugx(3,MSG_EXITFUNC, __func__);

}


/************************************************
 * All the functions below are related to parse *
 * data sent from No-break                      *
 ************************************************/

/**********************
 * Set Status         *
 **********************/
static void setStatus(int data)
{
	upsdebugx(3,MSG_ENTERFUNC, __func__);

	/* is output 220v? - Always false for this line of ups */
        ups.Status.VOut220=false;

	/* Network Mode */
        ups.Status.NetworkMode=((data & 0x20) != 0x20)?true:false;

	/* Battery Charging? */
        ups.Status.BatteryCharging=(((data & 0x2) == 2) && ups.Status.NetworkMode)?true:false;

	/* Battery in critical level */
	ups.Status.CriticalBattery=((data & 0x4) == 4)?true:false;

	/* Is output on? */
        ups.Status.Output=((data & 0x8) == 8)?true:false;

	/* Battery Mode */
        ups.Status.BatteryMode=(ups.Status.Output && !ups.Status.NetworkMode)?true:false;

	/* Overheat */
        ups.Status.Overheat=((data & 0x10) == 0x10)?true:false;

	/* Check if input voltage is 220volts */
	ups.Status.VIn220=((data & 0x40) == 0x40)?true:false;

	/* Overload */
	ups.Status.Overload=((data & 0x80) == 0x80)?true:false;

	upsdebugx(3,MSG_EXITFUNC, __func__);
}

/**********************
 * Set Output Voltage *
 **********************/
static void setOutputVoltage(int data)
{
	float a; /* Auxiliary variable */
	upsdebugx(3,MSG_ENTERFUNC, __func__);

	if (!ups.Status.Output){
		ups.Out.Voltage=0.0;

	} else if (ups.Status.NetworkMode) {
		ups.Out.Voltage=OutVoltage_X_Net[ups.Model][ups.RelayMode] * data + OutVoltage_Y_Net[ups.Model][ups.RelayMode];

		/* Need to adjust if model is PS350_CII */
		if ((ups.Model == PS350_CII) && (ups.Out.Current > 2.0))
			ups.Out.Voltage -= 5.0;
	} else {

        	switch (ups.Model)
        	{

			case PS800:
			case PS350_CII:
			case STAY1200:
			case STAY700:
			case BZ1500:
				a = data * 2;
				a /= 128.0;
				a = sqrt(a);
				ups.Out.Voltage=ups.Battery.Voltage * a * OutVoltage_X_Bat[ups.Model][ups.RelayMode] - ( ups.Out.Current * OutVoltage_Y_Bat[ups.Model][ups.RelayMode] );

				break;

			case PS2200:
				ups.Out.Voltage=0.0849 * pow(data,2.0) - 10.387 * data + 424.83999999999997;
				if (ups.Out.Voltage > 115.0)
					ups.Out.Voltage=116.0;
				

				break;

			case PS2200_22:
				ups.Out.Voltage=OutVoltage_X_Bat[ups.Model][ups.RelayMode] * data + OutVoltage_Y_Bat[ups.Model][ups.RelayMode];
				break;

			default:
				ups.Out.Voltage=0.0;
		}
	}
	upsdebugx(3,MSG_EXITFUNC, __func__);
}



/**********************
 * Set Input Voltage  *
 **********************/
static void setInputVoltage(int data)
{
	upsdebugx(3,MSG_ENTERFUNC, __func__);

	switch (ups.Model)
	{

		case PS800:
		case PS350_CII:
		case STAY700:
		case PS2200_22:
			if (data == 0){
				ups.In.Voltage=0;
			} else {
				ups.In.Voltage = InVoltage_X[ups.Model] * data + InVoltage_Y[ups.Model];
			}

			break;
		
		case STAY1200:
			if (ups.Status.VOut220){
				ups.In.Voltage = InVoltage_X[ups.Model] * data + InVoltage_Y[ups.Model];
			} else {
				ups.In.Voltage = InVoltage_X[ups.Model] * data + InVoltage_Y[ups.Model] - 3.0;
			}			
			break;

		case PS2200:
			if (ups.Status.VOut220){
                                ups.In.Voltage = InVoltage_X[ups.Model] * data + InVoltage_Y[ups.Model]; 
                        } else {
                                ups.In.Voltage = InVoltage_X_115_PS2200 * data + InVoltage_Y_115_PS2200;
                        }
                        break;	

		case BZ1500:
			ups.In.Voltage = -1.0E-004 * data * data + (InVoltage_X[ups.Model] * data + InVoltage_Y[ups.Model]);
			break;

		default:
			ups.In.Voltage=0.0;
	}

	/* Set Max and Min Voltage Seen */
	if (ups.In.Voltage > ups.In.VoltageMax)
		ups.In.VoltageMax = ups.In.Voltage;

	if (ups.In.Voltage < ups.In.VoltageMin)
		ups.In.VoltageMin = ups.In.Voltage;

	upsdebugx(3,MSG_EXITFUNC, __func__);
	
}

/************************
 * Set Battery Voltage  *
 ************************/
static void setBatteryVoltage(int data)
{
	upsdebugx(3,MSG_ENTERFUNC, __func__);
	ups.Battery.Voltage = BatteryVoltage_X[ups.Model] * data + BatteryVoltage_Y[ups.Model];
	upsdebugx(3,MSG_EXITFUNC, __func__);
}

/************************
 * Set Output Current   *
 ************************/
static void setOutputCurrent(int data)
{
	upsdebugx(3,MSG_ENTERFUNC, __func__);

	if (ups.Status.NetworkMode){
		ups.Out.Current = OutCurrent_X_Net[ups.Model] * data + OutCurrent_Y_Net[ups.Model];
	} else if (ups.Status.Output) {
		ups.Out.Current = OutCurrent_X_Bat[ups.Model] * data + OutCurrent_Y_Bat[ups.Model];
	} else {
		ups.Out.Current = 0;
	}

	upsdebugx(3,MSG_EXITFUNC, __func__);
}

/************************
 * Set Input Current    *
 ************************/
static void setInputCurrent(void)
{
	upsdebugx(3,MSG_ENTERFUNC, __func__);

	if (!ups.Status.NetworkMode){
		ups.In.Current = 0.0;
	} else {
		ups.In.Current = (1.1 * (ups.Out.ApparentPower / ups.In.Voltage));
	}

	upsdebugx(3,MSG_EXITFUNC, __func__);
}

/************************
 * Set Apparent Power   *
 ************************/
static void setApparentPower(void)
{
	upsdebugx(3,MSG_ENTERFUNC, __func__);
	ups.Out.ApparentPower = ups.Out.Voltage * ups.Out.Current;
	upsdebugx(3,MSG_EXITFUNC, __func__);
}

/************************
 * Set Efficiency       *
 ************************/
static void setEfficiency(void)
{
        upsdebugx(3,MSG_ENTERFUNC, __func__);
	if (ups.In.Current == 0){
		ups.Efficiency = 0;
	} else {
		ups.Efficiency = (int) (100 * (ups.Out.Current /  ups.In.Current));
	}

	if (ups.Efficiency > 100)
		ups.Efficiency = 100;

	upsdebugx(3,MSG_EXITFUNC, __func__);
}


/************************
 * Set Power Load       *
 ************************/
static void setPowerLoad(void)
{
	bool_t overload = false;
	upsdebugx(3,MSG_ENTERFUNC, __func__);
	
	
	if (ups.Out.RealPower > Overload_Threshold[ups.Model])
		overload = true;

	if (overload != ups.Status.Overload){
		upslogx(LOG_NOTICE,MSG_OVERLOAD_ODD, BoolName[overload], BoolName[ups.Status.Overload]);
	}
	

	ups.Status.Overload = overload;

	ups.Out.Load = (100 * (ups.Out.RealPower /  Overload_Threshold[ups.Model]));

	upsdebugx(3,MSG_EXITFUNC, __func__);

}

/************************
 * Set Real Power       *
 ************************/
static void setRealPower(int data)
{
	double pLin;
	double pVa;
	double pVa2;
	double difLin;
	double difNLin;
	double difNLin2;
	double tmp;
	upsdebugx(3,MSG_ENTERFUNC, __func__);
	
	switch (ups.Model)
	{
		case PS800:
		case PS350_CII:
		case STAY700:
			if (ups.Status.Output){

				pLin = UtilPower_Detect_C1_X[ups.Model][ups.RelayMode] * data + UtilPower_Detect_C1_Y[ups.Model][ups.RelayMode];
				pVa = UtilPower_Detect_C2_X[ups.Model][ups.RelayMode] * data + UtilPower_Detect_C2_Y[ups.Model][ups.RelayMode];
				pVa2 = UtilPower_Detect_C3_X[ups.Model][ups.RelayMode] * data + UtilPower_Detect_C3_Y[ups.Model][ups.RelayMode];

				difLin = fabs(pLin - ups.Out.ApparentPower);
				difNLin = fabs(pVa - ups.Out.ApparentPower);
				difNLin2 = fabs(pVa2 - ups.Out.ApparentPower);

				if ((difLin < difNLin) && (difLin < difNLin2)) {
					ups.Out.RealPower = UtilPower_C1_X[ups.Model][ups.RelayMode] * data + UtilPower_C1_Y[ups.Model][ups.RelayMode];
				} else if (difNLin < difNLin2) {
					ups.Out.RealPower = UtilPower_C2_X[ups.Model][ups.RelayMode] * data + UtilPower_C2_Y[ups.Model][ups.RelayMode];

				} else {
					ups.Out.RealPower = UtilPower_C2_X[ups.Model][ups.RelayMode] * data + UtilPower_C2_Y[ups.Model][ups.RelayMode];
				}
			} else {
				ups.Out.RealPower = 0;
			}

			if (ups.Out.ApparentPower < ups.Out.RealPower){
				tmp = ups.Out.ApparentPower;
				ups.Out.ApparentPower = ups.Out.RealPower;
				ups.Out.RealPower = tmp;
			}
			break;

		case STAY1200:

			if (ups.Out.Current < 0.7){
				ups.Out.RealPower = ups.Out.ApparentPower;
				break;
			} 
			
			pVa = 5.968 * ups.Out.ApparentPower - 284.36000000000001;
			pVa2 = 7.149 * ups.Out.ApparentPower - 567.17999999999995;
		
			if (fabs(pVa - data) < fabs(pVa2 - data)){
				/* pLin */
				ups.Out.RealPower = 0.1664 * data + 49.182000000000002;
			} else {
				/* pRe */
				ups.Out.RealPower = 0.1519 * data + 32.643999999999998;	
			}

			if (ups.Out.ApparentPower < ups.Out.RealPower){
				tmp = ups.Out.ApparentPower;
				ups.Out.ApparentPower = ups.Out.RealPower;
				ups.Out.RealPower = tmp;
			}
			break;

		case PS2200:
			if ((!ups.Status.Output) || (ups.Out.ApparentPower == 0.0)){
				ups.Out.RealPower = 0;
				break;
			} 

			ups.Out.RealPower = UtilPower_Detect_C1_X[PS2200][ups.RelayMode] * data + UtilPower_Detect_C1_Y[PS2200][ups.RelayMode];

			if (ups.Out.RealPower > ups.Out.ApparentPower)
				ups.Out.ApparentPower = ups.Out.RealPower;

			break;

		case PS2200_22:

			if ((!ups.Status.Output) || (ups.Out.ApparentPower == 0.0)){
				ups.Out.RealPower = 0;
				break;
			} 

			if (ups.Status.NetworkMode){

				ups.Out.RealPower = UtilPower_Detect_C1_X[PS2200][ups.RelayMode] * data + UtilPower_Detect_C1_Y[PS2200][ups.RelayMode];
			} else {
				ups.Out.RealPower = 0.1994 * data + 78.198999999999998;
			}

			if (ups.Out.RealPower > ups.Out.ApparentPower)
				ups.Out.ApparentPower = ups.Out.RealPower;

			break;

		case BZ1500:

			if (ups.Out.Current < 0.7){
				ups.Out.RealPower = ups.Out.ApparentPower;
				break;
			}

			ups.Out.RealPower = (0.1127 * data + 50.030999999999999);

			if (ups.Out.ApparentPower < ups.Out.RealPower){
				tmp = ups.Out.ApparentPower;
				ups.Out.ApparentPower = ups.Out.RealPower;
				ups.Out.RealPower = tmp;
			}
			break;

		default:

			ups.Out.RealPower = 0;

	}

	upsdebugx(3,MSG_EXITFUNC, __func__);
}

/***********************
 * Set Input Frequency *
 ***********************/
static void setInputFrequency(long int data)
{
	double Vin;
	double factor;
	upsdebugx(3,MSG_ENTERFUNC, __func__);

	if (ups.Model == STAY700){
		Vin = 30.0;
		factor = 0.371;
	} else {
		Vin = 0.0;
		factor = 0.37;
	}

	if (ups.In.Voltage > Vin){
		ups.In.Frequency = factor * (257 - (data >> 8));
	} else {
		ups.In.Frequency = 0.0;
	}

	upsdebugx(3,MSG_EXITFUNC, __func__);
}

/************************
 * Set Output Frequency *
 ************************/
static void setOutputFrequency(void)
{
	upsdebugx(3,MSG_ENTERFUNC, __func__);

	if (!ups.Status.Output){
		ups.Out.Frequency = 0.0;
	} else if (ups.Status.NetworkMode){
		ups.Out.Frequency = ups.In.Frequency;
	} else if (ups.Status.BatteryMode){
		ups.Out.Frequency = 60.0;
	} else {
		ups.Out.Frequency = 0.0;
	}
	upsdebugx(3,MSG_EXITFUNC, __func__);
}

/*******************************
 * Set Battery Percent Charged *
 *******************************/
static void setBatteryPercentCharged(void)
{
	upsdebugx(3,MSG_ENTERFUNC, __func__);

	if (ups.Status.BatteryCharging){
		ups.Battery.PercentCharged = (int)(100.0 * ((ups.Battery.Voltage - BatteryVoltage_Min[ups.Model]) / (BatteryVoltage_Max[ups.Model] - BatteryVoltage_Min[ups.Model])));
	} else {
		ups.Battery.PercentCharged = (int)(100.0 * ((ups.Battery.Voltage - BatteryVoltage_Min[ups.Model]) / (BatteryVoltage_Fluct[ups.Model] - BatteryVoltage_Min[ups.Model])));
	}

	if (ups.Battery.PercentCharged > 100){
		ups.Battery.PercentCharged = 100;
	} else if (ups.Battery.PercentCharged < 0) {
		ups.Battery.PercentCharged = 0;
	}

	upsdebugx(3,MSG_EXITFUNC, __func__);

}

/********************
 * Set Power Factor *
 ********************/
static void setPowerFactor(void)
{
	upsdebugx(3,MSG_ENTERFUNC, __func__);

	if (ups.Out.ApparentPower <= 0.0){
		ups.Out.PowerFactor = 0.0;
	} else {
		ups.Out.PowerFactor = ups.Out.RealPower / ups.Out.ApparentPower;
	
		if (ups.Out.PowerFactor > 1)
			ups.Out.PowerFactor = 1.0;
	}

	upsdebugx(3,MSG_EXITFUNC, __func__);

}

/****************
 * Set Autonomy *
 ****************/
static void setAutonomy(int data)
{
	upsdebugx(3,MSG_ENTERFUNC, __func__);

	switch (ups.Model)
	{
		case PS800:
			setAutonomyPS800(data);
			break;
		case PS350_CII:
			setAutonomyPS350_CII(data);
			break;
		case STAY700:
			setAutonomySTAY700(data);
			break;
		case PS2200:
		case PS2200_22:
			setAutonomyPS2200(data);
			break;
		case STAY1200:
		case BZ1500:
			setAutonomySTAY1200(data);
			break;
		default:
			ups.Battery.Autonomy = 0;
	}

	upsdebugx(3,MSG_EXITFUNC, __func__);

}

/**********************
 * Set Autonomy PS800 *
 **********************/
static void setAutonomyPS800(int data)
{
	int i;
        upsdebugx(3,MSG_ENTERFUNC, __func__);

	if (ups.Out.RealPower < 100.0){
		i = data - 151;
		if (i<1) {
			i = 1;
		} else if (i > (int)sizeof(Autonomy)){
			i = sizeof(Autonomy);
		}
		ups.Battery.Autonomy = Autonomy[sizeof(Autonomy)-i];
	} else if (ups.Out.RealPower < 180.0){
		if (ups.Battery.Voltage >= 24.1){
			ups.Battery.Autonomy = 15;
		} else  if (ups.Battery.Voltage >= 24.09){
			ups.Battery.Autonomy = 14;
		} else  if (ups.Battery.Voltage >= 24.07){
			ups.Battery.Autonomy = 13;
		} else  if (ups.Battery.Voltage >= 24.040001){
			ups.Battery.Autonomy = 12;
		} else  if (ups.Battery.Voltage >= 22.959999){
			ups.Battery.Autonomy = 11;
		} else  if (ups.Battery.Voltage >= 22.91){
			ups.Battery.Autonomy = 10;
		} else  if (ups.Battery.Voltage >= 22.85){
			ups.Battery.Autonomy = 9;
		} else  if (ups.Battery.Voltage >= 22.780001){
			ups.Battery.Autonomy = 8;
		} else  if (ups.Battery.Voltage >= 22.700001){
			ups.Battery.Autonomy = 7;
		} else  if (ups.Battery.Voltage >= 22.610001){
			ups.Battery.Autonomy = 6;
		} else  if (ups.Battery.Voltage >= 22.52){
			ups.Battery.Autonomy = 5;
		} else  if (ups.Battery.Voltage >= 22.4){
			ups.Battery.Autonomy = 4;
		} else  if (ups.Battery.Voltage >= 22.27){
			ups.Battery.Autonomy = 3;
		} else  if (ups.Battery.Voltage >= 22.0){
			ups.Battery.Autonomy = 2;
		} else  if (ups.Battery.Voltage >= 20.73){
			ups.Battery.Autonomy = 1;
		}
	} else if (ups.Out.RealPower < 200.0){
		if (ups.Battery.Voltage >= 22.940001){
			ups.Battery.Autonomy = 10;
		} else  if (ups.Battery.Voltage >= 22.9){
			ups.Battery.Autonomy = 7;
		} else  if (ups.Battery.Voltage >= 22.790001){
			ups.Battery.Autonomy = 6;
		} else  if (ups.Battery.Voltage >= 22.690001){
			ups.Battery.Autonomy = 5;
		} else  if (ups.Battery.Voltage >= 22.52){
			ups.Battery.Autonomy = 4;
		} else  if (ups.Battery.Voltage >= 22.389999){
			ups.Battery.Autonomy = 3;
		} else  if (ups.Battery.Voltage >= 22.17){
			ups.Battery.Autonomy = 2;
		} else  if (ups.Battery.Voltage >= 20.870001){
			ups.Battery.Autonomy = 1;
		}
	} else if (ups.Out.RealPower < 250.0){
		if (ups.Battery.Voltage >= 22.809999){
			ups.Battery.Autonomy = 7;
		} else  if (ups.Battery.Voltage >= 22.75){
			ups.Battery.Autonomy = 6;
		} else  if (ups.Battery.Voltage >= 22.67){
			ups.Battery.Autonomy = 5;
		} else  if (ups.Battery.Voltage >= 22.52){
			ups.Battery.Autonomy = 4;
		} else  if (ups.Battery.Voltage >= 22.34){
			ups.Battery.Autonomy = 3;
		} else  if (ups.Battery.Voltage >= 22.110001){
			ups.Battery.Autonomy = 2;
		} else  if (ups.Battery.Voltage >= 20.85){
			ups.Battery.Autonomy = 1;
		}
	} else if (ups.Out.RealPower < 300.0){ 
		if (data >= 154){
			ups.Battery.Autonomy = 3;
		} else if (data >= 154){
			ups.Battery.Autonomy = 2;
		} else if (data >= 149){
			ups.Battery.Autonomy = 1;
		}
	}

	if (ups.Battery.Extension != 0){
		ups.Battery.Autonomy = ups.Battery.Autonomy * (ups.Battery.Extension / 9.75);
	}


	upsdebugx(3,MSG_EXITFUNC, __func__);

}

/************************
 * Set Autonomy STAY700 *
 ************************/
static void setAutonomySTAY700(int data)
{
	double A55, A130, A190, A260;
	int P55W = data;
	int P130W = data;
	int P190W = data;
	int P260W = data;

	upsdebugx(3,MSG_ENTERFUNC, __func__);
	
	/* P55W Limits */
	if (data > 162){
		P55W = 162;
	} else if (data < 136){
		P55W = 136;
	}

	/* P130W Limits */
	if (data > 152){
		P130W = 152;
	} else if (data < 136){
		P130W = 136;
	}

	/* P190W Limits */
	if (data > 148){
		P190W = 148;
	} else if (data < 132){
		P190W = 132;
	}

	/* P260W Limits */
	if (data > 146){
		P260W = 146;
	} else if (data < 127){
		P260W = 127;
	}

	A55 = STAY700_DISCHARGE_055W[0] * pow(P55W, 2.0) + STAY700_DISCHARGE_055W[1] * P55W + STAY700_DISCHARGE_055W[2];
	A130 = STAY700_DISCHARGE_130W[0] * pow(P130W, 2.0) + STAY700_DISCHARGE_130W[1] * P130W + STAY700_DISCHARGE_130W[2];
	A190 = STAY700_DISCHARGE_190W[0] * pow(P190W, 2.0) + STAY700_DISCHARGE_190W[1] * P190W + STAY700_DISCHARGE_190W[2];
	A260 = STAY700_DISCHARGE_260W[0] * pow(P260W, 2.0) + STAY700_DISCHARGE_260W[1] * P260W + STAY700_DISCHARGE_260W[2];

	if (A55 < 0.0)
		A55 = 0.0;

	if (A130 < 0.0)
		A130 = 0.0;

	if (A190 < 0.0)
		A190 = 0.0;

	if (A260 < 0.0)
		A260 = 0.0;

	if (ups.Out.RealPower < 55.0){
		ups.Battery.Autonomy = 40;
	} else if (ups.Out.RealPower < 130.0){
		ups.Battery.Autonomy = (int)(ups.Out.RealPower - 55.0) / 75.0 * A130 + (130.0 - ups.Out.RealPower) / 75.0 * A55;
	} else if (ups.Out.RealPower < 190.0 ){
		ups.Battery.Autonomy = (int)(ups.Out.RealPower - 130.0) / 60.0 * A190 + (190.0 - ups.Out.RealPower) / 60.0 * A130;
	} else if (ups.Out.RealPower < 260.0 ){
		ups.Battery.Autonomy = (int)(ups.Out.RealPower - 190.0) / 70.0 * A260 + (260.0 - ups.Out.RealPower) / 70.0 * A190;
	} else {
		ups.Battery.Autonomy = 1;
	}

	if (ups.Battery.Autonomy < 0)
		ups.Battery.Autonomy = 0;

	if (ups.Battery.Extension != 0){
		ups.Battery.Autonomy = ups.Battery.Autonomy * (ups.Battery.Extension / 9.75);
	}

	upsdebugx(3,MSG_EXITFUNC, __func__);

}

/**************************
 * Set Autonomy PS350_CII *
 **************************/
static void setAutonomyPS350_CII(int data)
{

	upsdebugx(3,MSG_ENTERFUNC, __func__);

	if (ups.Out.RealPower < 120.0){
		/* The last number 0.5 is added to round the number */
		ups.Battery.Autonomy = (int)(2.0782 * pow(data, 2.0) - 603.64999999999998 * data + 43789.0 + 0.5);
		ups.Battery.Autonomy /= 60;
		if (ups.Battery.Autonomy > 17)
			ups.Battery.Autonomy = 17;

	} else if (ups.Out.RealPower < 210.0){
		/* The last number 0.5 is added to round the number */
		ups.Battery.Autonomy = (int)(0.455 * pow(data, 2.0) - 115.45 * data + 7297.3000000000002 + 0.5);
		ups.Battery.Autonomy /= 60;
		if (ups.Battery.Autonomy > 10)
			ups.Battery.Autonomy = 10;
	
	} else if (ups.Out.RealPower < 290.0){
		/* The last number 0.5 is added to round the number */
		ups.Battery.Autonomy = (int)(-0.3321 * pow(data, 2.0) + 107.92 * data - 8483.7000000000007  + 0.5);
		ups.Battery.Autonomy /= 60;
		if (ups.Battery.Autonomy > 5)
			ups.Battery.Autonomy = 5;

	} else if (ups.Out.RealPower < 330.0){
		/* The last number 0.5 is added to round the number */
		ups.Battery.Autonomy = (int)(-0.1636 * pow(data, 2.0) + 56.215000000000003 * data - 4564.3000000000002 + 0.5);
		ups.Battery.Autonomy /= 60;
		if (ups.Battery.Autonomy > 3)
			ups.Battery.Autonomy = 3;
	}


	if (ups.Battery.Extension != 0){
		ups.Battery.Autonomy = ups.Battery.Autonomy * (ups.Battery.Extension / 9.75);
	}

	upsdebugx(3,MSG_EXITFUNC, __func__);

}

/***********************
 * Set Autonomy PS2200 *
 ***********************/
static void setAutonomyPS2200(int data)
{

	double A = BatteryVoltage_X[ups.Model] * data + BatteryVoltage_Y[ups.Model];

	upsdebugx(3,MSG_ENTERFUNC, __func__);

	if (! ups.Status.Output){
		ups.Battery.Autonomy = 0;
		return;
	}

	if (ups.Out.RealPower < 103.0){
		ups.Battery.Autonomy = (int)(4.0561 * pow(A, 2.0) - 175.36000000000001 * A + 1899.4000000000001 + 0.5);
		if (ups.Battery.Autonomy > 68)
			ups.Battery.Autonomy = 68;

	} else if (ups.Out.RealPower < 242.0){
		ups.Battery.Autonomy = (int)(1.4616 * pow(A, 2.0) - 60.009 * A + 617.28999999999996 + 0.5);
		if (ups.Battery.Autonomy > 32)
			ups.Battery.Autonomy = 32;

	} else if (ups.Out.RealPower < 350.0){
		ups.Battery.Autonomy = (int)(-0.4561 * pow(A, 2.0) + 25.521000000000001  * A - 336.49000000000001 + 0.5); 
		if (ups.Battery.Autonomy > 16)
			ups.Battery.Autonomy = 16;

	} else if (ups.Out.RealPower < 470.0){
		ups.Battery.Autonomy = (int)(0.442 * pow(A, 2.0) - 16.548999999999999 * A + 152.03 + 0.5);
		if (ups.Battery.Autonomy > 14)
			ups.Battery.Autonomy = 14;

	} else if (ups.Out.RealPower < 617.0){
		ups.Battery.Autonomy = (int)(0.2526 * pow(A, 2.0) - 8.426600000000001 * A + 66.344999999999999 + 0.5);
		if (ups.Battery.Autonomy > 12)
			ups.Battery.Autonomy = 12;
        
	} else if (ups.Out.RealPower < 810.0){
		ups.Battery.Autonomy = (int)(0.187 * pow(A, 2.0) - 6.4851 * A + 54.984999999999999 + 0.5);
		if (ups.Battery.Autonomy > 9)
			ups.Battery.Autonomy = 9;
        
	} else if (ups.Out.RealPower < 1006.0){
		ups.Battery.Autonomy = (int)(-0.0116 * pow(A, 2.0) + 0.9699 * A - 14.324999999999999 + 0.5);
		if (ups.Battery.Autonomy > 4)
			ups.Battery.Autonomy = 4;

	} else if (ups.Out.RealPower < 1270.0){
		ups.Battery.Autonomy = (int)(-0.048 * pow(A, 2.0) + 3.0226 * A - 40.889000000000003 + 0.5);
		if (ups.Battery.Autonomy > 4)
			ups.Battery.Autonomy = 4;
	}
	
	if (ups.Battery.Extension != 0){
                ups.Battery.Autonomy = ups.Battery.Autonomy * (1.0 + ups.Battery.Extension / 14.0);
        }

	
	upsdebugx(3,MSG_EXITFUNC, __func__);

}

/*************************
 * Set Autonomy STAY1200 *
 *************************/
static void setAutonomySTAY1200(int data)
{

	upsdebugx(3,MSG_ENTERFUNC, __func__);


	if (ups.Out.RealPower < 120.0){
		ups.Battery.Autonomy = (int)(0.0982 * pow(data, 2.0) - 27.131 * data + 1875.1800000000001 + 0.5);
		if (ups.Battery.Autonomy > 60)
			ups.Battery.Autonomy = 60;

	} else if (ups.Out.RealPower < 230.0){
		ups.Battery.Autonomy = (int)(0.0663 * pow(data, 2.0) - 18.251999999999999 * data + 1257.2 + 0.5);
		if (ups.Battery.Autonomy > 30)
			ups.Battery.Autonomy = 30;

	} else if (ups.Out.RealPower < 330.0){
		ups.Battery.Autonomy = (int)(0.0225 * pow(data, 2.0) - 5.9113 * data + 389.07999999999998 + 0.5);
		if (ups.Battery.Autonomy > 13)
			ups.Battery.Autonomy = 13;

	} else if (ups.Out.RealPower < 430.0){
		ups.Battery.Autonomy = (int)(0.0009 * pow(data, 2.0) + 3.3172 * data - 272.33999999999997 + 0.5);
		if (ups.Battery.Autonomy > 10)
			ups.Battery.Autonomy = 10;

	} else if (ups.Out.RealPower < 530.0){
		ups.Battery.Autonomy = (int)(0.5711000000000001 * pow(data, 2.0) - 23.484000000000002 * data + 241.81 + 0.5);
		if (ups.Battery.Autonomy > 8)
			ups.Battery.Autonomy = 8;

	} else if (ups.Out.RealPower < 630.0){
		ups.Battery.Autonomy = (int)(0.0091 * pow(data, 2.0) - 2.3133 * data + 147.62 + 0.5);
		if (ups.Battery.Autonomy > 8)
			ups.Battery.Autonomy = 8;

	} else if (ups.Out.RealPower < 730.0){
		ups.Battery.Autonomy = (int)(0.0009 * pow(data, 2.0) - 1.422 * data + 136.22 + 0.5);
		if (ups.Battery.Autonomy > 6)
			ups.Battery.Autonomy = 6;
	}


	if (ups.Battery.Extension != 0){
		ups.Battery.Autonomy = ups.Battery.Autonomy * (ups.Battery.Extension / 9.75);
	}

	upsdebugx(3,MSG_EXITFUNC, __func__);

}

/*************************
 * Set Date/Time on UPS
 *************************/
static void setUPSClock( void )
{
	unsigned char package[12];
        int i, checksum = 0;
	time_t rawtime;
	struct tm *now;

	upsdebugx(3,MSG_ENTERFUNC, __func__);
	upsdebugx(1,MSG_SEPCLOCK);

	/* get current date/time */
	time(&rawtime);

	if (rawtime < nextupdate){
		upsdebugx(1,MSG_NEXTUPDATE, (int)rawtime, nextupdate);
		upsdebugx(1,MSG_SEPARATOR);

		return;
	} 

	nextupdate = rawtime + cycletime;

	now = localtime(&rawtime);
	ups.Localtime.Day = now->tm_mday;
	ups.Localtime.Month = now->tm_mon + 1;
	ups.Localtime.Year = now->tm_year+1900;
	ups.Localtime.Hours = now->tm_hour;
	ups.Localtime.Minutes = now->tm_min;
	ups.Localtime.Seconds = now->tm_sec;
	ups.Localtime.WeekDay = now->tm_wday;
	upsdebugx(1, MSG_SYSDATETIME, ups.Localtime.Day, ups.Localtime.Month, ups.Localtime.Year, ups.Localtime.Hours, ups.Localtime.Minutes, ups.Localtime.Seconds, ups.Localtime.WeekDay);

	N_YEARS = (int)(ups.Localtime.Year - BASE_YEAR) / 16;

	package[0] = 0xCF;
	package[1] = ups.Localtime.Hours;
	package[2] = ups.Localtime.Minutes;
	package[3] = ups.Localtime.Seconds;
	package[4] = ups.Schedule.HourOn;
	package[5] = ups.Schedule.MinuteOn;
	package[6] = ups.Schedule.HourOff;
	package[7] = ups.Schedule.MinuteOff;
	package[8] = (ups.Localtime.WeekDay << 5);
	package[8] |= ups.Localtime.Day;
	package[9] = (ups.Localtime.Month << 4);
	package[9] |= (ups.Localtime.Year - BASE_YEAR) % 16;
	/* Convert from internal to UPS representation */
	package[10] = 0x7F & rotateleft(ups.Schedule.DaysOfWeek, 1, 7);
	upsdebugx(1,"-->DaysOfWeek: %d, UPS WeekDay: %d, After Rotation: %02x",ups.Schedule.DaysOfWeek,1+ups.Localtime.WeekDay,  package[10]);

        for(i=0; i < 11; i++)
          checksum += package[i];

        package[11] = checksum % 256;

	upsdebug_hex(1, MSG_CMDPKG, (char *)package, sizeof(package));

	i = ser_send_buf(upsfd, (char *)package, sizeof(package));
	if (i != sizeof(package)){
		upslogx(LOG_ERR, MSG_SENDCMDERROR, __func__, i);
	}

	upsdebugx(1,MSG_SEPARATOR);
	upsdebugx(3,MSG_EXITFUNC, __func__);
}

/*************************
 * Read parameters and set variables
 *************************/
static void readParameters( void )
{
	int h,m;
	upsdebugx(3,MSG_ENTERFUNC, __func__);

	upsdebugx(1,MSG_SEPPARMS);

	if (testvar("sniffer")) detectPackages();

	if (testvar("battery_extension")){
		ups.Battery.Extension = atoi(getval("battery_extension"));
		upslogx(LOG_INFO,MSG_BATEXTREAD,ups.Battery.Extension);
	}

	if (testvar("clockupdate")){
		cycletime = atoi(getval("clockupdate"));
		upslogx(LOG_INFO,MSG_CLOCKREAD,cycletime);
	} else {
		/* Default value */
		cycletime = BASE_CLOCK_UPDATE_CYCLE;
	}

	if (testvar("time_on")){
		h=-1; m=-1;
		if (sscanf(getval("time_on"), "%d:%d",&h,&m) == 2){
			if ( (h>=0) && (h<24) && (m>=0) && (m<60)){
				ups.Schedule.HourOn=h;
				ups.Schedule.MinuteOn=m;
				upslogx(LOG_INFO,MSG_TIMEONREAD,ups.Schedule.HourOn, ups.Schedule.MinuteOn);
			}
		}
	}

	if (testvar("time_off")){
		h=-1; m=-1;
		if (sscanf(getval("time_off"), "%d:%d",&h,&m) == 2){
			if ( (h>=0) && (h<24) && (m>=0) && (m<60)){
				ups.Schedule.HourOff=h;
				ups.Schedule.MinuteOff=m;
				upslogx(LOG_INFO,MSG_TIMEOFFREAD,ups.Schedule.HourOff, ups.Schedule.MinuteOff);
			}
		}
	}

	if (testvar("daysofweek")){
		ups.Schedule.DaysOfWeek = BinStr2Byte(getval("daysofweek")) & 0x7F;
		upslogx(LOG_INFO,MSG_DOWREAD,ups.Schedule.DaysOfWeek);
	}


	upsdebugx(1,MSG_SEPARATOR);
}

/*************************
 * NUT function implementation
 *************************/
static int instcmd(const char *cmdname, const char *extra)
{
	int r;
	upsdebugx(3,MSG_ENTERFUNC, __func__);

	if (!strcasecmp(cmdname, "shutdown.stayoff")){
		r = ser_send_buf(upsfd, CMD_SHUTDOWN, 4);
	} else if (!strcasecmp(cmdname, "load.on")){
		r = ser_send_buf(upsfd, CMD_OUTPUT_ON, 4);
	} else if (!strcasecmp(cmdname, "load.off")){
		r = ser_send_buf(upsfd, CMD_OUTPUT_OFF, 4);
	} else if (!strcasecmp(cmdname, "test.failure.start")){
		r = ser_send_buf(upsfd, CMD_INPUT_OFF, 4);
	} else if (!strcasecmp(cmdname, "test.failure.stop")){
		r = ser_send_buf(upsfd, CMD_INPUT_ON, 4);
	} else {
		upslogx(LOG_NOTICE, "instcmd: unknown command [%s]", cmdname);
		return STAT_INSTCMD_UNKNOWN;
	}	

	if (r != 4){
		upslogx(LOG_ERR,MSG_SENDCMDERROR, cmdname, r);
		return STAT_INSTCMD_UNKNOWN;
	} else {
		upslogx(LOG_INFO, MSG_SENDCMDOK, cmdname, r);
		return STAT_INSTCMD_HANDLED;
	}
	upsdebugx(3,MSG_EXITFUNC, __func__);
}
