/* apcbzbr.h - APC BZ-BR models (Brazil)

   Copyright (C) 2014  Carlos Guidugli  <guidugli@gmail.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

   2014/09/22 - Version 0.01 - Initial release

*/

#ifndef INCLUDED_APCBZBR_H
#define INCLUDED_APCBZBR_H

#define ENDCHAR '\r'

#define DRIVER_NAME     "APC BZ-BR UPS driver"
#define DRIVER_VERSION  "0.01"
#define DRIVER_AUTHOR	"Carlos Guidugli <guidugli@gmail.com>"
#define MANUFACTURER	"APC"

typedef enum { false, true } bool_t;
const char* BoolName[] = { "False/Off", "True/On" };

/* number of times to retry UPS detection */
#define RETRIES	25

/* Default Dataset Size */
#define DS_SIZE 25

/* Define Base Year */
#define BASE_YEAR 2014

/* Default clock update cycle time */
#define BASE_CLOCK_UPDATE_CYCLE 86400

/* Default low battery charge percent */
#define LOW_BATT_CHARGE 30

/* Default warning battery charge percent */
#define WARN_BATT_CHARGE 50

/* Default low battery runtime */
#define BATT_RUNTIME_LOW 180

/* Time in seconds to shutdown server before scheduled shutdown */
#define SCHEDULED_FSD_SECONDS 600

/* Models */
typedef enum { PS350_CII, PS800, STAY1200, PS2200, PS2200_22, STAY700, BZ1500, UNKNOWN } model_t;

const char* ModelName[] = {"PS350_CII", "PS800", "STAY1200", "PS2200", "PS2200_22", "STAY700", "BZ1500", "UNKNOWN"}; 

const unsigned char DoWFilter[] = { 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01 };
 
/*********************************************************************************************
 *********************************** STRUCTURES **********************************************
 *********************************************************************************************/
/* Ups data records */
typedef struct {
	double Voltage;
	double VoltageMax;
	double VoltageMin;
	double Current;
	double Frequency;
} power_in_t;

typedef struct {
        double Voltage;
        double Current;
        double ApparentPower;
        double RealPower;
        double Frequency;
	double PowerFactor;
	double Load; /* Percentage of total load */
} power_out_t;

typedef struct {
        double Voltage;
        double Current;
	int    PercentCharged;
	int    Extension;
	int    Autonomy;
} power_bat_t;


typedef struct {
	bool_t VIn220; /* true if input voltage is 220 volts */
	bool_t VOut220; /* true if output voltage is 220 volts */
	bool_t Output; /* true if output is on, false otherwise */
	bool_t CriticalBattery; /* Battery in critical state */
	bool_t NetworkMode; /* Network mode state */
	bool_t BatteryMode; /* Battery mode state */
	bool_t BatteryCharging; /* Battery charging state */
	bool_t Overload; 
	bool_t Overheat; 
} status_t;

typedef struct {
	int	Day;
	int	Year;
	int	Month;
	int	Hours;
	int	Minutes;
	int 	Seconds;
	int	WeekDay;
} mytime_t;

typedef struct {
        int          HourOn; /* time to turn on the UPS */
        int          HourOff; /* time to turn off the UPS */
        int          MinuteOn; /* time to turn on the UPS */
        int          MinuteOff; /* time to turn off the UPS */
	/* DaysOfWeek represents the days that the ups should turn on/off 
	   The lsb (least significant bit) represents saturday and the hsb
	   represents sunday */
        int          DaysOfWeek; 
} schedule_t;

typedef struct {
	mytime_t     Localtime;
	schedule_t   Schedule;
	model_t      Model;
	double       Temperature;
	int 	     Efficiency;
	power_in_t   In;
	power_out_t  Out;
	power_bat_t  Battery;
	status_t     Status;
	int	     RelayMode;
} ups_t;

/* variable to hold UPS info */
ups_t ups;

/* UPS has only 4 bits to hold year information, so this variable
   is used to, based on the clock of the system, to define the 
   multiplication needed to get current year. 
   Basically: CURRENT YEAR = 16 * N_YEARS + BASE_YEAR
*/
int N_YEARS;

/* Variable to keep timer to update clock */
int nextupdate;
/* Variable that holds the time, in seconds, to wait between clock updates */
int cycletime;

/*********************************************************************************************
 ************************************ MESSAGES ***********************************************
 *********************************************************************************************/
#define MSG_BUFFER		"Input Buffer contents"
#define MSG_CHECKSUM_UNMATCH	"Calculated checksum (%02x) do not match dataset checksum (%02x)"
#define MSG_APCMODEL		"Detected %s model on %s"
#define MSG_DATETIME		"\tUPS Date/Time (dd/mm/yyyy HH:mm): %02d/%02d/%04d %02d:%02d:%02d, Weekday: %d"
#define MSG_SYSDATETIME		"SYSTEM Date/Time (dd/mm/yyyy HH:mm): %02d/%02d/%04d %02d:%02d:%02d, Weekday: %d"
#define MSG_SCHEDULE		"\tUPS Scheduled start: %02d:%02d, UPS Scheduled Stop: %02d:%02d, Days of week: %d"
#define MSG_ENTERFUNC		"Starting function %s..."
#define MSG_EXITFUNC		"Exiting function %s..."
#define MSG_RELAYMODE		"\tRelay mode: %02x"
#define MSG_OVERLOAD_ODD	"Overload not match between power usage (%s) and Overload Bit (%s)"
#define MSG_EFFICIENCY		"\tEfficiency (percent): %d"
#define MSG_TEMPERATURE		"\tTemperature (celcius): %.1f"

#define MSG_SENDCMDERROR	"Error sending command (%s) to UPS (%d bytes sent)"
#define MSG_SENDCMDOK		"Command (%s) successfully sent to UPS (%d bytes sent)"
#define MSG_CMDPKG		"Command package contents"

#define MSG_STDPKG		"STD PACKAGE:"
#define MSG_NOTSTDPKG		"OTHER PACKAGE:"
#define MSG_BATEXTREAD		"Battery extension parameter set to %d"
#define MSG_CLOCKREAD		"Clock update interval parameter set to %d"
#define MSG_TIMEONREAD		"UPS Power on time parameter set to %02d:%02d"
#define MSG_TIMEOFFREAD		"UPS Power off time parameter set to %02d:%02d"
#define MSG_DOWREAD		"Days of week parameter set to %d"
#define MSG_ERRPKGSIZE		"Dataset has not the correct size (got %d instead of %d)"
#define MSG_SETVAR              "Var Name: %s, Var Value: %s"
#define MSG_NEXTUPDATE          "setUPSClock: Now [%d] < Cycle [%d]"
#define MSG_SSDIMM1		"Schedule shutdown for --> Today: %s, Tomorrow: %s"
#define MSG_SSDIMM2		"Shutdown time: %d, Now: %d"
#define MSG_SSDIMM3		"Return value: %s"

#define MSG_UPSINPUT		"UPS Input information:"
#define MSG_UPSOUTPUT		"UPS Output information:"
#define MSG_UPSBATTERY		"UPS Battery information:"
#define MSG_UPSSTATUS		"UPS Status information:"
#define MSG_UPSINFO		"UPS information:"
#define MSG_VOLTAGE		"\tVoltage: %.1f"
#define MSG_VOLTAGEMAX		"\tMaximum Voltage: %.1f"
#define MSG_VOLTAGEMIN		"\tMinimum Voltage: %.1f"
#define MSG_CURRENT		"\tCurrent: %.1f"
#define MSG_FREQUENCY		"\tFrequency: %.1f"
#define MSG_APP_POWER		"\tApparent Power: %.1f"
#define MSG_REAL_POWER		"\tReal Power: %.1f"
#define MSG_PWR_FACTOR		"\tPower Factor: %.2f"
#define MSG_LOAD		"\tLoad: %.1f"
#define MSG_PERC_CHARGED	"\tPercent Charged: %d"
#define MSG_EXTENSION		"\tBattery Extension: %d Ah"
#define MSG_AUTONOMY		"\tBattery Autonomy: %d minutes"
#define MSG_VOUT220             "\tOutput 220v bit: %s"
#define MSG_BATCHARGE           "\tBattery Charging bit: %s"
#define MSG_CRITBAT             "\tCritical Battery bit: %s"
#define MSG_OUTPUT              "\tOutput bit: %s"
#define MSG_OVERHEAT            "\tOverheat bit: %s"
#define MSG_NETMODE             "\tNetwork Mode bit: %s"
#define MSG_BATMODE             "\tBattery Mode bit: %s"
#define MSG_VIN220              "\tInput 220v bit: %s"
#define MSG_OVERLOAD            "\tOverload Bit: %s"

#define MSG_UNKNOWN_MODEL	"No known model found"
#define MSG_UNKDS		"Unknown dataset format (DS0: %02x, DS24: %02x)"

#define MSG_DS			"Dataset Data:"
#define MSG_DSHEADER		"\t[0] Header: %02x"
#define MSG_DSOUTVOL		"\t[1] Output Voltage: %02x"
#define MSG_DSINVOL		"\t[2] Input Voltage: %02x"
#define MSG_DSBATVOL		"\t[3] Battery Voltage: %02x"
#define MSG_DSTEMP		"\t[4] UPS Temperature: %02x"
#define MSG_DSOUTCUR		"\t[5] Output Current: %02x"
#define MSG_DSRELAY		"\t[6] Relay Configuration: %02x"
#define MSG_DSREALPWR1		"\t[7] Real Power 1st Byte: %02x"
#define MSG_DSREALPWR2		"\t[8] Real Power 2nd Byte: %02x"
#define MSG_DSTIMESEC		"\t[9] UPS Time - Seconds: %02x"
#define MSG_DSTIMEMIN		"\t[10] UPS Time - Minutes: %02x"
#define MSG_DSTIMEHOUR		"\t[11] UPS Time - Hours: %02x"
#define MSG_DSNOTUSED		"\t[12] NOT USED: %02x"
#define MSG_DSHON		"\t[13] Time to switch on - hours: %02x"
#define MSG_DSMON		"\t[14] Time to switch on - minutes: %02x"
#define MSG_DSHOFF              "\t[15] Time to switch off - hours: %02x"
#define MSG_DSMOFF              "\t[16] Time to switch off - minutes: %02x"
#define MSG_DSWEEKDAYS          "\t[17] Weekdays to turn ups on/off: %02x"
#define MSG_DSDATE1             "\t[18] UPS Date - Days/Week Day: %02x"
#define MSG_DSDATE2             "\t[19] UPS Date - month/year: %02x"
#define MSG_DSSTATUS            "\t[20] Status Bits: %02x"
#define MSG_DSFREQ1             "\t[21] Frequency 1st Byte: %02x"
#define MSG_DSFREQ2             "\t[22] Frequency 2nd Byte: %02x"
#define MSG_DSCHKSUM            "\t[23] Checksum: %02x"
#define MSG_DSEODS              "\t[24] End of dataset: %02x"

#define MSG_SEPARATOR		"=========================================================="
#define MSG_SEPPARMS            "======================= PARAMETERS ======================="
#define MSG_SEPINFO             "======================= INFORMATION ======================"
#define MSG_SEPDSPACK           "==================== DATA SET PACKAGE ===================="
#define MSG_SEPREADSER          "===================== READING SERIAL ====================="
#define MSG_SEPCLOCK            "===================== SET UPS CLOCK ======================"


/*********************************************************************************************
 ******************************* UPS DATA TABLES *********************************************
 *********************************************************************************************/

/* X is used to multiply values and Y is used to sum values */

/* Output voltage when power network mode */
double	OutVoltage_X_Net[][8]=
{
	/* PS350_CII */ 
	{ 0.9286, 1.0186, 1.0414, 0.0, 0.8995, 0.9641, 0.0, 0.9507 },
	/* PS800 */
	{ 0.7058159232072275, 0.835352100910534, 0.8900756564307967, 0.0, 0.6681365671143182, 0.743826242189824, 0.0, 0.8654262224145392 },
	/* STAY1200 */
	{ 1.1549, 1.0925, 0.0, 0.0, 1.0929, 1.0885, 0.0, 0.8654262224145392D },
	/* PS2200 */
	{ 1.61, 1.71, 0.0, 0.0, 1.64, 1.64, 0.0, 1.72 },
	/* PS2200_22 */
	{ 2.5085, 2.8814, 3.0, 0.0, 0.0, 3.4571, 3.2, 0.0 },
	/* STAY700 */
	{ 0.7020000219345093, 0.8406000137329102, 0.0, 0.0, 0.0, 0.8330000042915344 },
	/* BZ1500 */
	{ 0.9266, 0.9266, 0.9266, 0.9266, 0.9266, 0.9266, 0.9266, 0.9266 },
	/* UNKNOWN */
	{ 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0}
};

double 	OutVoltage_Y_Net[][8]=
{
	/* PS350_CII */ 
	{ 9.9429, 9.2431, 16.933, 0.0, 13.109999999999999, 14.936, 0.0, 26.122 },
        /* PS800 */
        { 12.141, 5.2752, 10.43, 0.0, 17.565000000000001, 17.593, 0.0, 13.677 },
        /* STAY1200 */
        { -6.9157, 11.026, 10.43, 0.0, -0.6109, 12.18, 0.0, 13.677 },
        /* PS2200 */
        { -1.03, 2.45, 0.0, 0.0, -3.06, 6.94, 0.0, 11.630000000000001 },
        /* PS2200_22 */
	{ 30.507999999999999, 22.288, 30.5, 0.0, 0.0, -15.057, 18.199999999999999, 0.0 },
        /* STAY700 */
	{ 14.015000343322754, 12.130000114440918, 0.0, 0.0, 0.0, 12.5 },
        /* BZ1500 */
        { 5.0694, 5.0694, 5.0694, 5.0694, 5.0694, 5.0694, 5.0694, 5.0694 },
        /* UNKNOWN */
        { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0}
}; 

/* Output voltage when in battery mode */
double  OutVoltage_X_Bat[][8]= 
{
        /* PS350_CII */
	{ 13.9, 14.699999999999999, 15.470000000000001, 0.0, 0.0, 0.0, 0.0, 0.0 },
        /* PS800 */
        { 15.220000000000001, 16.510000000000002, 17.600000000000001, 0.0, 0.0, 0.0, 0.0, 0.0 },
        /* STAY1200 */
        { 7.9, 9.1, 17.600000000000001, 0.0, 0.0, 0.0, 0.0, 0.0 },
        /* PS2200 */
        { 7.9, 9.1, 17.600000000000001, 0.0, 0.0, 0.0, 0.0, 0.0 },
        /* PS2200_22 */
        { 2.5, 2.5, 2.5, 2.5, 2.5, 2.5, 2.5, 2.5 },
        /* STAY700 */
        { 15.220000000000001, 16.510000000000002, 17.600000000000001, 0.0, 0.0, 0.0, 0.0, 0.0 },
        /* BZ1500 */
        { 7.6, 9.73, 7.8, 9.58, 0.0, 0.0, 0.0, 0.0 },
        /* UNKNOWN */
        { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0}
};

double  OutVoltage_Y_Bat[][8]=
{
        /* PS350_CII */
        { 11.800000000000001, 10.52, 11.0, 0.0, 0.0, 0.0, 0.0, 0.0 },
        /* PS800 */
        { 10.82, 12.289999999999999, 13.699999999999999, 0.0, 0.0, 0.0, 0.0, 0.0 },
        /* STAY1200 */
        { 5.59, 9.470000000000001, 13.699999999999999, 0.0, 0.0, 0.0, 0.0, 0.0 },
        /* PS2200 */
        { 5.59, 9.470000000000001, 13.699999999999999, 0.0, 0.0, 0.0, 0.0, 0.0 },
        /* PS2200_22 */
        { 39.5, 39.5, 39.5, 39.5, 39.5, 39.5, 39.5, 39.5 },
        /* STAY700 */
        { 10.82, 12.289999999999999, 13.699999999999999, 0.0, 0.0, 0.0, 0.0, 0.0 },
        /* BZ1500 */
        { 6.58, 10.9, 5.32, 11.0, 0.0, 0.0, 0.0, 0.0 },
        /* UNKNOWN */
        { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0}
};

double  OutVoltage_MAX[8] = { 134.0, 128.0, 128.0, 128.0, 270.0, 128.0, 128.0, 0.0 };
double  OutVoltage_MIN[8] = { 96.0, 96.0, 96.0, 96.0, 170.0, 96.0, 96.0, 0.0 };

/* Input voltage */
/* PS350_CII, PS800, STAY1200, PS2200, PS2200_22, STAY700, BZ1500, UNKNOWN */
double  InVoltage_X[]={ 1.7875, 1.64, 1.64, 1.61, 1.5068, 1.68, 1.8, 0.0 }; 
double  InVoltage_Y[]={ 24.582001, 3.34, 9.34, -1.15, 11.37, 1.72, 2.224, 0.0 };
double	InVoltage_X_115_PS2200 = 1.58;
double	InVoltage_Y_115_PS2200 = 2.86;
double  InVoltage_MAX[][8] = 
{
	/* 110 Volts */
	{ 180.0, 180.0, 180.0, 180.0, 0.0, 180.0, 180.0, 0.0 },
	/* 220 Volts */
	{ 280.0, 280.0, 280.0, 280.0, 270.0, 280.0, 280.0, 0.0 }
};
double  InVoltage_MIN[][8] = 
{
	/* 110 Volts */
	{ 50.0, 50.0, 50.0, 50.0, 0.0, 50.0, 50.0, 0.0 },
	/* 220 Volts */
	{ 130.0, 130.0, 130.0, 130.0, 170.0, 130.0, 130.0, 0.0 }
};

double  InVoltageNominal[][8] =
{
	/* 110 Volts */
	{ 115.0, 115.0, 115.0, 115.0, 115.0, 115.0, 115.0, 0.0 },
	/* 220 Volts */
	{ 220.0, 220.0, 220.0, 220.0, 220.0, 220.0, 220.0, 0.0 }
};

double  OutVoltageNominal[][8] =
{
        /* 110 Volts */
        { 115.0, 115.0, 115.0, 115.0, 115.0, 115.0, 115.0, 0.0 },
        /* 220 Volts */
        { 220.0, 220.0, 220.0, 220.0, 220.0, 220.0, 220.0, 0.0 }
};


double  InCurrentNominal[][8] = 
{	
	/* 110 Volts */
	{ 0, 6, 10.5, 19, 0.0, 6, 13, 0 },
	/* 220 Volts */
	{ 0, 0, 6, 10, 10, 3.2, 6.8, 0 }
};

int     NominalFrequency[] = { 60, 60, 60, 60, 60, 60, 60, 0 };


/* Output Current when power network mode */
double	OutCurrent_X_Net[]={ 0.0487, 0.09580012262415696, 0.1297000038910001, 0.1500000059604645, 0.06340000033378601, 0.09580012262415696, 0.1264, 0.0 };
double	OutCurrent_Y_Net[]={ 0.1927, 0.4442075337597726,  0.5387060281204547, 0.7900000214576721, 0.409900009632111,   0.4442075337597726,  0.522,  0.0 };

/* Output current when in battery mode */
double  OutCurrent_X_Bat[]={ 0.0476, 0.0968054211035818, 0.1372, 0.1500000059604645, 0.06560000032186508, 0.0968054211035818, 0.1303, 0.0 };
double  OutCurrent_Y_Bat[]={ 0.2016, 0.4291845493562232, 0.3456, 0.699999988079071,  0.3853000104427338,  0.4291845493562232, 0.468,  0.0 };

/* Battery voltage */
double  BatteryVoltage_X[]={ 0.0711, 0.0711, 0.1551, 0.16, 0.16, 0.07410000264644623, 0.1513, 0.0 };
double  BatteryVoltage_Y[]={ 0.2525, 0.2525, 0.2525, -0.76, -0.76, 0.6104999780654907, 0.7153000000000001, 0.0 };
double	BatteryVoltage_Min[]={ 10.0, 10.0, 20.0, 20.0, 20.0, 10.0, 20.0, 0.0 };
double	BatteryVoltage_Max[]={ 14.5, 14.5, 29.5, 29.5, 29.5, 14.5, 29.5, 0.0 };
double	BatteryVoltage_Fluct[]={ 13.5, 13.54, 27.0, 27.0, 27.0, 13.54, 27.0, 0.0 };
int     BatteryVoltageNominal[] = { 12, 12,  24, 24, 24, 12, 24, 0 };
const char*   BatteryType[] = { "PbAc", "PbAc", "PbAc", "PbAc", "PbAc", "PbAc", "PbAc", "" };
int     BatteryCapacity[] = { 7, 7, 7, 14, 14, 7, 14, 0 };
int     BatteryPacks[] = { 1, 1, 2, 4, 4, 1, 2, 0 };


double  LowBattery_Threshold[] = { 11.0, 11.0, 19.0, 19.0, 19.0, 11.0, 19.0, 0.0 };
double  CritBattery_Threshold[] = { 10.0, 10.0, 10.0, 10.0, 10.0, 11.0, 10.0, 0.0 };

/* Power values */
double  UtilPower_Detect_C1_X[][8]=
{
        /* PS350_CII */
        { 0.0469, 0.0518, 0.0577, 0.0, 0.0462, 0.0519, 0.0, 0.0571 },
        /* PS800 */
        { 0.079, 0.089, 0.0972, 0.0, 0.0805, 0.0883, 0.0, 0.09810000000000001 },
        /* STAY1200 - Not Used */
        { 0.079, 0.089, 0.0972, 0.0, 0.0805, 0.0883, 0.0, 0.09810000000000001 },
        /* PS2200 */
        { 0.24, 0.26, 0.0, 0.0, 0.24, 0.26, 0.0, 0.28 },
        /* PS2200_22 */
        { 0.2262, 0.2262, 0.2536, 0.24, 0.24, 0.2303, 0.2535, 0.28 },
        /* STAY700 */
        { 0.079, 0.089, 0.0972, 0.0, 0.0805, 0.0883, 0.0, 0.09810000000000001 },
        /* BZ1500 */
        { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 },
        /* UNKNOWN */
        { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 }
};

double  UtilPower_Detect_C1_Y[][8]=
{
        /* PS350_CII */
        { 29.599, 28.41, 26.039000000000001, 0.0, 30.108000000000001, 27.038, 0.0, 26.521999999999998 },
        /* PS800 */
        { 49.106999999999999, 45.448999999999998, 48.091999999999999, 0.0, 43.633000000000003, 47.585000000000001, 0.0, 48.831000000000003 },
        /* STAY1200 - Not Used */
        { 49.106999999999999, 45.448999999999998, 48.091999999999999, 0.0, 43.633000000000003, 47.585000000000001, 0.0, 48.831000000000003 },
        /* PS2200 */
        { 83.150000000000006, 81.230000000000004, 0.0, 0.0, 3.489999999999995, 79.049999999999997, 0.0, 85.670000000000002 },
        /* PS2200_22 */
        { 21.074999999999999, 21.074999999999999, 46.886000000000003, 20.280999999999999, 36.957999999999998, 43.75, 36.290999999999997, 85.670000000000002 },
        /* STAY700 */
        { 49.106999999999999, 45.448999999999998, 48.091999999999999, 0.0, 43.633000000000003, 47.585000000000001, 0.0, 48.831000000000003 },
        /* BZ1500 */
        { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 },
        /* UNKNOWN */
        { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 }
};

double  UtilPower_Detect_C2_X[][8]=
{
        /* PS350_CII */
        { 0.575, 0.0612, 0.0617, 0.0, 0.0551, 0.0591, 0.0, 0.0703 },
        /* PS800 */
        { 0.07630000000000001, 0.081, 0.0919, 0.0, 0.0741, 0.0828, 0.0, 0.09379999999999999 },
        /* STAY1200 */
        { 0.07630000000000001, 0.081, 0.0919, 0.0, 0.0741, 0.0828, 0.0, 0.09379999999999999 },
        /* PS2200 */
        { 0.07630000000000001, 0.081, 0.0919, 0.0, 0.0741, 0.0828, 0.0, 0.09379999999999999 },
        /* PS2200_22 */
        { 0.07630000000000001, 0.081, 0.0919, 0.0, 0.0741, 0.0828, 0.0, 0.09379999999999999 },
        /* STAY700 */
        { 0.07630000000000001, 0.081, 0.0919, 0.0, 0.0741, 0.0828, 0.0, 0.09379999999999999 },
        /* BZ1500 */
        { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 },
        /* UNKNOWN */
        { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 }
};

double  UtilPower_Detect_C2_Y[][8]=
{
        /* PS350_CII */
        { 34.363999999999997, 35.927999999999997, 48.289000000000001, 0.0, 32.933999999999997, 34.054000000000002, 0.0, 18.75 },
        /* PS800 */
        { 81.731999999999999, 94.459000000000003, 86.686000000000007, 0.0, 84.656999999999996, 84.998999999999995, 0.0, 78.096999999999994 },
        /* STAY1200 */
        { 81.731999999999999, 94.459000000000003, 86.686000000000007, 0.0, 84.656999999999996, 84.998999999999995, 0.0, 78.096999999999994 },
        /* PS2200 */
        { 81.731999999999999, 94.459000000000003, 86.686000000000007, 0.0, 84.656999999999996, 84.998999999999995, 0.0, 78.096999999999994 },
        /* PS2200_22 */
        { 81.731999999999999, 94.459000000000003, 86.686000000000007, 0.0, 84.656999999999996, 84.998999999999995, 0.0, 78.096999999999994 },
        /* STAY700 */
        { 81.731999999999999, 94.459000000000003, 86.686000000000007, 0.0, 84.656999999999996, 84.998999999999995, 0.0, 78.096999999999994 },
        /* BZ1500 */
        { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 },
        /* UNKNOWN */
        { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 }
};

double  UtilPower_Detect_C3_X[][8]=
{
        /* PS350_CII */
        { 0.0514, 0.0573, 0.0673, 0.0, 0.0534, 0.0531, 0.0, 0.0621 },
        /* PS800 */
        { 0.07439999999999999, 0.0808, 0.0885, 0.0, 0.0732, 0.08400000000000001, 0.0, 0.0955 },
        /* STAY1200 */
        { 0.07439999999999999, 0.0808, 0.0885, 0.0, 0.0732, 0.08400000000000001, 0.0, 0.0955 },
        /* PS2200 */
        { 0.07439999999999999, 0.0808, 0.0885, 0.0, 0.0732, 0.08400000000000001, 0.0, 0.0955 },
        /* PS2200_22 */
        { 0.07439999999999999, 0.0808, 0.0885, 0.0, 0.0732, 0.08400000000000001, 0.0, 0.0955 },
        /* STAY700 */
        { 0.07439999999999999, 0.0808, 0.0885, 0.0, 0.0732, 0.08400000000000001, 0.0, 0.0955 },
        /* BZ1500 */
        { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 },
        /* UNKNOWN */
        { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 }
};

double  UtilPower_Detect_C3_Y[][8]=
{
        /* PS350_CII */
        { 61.183999999999997, 52.561, 29.872, 0.0, 43.087000000000003, 60.911999999999999, 0.0, 43.997 },
        /* PS800 */
        { 122.06, 122.90000000000001, 125.75, 0.0, 120.39, 108.52, 0.0, 92.239000000000004 },
        /* STAY1200 */
        { 122.06, 122.90000000000001, 125.75, 0.0, 120.39, 108.52, 0.0, 92.239000000000004 },
        /* PS2200 */
        { 122.06, 122.90000000000001, 125.75, 0.0, 120.39, 108.52, 0.0, 92.239000000000004 },
        /* PS2200_22 */
        { 122.06, 122.90000000000001, 125.75, 0.0, 120.39, 108.52, 0.0, 92.239000000000004 },
        /* STAY700 */
        { 122.06, 122.90000000000001, 125.75, 0.0, 120.39, 108.52, 0.0, 92.239000000000004 },
        /* BZ1500 */
        { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 },
        /* UNKNOWN */
        { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 }
};

double  UtilPower_C1_X[][8]=
{
        /* PS350_CII */
        { 0.0469, 0.0518, 0.0577, 0.0, 0.0462, 0.0519, 0.0, 0.0571 },
        /* PS800 */
        { 0.08040007075206226, 0.08939999999999999, 0.0999, 0.0, 0.0813, 0.0905, 0.0, 0.1005 },
        /* STAY1200 */
        { 0.08040007075206226, 0.08939999999999999, 0.0999, 0.0, 0.0813, 0.0905, 0.0, 0.1005 },
        /* PS2200 */
        { 0.08040007075206226, 0.08939999999999999, 0.0999, 0.0, 0.0813, 0.0905, 0.0, 0.1005 },
        /* PS2200_22 */
        { 0.08040007075206226, 0.08939999999999999, 0.0999, 0.0, 0.0813, 0.0905, 0.0, 0.1005 },
        /* STAY700 */
        { 0.08040007075206226, 0.08939999999999999, 0.0999, 0.0, 0.0813, 0.0905, 0.0, 0.1005 },
        /* BZ1500 */
        { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 },
        /* UNKNOWN */
        { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 }
};

double  UtilPower_C1_Y[][8]=
{
        /* PS350_CII */
        { 29.599, 28.41, 26.039000000000001, 0.0, 30.108000000000001, 27.038, 0.0, 26.521999999999998 },
        /* PS800 */
	{ 45.292000000000002, 41.927999999999997, 41.726999999999997, 0.0, 40.268999999999998, 41.810000000000002, 0.0, 43.457999999999998 },
        /* STAY1200 */
	{ 45.292000000000002, 41.927999999999997, 41.726999999999997, 0.0, 40.268999999999998, 41.810000000000002, 0.0, 43.457999999999998 },
        /* PS2200 */
	{ 45.292000000000002, 41.927999999999997, 41.726999999999997, 0.0, 40.268999999999998, 41.810000000000002, 0.0, 43.457999999999998 },
        /* PS2200_22 */
	{ 45.292000000000002, 41.927999999999997, 41.726999999999997, 0.0, 40.268999999999998, 41.810000000000002, 0.0, 43.457999999999998 },
        /* STAY700 */
	{ 45.292000000000002, 41.927999999999997, 41.726999999999997, 0.0, 40.268999999999998, 41.810000000000002, 0.0, 43.457999999999998 },
        /* BZ1500 */
        { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 },
        /* UNKNOWN */
        { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 }
};

double  UtilPower_C2_X[][8]=
{
        /* PS350_CII */
        { 0.0451, 0.0505, 0.0545, 0.0, 0.0462, 0.0491, 0.0, 0.0298 },
        /* PS800 */
	{ 0.08630063689870031, 0.0946, 0.1068, 0.0, 0.08599999999999999, 0.09669999999999999, 0.0, 0.1088 },
        /* STAY1200 */
	{ 0.08630063689870031, 0.0946, 0.1068, 0.0, 0.08599999999999999, 0.09669999999999999, 0.0, 0.1088 },
        /* PS2200 */
	{ 0.08630063689870031, 0.0946, 0.1068, 0.0, 0.08599999999999999, 0.09669999999999999, 0.0, 0.1088 },
        /* PS2200_22 */
	{ 0.08630063689870031, 0.0946, 0.1068, 0.0, 0.08599999999999999, 0.09669999999999999, 0.0, 0.1088 },
        /* STAY700 */
	{ 0.08630063689870031, 0.0946, 0.1068, 0.0, 0.08599999999999999, 0.09669999999999999, 0.0, 0.1088 },
        /* BZ1500 */
        { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 },
        /* UNKNOWN */
        { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 }
};

double  UtilPower_C2_Y[][8]=
{
        /* PS350_CII */
        { 13.973000000000001, 8.968500000000001, 16.709, 0.0, 12.000999999999999, 15.302, 0.0, 57.213000000000001 },
        /* PS800 */
        { 8.3927, 9.2393, 8.2852, 0.0, 8.301, 6.7636, 0.0, 8.2842 },
        /* STAY1200 */
        { 8.3927, 9.2393, 8.2852, 0.0, 8.301, 6.7636, 0.0, 8.2842 },
        /* PS2200 */
        { 8.3927, 9.2393, 8.2852, 0.0, 8.301, 6.7636, 0.0, 8.2842 },
        /* PS2200_22 */
        { 8.3927, 9.2393, 8.2852, 0.0, 8.301, 6.7636, 0.0, 8.2842 },
        /* STAY700 */
        { 8.3927, 9.2393, 8.2852, 0.0, 8.301, 6.7636, 0.0, 8.2842 },
        /* BZ1500 */
        { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 },
        /* UNKNOWN */
        { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 }
};

double  UtilPower_C3_X[][8]=
{
        /* PS350_CII */
        { 0.0421, 0.0474, 0.0593, 0.0, 0.0438, 0.0449, 0.0, 0.0522 },
        /* PS800 */
        { 0.0896001146881468, 0.09909999999999999, 0.1116, 0.0, 0.09669999999999999, 0.1068, 0.0, 0.1169 },
        /* STAY1200 */
        { 0.0896001146881468, 0.09909999999999999, 0.1116, 0.0, 0.09669999999999999, 0.1068, 0.0, 0.1169 },
        /* PS2200 */
        { 0.0896001146881468, 0.09909999999999999, 0.1116, 0.0, 0.09669999999999999, 0.1068, 0.0, 0.1169 },
        /* PS2200_22 */
        { 0.0896001146881468, 0.09909999999999999, 0.1116, 0.0, 0.09669999999999999, 0.1068, 0.0, 0.1169 },
        /* STAY700 */
        { 0.0896001146881468, 0.09909999999999999, 0.1116, 0.0, 0.09669999999999999, 0.1068, 0.0, 0.1169 },
        /* BZ1500 */
        { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 },
        /* UNKNOWN */
        { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 }
};

double  UtilPower_C3_Y[][8]=
{
        /* PS350_CII */
        { 23.795999999999999, 15.544, -4.6501, 0.0, 19.27, 31.405000000000001, 0.0, 21.359999999999999 },
        /* PS800 */
        { -31.114999999999998, -33.777000000000001, -33.826000000000001, 0.0, -59.512999999999998, -57.728999999999999, 0.0, -41.332999999999998 },
        /* STAY1200 */
        { -31.114999999999998, -33.777000000000001, -33.826000000000001, 0.0, -59.512999999999998, -57.728999999999999, 0.0, -41.332999999999998 },
        /* PS2200 */
        { -31.114999999999998, -33.777000000000001, -33.826000000000001, 0.0, -59.512999999999998, -57.728999999999999, 0.0, -41.332999999999998 },
        /* PS2200_22 */
        { -31.114999999999998, -33.777000000000001, -33.826000000000001, 0.0, -59.512999999999998, -57.728999999999999, 0.0, -41.332999999999998 },
        /* STAY700 */
        { -31.114999999999998, -33.777000000000001, -33.826000000000001, 0.0, -59.512999999999998, -57.728999999999999, 0.0, -41.332999999999998 },
        /* BZ1500 */
        { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 },
        /* UNKNOWN */
        { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 }
};

int Autonomy[] = { 26, 25, 24, 23, 22, 21, 20, 18, 17, 16, 14, 13, 11, 9, 7, 4, 2, 1 };

double STAY700_DISCHARGE_055W[] = { 0.0339, -8.75681, 561.69 };
double STAY700_DISCHARGE_130W[] = { 0.0126, -2.981, 173.12 };
double STAY700_DISCHARGE_190W[] = { 0.0127, -3.2775, 210.72 };
double STAY700_DISCHARGE_260W[] = { -0.0012, 0.4685, -41.169998 };
double STAY700_DISCHARGE_320W[] = { -0.0025, 0.8645, -73.039001 };

double Overload_Threshold[] = { 655.0, 435.0, 655.0, 1200.0, 1200.0, 300.0, 820.0 };
int NominalApparentPower[] = { 350, 800, 1200, 2200, 2200, 700, 1500, 0 };
int NominalRealPower[] = { 0, 420, 600, 1360, 1360, 350, 825, 0 };
double  NominalPowerFactor[] = { 0, 0.525, 0.5, 0.618, 0.618, 0.5, 0.55, 0.0 };

/* Command packages */
char CMD_SHUTDOWN[] = { 204, 9, 9, 18 };
char CMD_INPUT_ON[] = { 204, 3, 3, 6 };
char CMD_INPUT_OFF[] = { 204, 4, 4, 8 };
char CMD_OUTPUT_ON[] = { 204, 1, 1, 2 };
char CMD_OUTPUT_OFF[] = { 204, 2, 2, 4 };


/*********************************************************************************************
 **************************** FUNCTION DECLARATIONS  *****************************************
 *********************************************************************************************/

/* Function declarations */

static unsigned char rotateleft(unsigned char, unsigned char, unsigned char);
static unsigned char rotateright(unsigned char, unsigned char, unsigned char);
const char *byte_to_binary(int);
static unsigned char BinStr2Byte(char *);
static void detectPackages(void);
static void printDBGInformation(void);
static void printDBGDataSet(const unsigned char *);
static bool_t scheduleShutdownImminent(void);

static int  instcmd(const char *, const char *);
static int setvar(const char *, const char *);
static void apc_update_ups_info(const unsigned char * );
static void apc_readserial(void);
static void setStatus(int);
static void setDStates(void);
static void setRWDStates(void);
static void setUPSClock( void );
static void readParameters( void );


/* Out */

static void setOutputVoltage(int); /* depends on Out.Current, Battery.Voltage */
static void setOutputCurrent(int);
static void setApparentPower(void); /* depends on Out.Voltage & Out.Current */
static void setRealPower(int); /* depends on Out.ApparentPower, Out.Current */
static void setOutputFrequency(void); /* depends on In.Frequency */
static void setPowerFactor(void); /* depends on Out.ApparentPower, Out.RealPower */
static void setPowerLoad(void); /* Out.RealPower */


/* In */
static void setInputVoltage(int);
static void setInputCurrent(void); /* depends on In.Voltage & Out.ApparentPower */
static void setInputFrequency(long int); /* Depends on In.Voltage */

/* Battery */
static void setBatteryVoltage(int);
static void setBatteryPercentCharged(void); /* Depends on Battery.Voltage */
static void setAutonomy(int); /* depends on Battery.Voltage */
static void setAutonomyPS800(int);
static void setAutonomyPS350_CII(int);
static void setAutonomySTAY700(int);
static void setAutonomyPS2200(int);
static void setAutonomySTAY1200(int);

static void setEfficiency(void); /* Depends on In and Out Current */

#endif /* INCLUDED_APCBZBR_H */






