## OpenCA::Token::nCipher.pm 
##
## Copyright (C) 2003,2004 Michael Bell <michael.bell@web.de>
## All rights reserved.
##
## Adapted to nCipher HSM: 2004-05-25 Martin Bartosch <m.bartosch@cynops.de>
## code based on OpenCA::Token::OpenSSL and others
##
##    This library is free software; you can redistribute it and/or
##    modify it under the terms of the GNU Lesser General Public
##    License as published by the Free Software Foundation; either
##    version 2.1 of the License, or (at your option) any later version.
##
##    This library is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
##    Lesser General Public License for more details.
##
##    You should have received a copy of the GNU Lesser General Public
##    License along with this library; if not, write to the Free Software
##    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
##
##
##
## #########################################################################

use strict;

###############################################################
##        ============    nCipher Token    =============     ##
###############################################################

package OpenCA::Token::nCipher;

use OpenCA::OpenSSL;

use FileHandle;
our ($STDERR, $STDOUT);
$STDOUT = \*STDOUT;
$STDERR = \*STDERR;

our ($errno, $errval);
our $AUTOLOAD;

($OpenCA::Token::nCipher::VERSION = '$Revision$' )=~ s/(?:^.*: (\d+))|(?:\s+\$$)/defined $1?"0\.9":""/eg;

# Errorcode prefix: 715*
# 71 token modules
# 5  nCipher module

# Preloaded methods go here.

## create a new OpenSSL tokens
sub new {
    my $that = shift;
    my $class = ref($that) || $that;

    my $self = {
                DEBUG     => 0,
                debug_fd  => $STDERR,
                ## debug_msg => ()
               };

    bless $self, $class;

    my $keys = { @_ };
    $self->{CRYPTO}       = $keys->{OPENCA_CRYPTO};
    $self->{gettext}      = $keys->{GETTEXT};
    $self->{NAME}         = $keys->{OPENCA_TOKEN};
    $self->{MODE}         = $keys->{TOKEN_MODE};

    return $self->setError (7151010,
               $self->{gettext} ("Crypto layer is not defined."))
        if (not $self->{CRYPTO});
    return $self->setError (7151012,
               $self->{gettext} ("The name of the token is not defined."))
        if (not $self->{NAME});

    $keys->{ENGINE} = "chil";
    $keys->{CERT} = $keys->{PEM_CERT};
    $keys->{DEBUG} = 1;

    $self->{OPENSSL} = OpenCA::OpenSSL->new ( %{$keys} );

    return $self->setError ($OpenCA::OpenSSL::errno, $OpenCA::OpenSSL::errval)
        if (not $self->{OPENSSL});

    return $self;
}

sub setError {
    my $self = shift;
    
    if (scalar (@_) == 4) {
        my $keys = { @_ };
        $errval = $keys->{ERRVAL};
        $errno  = $keys->{ERRNO};
    } else {
        $errno  = $_[0];
        $errval = $_[1];
    }
    
    return undef if (not $errno);
    
    print $STDERR "PKI Master Alert: OpenCA::Token::nCipher error\n";
    print $STDERR "PKI Master Alert: Aborting all operations\n";
    print $STDERR "PKI Master Alert: Error:   $errno\n";
    print $STDERR "PKI Master Alert: Message: $errval\n";
    print $STDERR "PKI Master Alert: debugging messages of empty token follow\n"
;
    $self->{debug_fd} = $STDERR;
    $self->debug ();
    $self->{debug_fd} = $STDOUT;

    ## support for: return $self->setError (1234, "Something fails.") if (not $xyz);
    return undef;
}

sub errno {
    my $self = shift;
    return $self->{errno};
}

sub errval {
    my $self = shift;
    return $self->{errval};
}

sub debug {

    my $self = shift;
    if ($_[0]) {
        $self->{debug_msg}[scalar @{$self->{debug_msg}}] = $_[0];
        $self->debug () if ($self->{DEBUG});
    } else {
        my $oldfh;
        if ($self->{errno})
        {
            $oldfh = select $self->{debug_fd};
            print "PKI Debugging: OpenCA::Token::nCipher error\n";
            print "PKI Debugging: Aborting all operations\n";
            print "PKI Debugging: Error:   ".$self->{errno}."\n";
            print "PKI Debugging: Message: ".$self->{errval}."\n";
            print "PKI Debugging: debugging messages of OpenSSL token follow\n";
            select $oldfh;
        }
        my $msg;
        foreach $msg (@{$self->{debug_msg}}) {
            $msg =~ s/ /&nbsp;/g if ($self->{debug_fd} eq $STDOUT);
            my $oldfh = select $self->{debug_fd};
            print "OpenCA::Token::OpenSSL->$msg<br>\n";
            select $oldfh;
        }
        $self->{debug_msg} = ();
    }
}

# fake login function; usually the PKI admin unlocks the HSM in another
# session by running /opt/nfast/bin/with-nfast pause
# after this has completed the private key is accessible in the system,
# so just verify if we can access the private key
#
sub login {
    my $self = shift;

    if (! $self->online())
    {
	return $self->setError (7153050,
				$self->{gettext} ("HSM keys are not preloaded"));
    }

    $self->{ONLINE} = 1;
    return 1;
}

# fake logout function
sub logout {
    my $self = shift;

    $self->{ONLINE} = 0;
    return 1;
}

sub online {
    my $self = shift;
    return 1;
}

sub keyOnline {
    my $self = shift;
    return undef if (not $self->{ONLINE});
    return 1;
}

sub getMode {
    return "standby";
}

# # failover to default token OpenSSL which uses -engine
# # see new to get an idea what is going on
sub AUTOLOAD {
    my $self = shift;
    if ($AUTOLOAD =~ /OpenCA::OpenSSL/)
    {
        print STDERR "PKI Master Alert: OpenCA::OpenSSL is missing a function\n";
        print STDERR "PKI Master Alert: $AUTOLOAD\n";
        $self->setError (666,
			 $self->{gettext} ("OpenCA::OpenSSL is missing a function. __FUNCTION
__",
					   "__FUNCTION__", $AUTOLOAD));
        return undef;
    }
    $self->debug ("OpenCA::Token::nCipher: AUTOLOAD => $AUTOLOAD");

    return 1 if ($AUTOLOAD eq 'OpenCA::Token::nCipher::DESTROY');

    my $function = $AUTOLOAD;
    $function =~ s/.*:://g;

    $self->debug ("nCipher AUTOLOAD function $function");
    
    my $ret = $self->{OPENSSL}->$function ( @_ );
    $self->setError ($OpenCA::OpenSSL::errno, $OpenCA::OpenSSL::errval);
    return $ret;
 }


sub genKey {
    my $self = shift;

    my $keys = { @_ };

    return $self->setError (7154001,
			    $self->{gettext} ("Key generation not supported."));
}

1;
