/*
 * This file implements the ppc64 specific
 * support for the perfmon2 interface
 *
 * Copyright (c) 2005 David Gibson, IBM Corporation.
 *
 * based on versions for other architectures:
 * Copyright (c) 2005-2006 Hewlett-Packard Development Company, L.P.
 * Contributed by Stephane Eranian <eranian@hpl.hp.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA
  */
#include <linux/interrupt.h>
#include <linux/perfmon.h>

/*
 * collect pending overflowed PMDs. Called from pfm_ctxsw()
 * and from PMU interrupt handler. Must fill in set->povfl_pmds[]
 * and set->npend_ovfls. Interrupts are masked
 */
static void __pfm_get_ovfl_pmds(struct pfm_context *ctx, struct pfm_event_set *set)
{
 	u64 new_val, wmask;
	u64 *used_mask;
	unsigned int i, max;

	max = pfm_pmu_conf->max_cnt_pmd;
	used_mask = set->used_pmds;
	wmask = PFM_ONE_64 << pfm_pmu_conf->counter_width;

	for (i = 0; i < max; i++) {
		/* assume all PMD are counters */
		if (pfm_bv_isset(used_mask, i)) {
			new_val = pfm_arch_read_pmd(ctx, i);

 			PFM_DBG_ovfl("pmd%u new_val=0x%llx bit=%d\n",
  				 i, (unsigned long long)new_val,
				 (new_val&wmask) ? 1 : 0);

  			if (new_val & wmask) {
				pfm_bv_set(set->povfl_pmds, i);
				set->npend_ovfls++;
			}
		}
	}
}
static void pfm_stop_active(struct task_struct *task, struct pfm_context *ctx,
			       struct pfm_event_set *set)
{
	unsigned int i, max;

	max = pfm_pmu_conf->max_pmc;

	/*
	 * clear enable bits
	 */
	for (i = 0; i < max; i++) {
		if (pfm_bv_isset(set->used_pmcs, i))
			pfm_arch_write_pmc(ctx, i,0);
	}

	if (set->npend_ovfls)
		return;

	__pfm_get_ovfl_pmds(ctx, set);
}

/*
 * Called from pfm_ctxsw(). Task is guaranteed to be current.
 * Context is locked. Interrupts are masked. Monitoring is active.
 * PMU access is guaranteed. PMC and PMD registers are live in PMU.
 *
 * for per-thread:
 * 	must stop monitoring for the task
 * Return:
 * 	non-zero : did not save PMDs (as part of stopping the PMU)
 * 	       0 : saved PMDs (no need to save them in caller)
 */
int pfm_arch_ctxswout_thread(struct task_struct *task, struct pfm_context *ctx,
		       	      struct pfm_event_set *set)
{
	/*
	 * disable lazy restore of PMC registers.
	 */
	set->priv_flags |= PFM_SETFL_PRIV_MOD_PMCS;

	pfm_stop_active(task, ctx, set);

	return 1;
}

/*
 * Called from pfm_stop() and pfm_ctxsw() when idle
 * task and EXCL_IDLE is on.
 *
 * Interrupts are masked. Context is locked. Set is the active set.
 *
 * For per-thread:
 *   task is not necessarily current. If not current task, then
 *   task is guaranteed stopped and off any cpu. Access to PMU
 *   is not guaranteed. Interrupts are masked. Context is locked.
 *   Set is the active set.
 *
 * For system-wide:
 * 	task is current
 *
 * must disable active monitoring.
 */
void pfm_arch_stop(struct task_struct *task, struct pfm_context *ctx,
		struct pfm_event_set *set)
{
	/*
	 * stop live registers and collect pending overflow
	 */
	if (task == current)
		pfm_stop_active(task, ctx, set);
}

/*
 * function called from pfm_unload_context_*(). Context is locked.
 * interrupts are masked. task is not guaranteed to be current task.
 * Access to PMU is not guaranteed.
 *
 * function must do whatever arch-specific action is required on unload
 * of a context.
 *
 * called for both system-wide and per-thread. task is NULL for ssytem-wide
 */
void pfm_arch_unload_context(struct pfm_context *ctx, struct task_struct *task)
{
}

/*
 * called from pfm_start() or pfm_ctxsw() when idle task and
 * EXCL_IDLE is on.
 *
 * Interrupts are masked. Context is locked. Set is the active set.
 *
 * For per-trhead:
 * 	Task is not necessarily current. If not current task, then task
 * 	is guaranteed stopped and off any cpu. Access to PMU is not guaranteed.
 *
 * For system-wide:
 * 	task is always current
 *
 * must enable active monitoring.
 */
static void __pfm_arch_start(struct task_struct *task, struct pfm_context *ctx,
			     struct pfm_event_set *set)
{
	unsigned int i, max_pmc;

	if (task != current)
		return;

	max_pmc = pfm_pmu_conf->max_pmc;

	for (i = 0; i < max_pmc; i++) {
		if (pfm_bv_isset(set->used_pmcs, i))
		    pfm_arch_write_pmc(ctx, i, set->pmcs[i]);
	}
}

void pfm_arch_start(struct task_struct *task, struct pfm_context *ctx,
		    struct pfm_event_set *set)
{
	/*
	 * mask/unmask uses start/stop mechanism, so we cannot allow
	 * while masked.
	 */
	if (ctx->state == PFM_CTX_MASKED)
		return;

	__pfm_arch_start(task, ctx, set);
}

/*
 * function called from pfm_switch_sets(), pfm_context_load_thread(),
 * pfm_context_load_sys(), pfm_ctxsw(), pfm_switch_sets()
 * context is locked. Interrupts are masked. set cannot be NULL.
 * Access to the PMU is guaranteed.
 *
 * function must restore all PMD registers from set.
 */
void pfm_arch_restore_pmds(struct pfm_context *ctx, struct pfm_event_set *set)
{
	u64 ovfl_mask, val, *pmds;
	u64 *impl_pmds;
	unsigned int i;
	unsigned int max_pmd;

	max_pmd = pfm_pmu_conf->max_pmd;
	ovfl_mask = pfm_pmu_conf->ovfl_mask;
	impl_pmds = pfm_pmu_conf->impl_pmds;
	pmds = set->view->set_pmds;

	/*
	 * must restore all pmds to avoid leaking
	 * information to user.
	 */
	for (i = 0; i < max_pmd; i++) {

		if (pfm_bv_isset(impl_pmds, i) == 0)
			continue;

		val = pmds[i];

		/*
		 * set upper bits for counter to ensure
		 * overflow will trigger
		 */
		val &= ovfl_mask;

		pfm_arch_write_pmd(ctx, i, val);
	}
}

/*
 * function called from pfm_switch_sets(), pfm_context_load_thread(),
 * pfm_context_load_sys(), pfm_ctxsw(), pfm_switch_sets()
 * context is locked. Interrupts are masked. set cannot be NULL.
 * Access to the PMU is guaranteed.
 *
 * function must restore all PMC registers from set, if needed.
 */
void pfm_arch_restore_pmcs(struct pfm_context *ctx, struct pfm_event_set *set)
{
	u64 *impl_pmcs;
	unsigned int i, max_pmc;

	max_pmc = pfm_pmu_conf->max_pmc;
	impl_pmcs = pfm_pmu_conf->impl_pmcs;

	/*
	 * - by default no PMCS measures anything
	 * - on ctxswout, all used PMCs are disabled (cccr enable bit cleared)
	 * hence when masked we do not need to restore anything
	 */
	if (ctx->state == PFM_CTX_MASKED || ctx->flags.started == 0)
		return;

	/*
	 * restore all pmcs
	 */
	for (i = 0; i < max_pmc; i++)
		if (pfm_bv_isset(impl_pmcs, i))
			pfm_arch_write_pmc(ctx, i, set->pmcs[i]);
}

/*
 * called from __pfm_interrupt_handler(). ctx is not NULL.
 * ctx is locked. PMU interrupt is masked.
 *
 * must stop all monitoring to ensure handler has consistent view.
 * must collect overflowed PMDs bitmask  into povfls_pmds and
 * npend_ovfls. If no interrupt detected then npend_ovfls
 * must be set to zero.
 */
void pfm_arch_intr_freeze_pmu(struct pfm_context *ctx)
{
	/*
	 * if pfm_pmu_conf is NULL then ctx is NULL
	 * so we are covered for both here.
	 */
	if (pfm_pmu_conf == NULL)
		return;

	pfm_stop_active(current, ctx, ctx->active_set);
}

/*
 * unfreeze PMU from pfm_do_interrupt_handler()
 * ctx may be NULL for spurious
 */
void pfm_arch_intr_unfreeze_pmu(struct pfm_context *ctx)
{
	if (ctx == NULL)
		return;
	pfm_arch_restore_pmcs(ctx, ctx->active_set);
}

void pfm_arch_mask_monitoring(struct pfm_context *ctx)
{
	/*
	 * on ppc64 masking/unmasking uses start/stop
	 * mechanism
	 */
  	pfm_arch_stop(current, ctx, ctx->active_set);
}

void pfm_arch_unmask_monitoring(struct pfm_context *ctx)
{
	/*
	 * on ppc64 masking/unmasking uses start/stop
	 * mechanism
	 */
	__pfm_arch_start(current, ctx, ctx->active_set);
}

char *pfm_arch_get_pmu_module_name(void)
{
  unsigned int pvr = mfspr(SPRN_PVR);

  switch (PVR_VER(pvr))
    {
    case 0x0004: /* 604 */
    case 0x0009: /* 604e;  */
    case 0x000A: /* 604ev */
    case 0x0008: /* 750/740 */
    case 0x7000: /* 750FX */
    case 0x7001:
    case 0x7002: /* 750GX */
    case 0x000C: /* 7400 */
    case 0x800C: /* 7410 */
    case 0x8000: /* 7451/7441 */
    case 0x8001: /* 7455/7445 */
    case 0x8002: /* 7457/7447 */
    case 0x8003: /* 7447A */
    case 0x8004: /* 7448 */
      return("perfmon_ppc32");
    case PV_POWER4:
    case PV_POWER4p:
      return "perfmon_power4";
    case PV_POWER5:
    case PV_POWER5p:
      return "perfmon_power5";
    case PV_970:
    case PV_970FX:
    case PV_970MP:
      return "perfmon_ppc970";
    case PV_BE:
      return "perfmon_cell";
    }
  return NULL;
}
