/*
 * Copyright (c) 2005 David Gibson, IBM Corporation.
 *
 * Based on other versions:
 * Copyright (c) 2005 Hewlett-Packard Development Company, L.P.
 * Contributed by Stephane Eranian <eranian@hpl.hp.com>
 *
 * This file contains ppc64 specific definitions for the perfmon
 * interface.
 *
 * This file MUST never be included directly. Use linux/perfmon.h.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA
  */
#ifndef _ASM_PPC64_PERFMON_H_
#define _ASM_PPC64_PERFMON_H_

#ifdef __KERNEL__

#include <asm/pmc.h>

enum ppc32_pmu_type {
	PM_NONE,
	PM_604,
	PM_604e,
	PM_750,	/* XXX: Minor event set diffs between IBM and Moto. */
	PM_7400,
	PM_7450,
};

struct pfm_arch_pmu_info {
	enum ppc32_pmu_type pmu_style;
};

#if defined(CONFIG_PPC32)
#define PFM_ARCH_PMD_STK_ARG	6 /* conservative value */
#define PFM_ARCH_PMC_STK_ARG	6 /* conservative value */
#else
#define PFM_ARCH_PMD_STK_ARG	8 /* conservative value */
#define PFM_ARCH_PMC_STK_ARG	8 /* conservative value */
#endif

static inline void pfm_arch_resend_irq(void)
{}

static inline void pfm_arch_serialize(void)
{}


static inline void pfm_arch_unfreeze_pmu(void)
{}

static inline void pfm_arch_write_pmc(struct pfm_context *ctx,
				      unsigned int cnum,
				      u64 value)
{
  /*
   * we only write to the actual register when monitoring is
   * active (pfm_start was issued)
   */
  if (ctx && (ctx->flags.started == 0)) return;

	switch (cnum) {
	case 0:
		mtspr(SPRN_MMCR0, value);
		break;
	case 1:
		mtspr(SPRN_MMCR1, value);
		break;
	case 2:
#if defined(CONFIG_PPC32)
	  mtspr(SPRN_MMCR2, value);
#else
	  mtspr(SPRN_MMCRA, value);
#endif
	  break;
	default:
		BUG();
	}
}

static inline void pfm_arch_write_pmd(struct pfm_context *ctx,
				      unsigned int cnum, u64 value)
{
  value &= pfm_pmu_conf->ovfl_mask;

	switch (cnum) {
	case 0:
		mtspr(SPRN_PMC1, value);
		break;
	case 1:
		mtspr(SPRN_PMC2, value);
		break;
	case 2:
		mtspr(SPRN_PMC3, value);
		break;
	case 3:
		mtspr(SPRN_PMC4, value);
		break;
	case 4:
		mtspr(SPRN_PMC5, value);
		break;
	case 5:
		mtspr(SPRN_PMC6, value);
		break;
#if defined(CONFIG_PPC64)
	case 6:
		mtspr(SPRN_PMC7, value);
		break;
	case 7:
		mtspr(SPRN_PMC8, value);
		break;
#endif
	default:
		BUG();
	}
}

static inline u64 pfm_arch_read_pmd(struct pfm_context *ctx, unsigned int cnum)
{
	switch (cnum) {
	case 0:
		return mfspr(SPRN_PMC1);
		break;
	case 1:
		return mfspr(SPRN_PMC2);
		break;
	case 2:
		return mfspr(SPRN_PMC3);
		break;
	case 3:
		return mfspr(SPRN_PMC4);
		break;
	case 4:
		return mfspr(SPRN_PMC5);
		break;
	case 5:
		return mfspr(SPRN_PMC6);
		break;
#if defined(CONFIG_PPC64)
	case 6:
		return mfspr(SPRN_PMC7);
		break;
	case 7:
		return mfspr(SPRN_PMC8);
		break;
#endif
	default:
		BUG();
		return 0;
	}
}

static inline u64 pfm_arch_read_pmc(struct pfm_context *ctx, unsigned int cnum)
{
	switch (cnum) {
	case 0:
		return mfspr(SPRN_MMCR0);
		break;
	case 1:
		return mfspr(SPRN_MMCR1);
		break;
	case 2:
#if defined(CONFIG_PPC32)
	  return mfspr(SPRN_MMCRA);
#else
	  return mfspr(SPRN_MMCRA);
#endif
	  break;
	default:
		BUG();
		return 0;
	}
}

/*
 * on som PMU models, the upper bits of a counter must be set in order
 * for the overflow interrupt to happen. On overflow, the counter
 * has wrapped around, and the upper bits are now cleared. This
 * function set them back.
 *
 * The current version loses whatever is remaining in the counter,
 * which is usually not zero but has a small count. In order not
 * to loose this count, we do a read-modify-write to set the upper
 * bits while preserving the low-order bits. This is slow but
 * works.
 */
static inline void pfm_arch_ovfl_reset_pmd(struct pfm_context *ctx,
					   unsigned int cnum)
{
  u64 val;
  val = pfm_arch_read_pmd(ctx, cnum);
  /* This masks out overflow bit 31 */
  pfm_arch_write_pmd(ctx, cnum, val);
}

/*
 * At certain points, perfmon needs to know if monitoring has been
 * explicitely started/stopped by user via pfm_start/pfm_stop. The
 * information is tracked in flags.started. However on certain
 * architectures, it may be possible to start/stop directly from
 * user level with a single assembly instruction bypassing
 * the kernel. This function must be used to determine by
 * an arch-specific mean if monitoring is actually started/stopped.
 * If there is no other way but to go through pfm_start/pfm_stop
 * then this function can simply return 0
 */
static inline int pfm_arch_is_active(struct pfm_context *ctx)
{
	return 0;
}

static inline void pfm_arch_ctxswout_sys(struct task_struct *task,
		           		 struct pfm_context *ctx,
					 struct pfm_event_set *set)
{
}

static inline void pfm_arch_ctxswin_sys(struct task_struct *task,
                       struct pfm_context *ctx, struct pfm_event_set *set)
{}

static inline void pfm_arch_ctxswin_thread(struct task_struct *task,
                       struct pfm_context *ctx, struct pfm_event_set *set)
{}

int  pfm_arch_is_monitoring_active(struct pfm_context *ctx);
int  pfm_arch_ctxswout_thread(struct task_struct *task, struct pfm_context *ctx,
			      struct pfm_event_set *set);
void pfm_arch_stop(struct task_struct *task, struct pfm_context *ctx,
			  struct pfm_event_set *set);
void pfm_arch_start(struct task_struct *task, struct pfm_context *ctx,
			   struct pfm_event_set *set);
void pfm_arch_restore_pmds(struct pfm_context *ctx, struct pfm_event_set *set);
void pfm_arch_restore_pmcs(struct pfm_context *ctx, struct pfm_event_set *set);
void pfm_arch_intr_freeze_pmu(struct pfm_context *ctx);
void pfm_arch_intr_unfreeze_pmu(struct pfm_context *ctx);
char *pfm_arch_get_pmu_module_name(void);
void pfm_arch_mask_monitoring(struct pfm_context *ctx);
void pfm_arch_unmask_monitoring(struct pfm_context *ctx);

static inline int pfm_arch_pmu_config_init(struct pfm_pmu_config *cfg)
{
	return 0;
}

static inline int pfm_arch_context_initialize(struct pfm_context *ctx,
					       u32 ctx_flags)
{
	return 0;
}

void pfm_arch_unload_context(struct pfm_context *ctx, struct task_struct *task);

static inline int pfm_arch_reserve_session(struct pfm_sessions *session,
					   struct pfm_context *ctx,
					   unsigned int cpu)
{
	return 0;
}

static inline void pfm_arch_release_session(struct pfm_sessions *session,
					    struct pfm_context *ctx,
					    u32 cpu)
{}

/*
 * function called from pfm_setfl_sane(). Context is locked
 * and interrupts are masked.
 * The value of flags is the value of ctx_flags as passed by
 * user.
 *
 * function must check arch-specific set flags.
 * Return:
 * 	1 when flags are valid
 *      0 on error
 */
static inline int pfm_arch_setfl_sane(struct pfm_context *ctx, u32 flags)
{
	return 0;
}

static inline void pfm_arch_stop_thread(struct task_struct *task,
					struct pfm_context *ctx,
		     			struct pfm_event_set *set)
{
	pfm_arch_stop(task, ctx, set);
}

static inline void pfm_arch_start_thread(struct task_struct *task,
					 struct pfm_context *ctx,
					 struct pfm_event_set *set)
{
	pfm_arch_start(task, ctx, set);
}

static inline void pfm_arch_stop_sys(struct task_struct *task,
				     struct pfm_context *ctx,
				     struct pfm_event_set *set)
{
	pfm_arch_stop(task, ctx, set);
}

static inline void pfm_arch_start_sys(struct task_struct *task,
				      struct pfm_context *ctx,
				      struct pfm_event_set *set)
{
	pfm_arch_start(task, ctx, set);
}

static inline void pfm_arch_show_session(struct seq_file *m)
{}

static inline int pfm_arch_init(void)
{
	return 0;
}

static inline void pfm_arch_init_percpu(void)
{}

static inline int pfm_arch_load_context(struct pfm_context *ctx,
					struct task_struct *task)
{
	return 0;
}

struct pfm_arch_context {
	/* empty */
};

#define PFM_ARCH_CTX_SIZE	sizeof(struct pfm_arch_context)

#endif /* __KERNEL__ */
#endif /* _ASM_PPC64_PERFMON_H_ */
