/* concurrent.c
 * based on self.c from libpfm examples
 *
 * Copyright (c) 2002-2006 Hewlett-Packard Development Company, L.P.
 * Contributed by Stephane Eranian <eranian@hpl.hp.com>
 * Adapted by Will Cohen <wcohen@redhat.com>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * This file is tests libpfm, a performance monitoring support library for
 * applications on Linux.
 */

#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/wait.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <math.h>

#include <perfmon/pfmlib.h>
#include <perfmon/perfmon.h>

#include "detect_pmcs.h"

#define NUM_PMCS PFMLIB_MAX_PMCS
#define NUM_PMDS PFMLIB_MAX_PMDS

#include "events.h"

/* arguments passed in 
   $1 number of trigger_events per loop
   $2 number of trigger_event loops to run
   $3 percentage error allowed
*/

int count=DEFAULT_TRIGGER_COUNT;
int iterations=DEFAULT_ITERATIONS;
double allowed_error=DEFAULT_ERROR;

void parse_args(int argc, char *argv[])
{
	switch(argc){
	case 4:
		allowed_error=atof(argv[3]);
	case 3:
		iterations=atoi(argv[2]);
	case 2:
		count=atoi(argv[1]);
	default:
		break;
	}
}

static void fatal_error(char *fmt,...) __attribute__((noreturn));

static void
fatal_error(char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);

	exit(1);
}

int setup_pfmon(pfmlib_input_param_t *inp,
		pfmlib_output_param_t *outp,
		pfarg_pmd_t pd[NUM_PMDS],
		pfarg_pmc_t pc[NUM_PMCS])
{
	char **p;
	unsigned int i;
	int ret, ctx_fd;
	pfarg_ctx_t ctx;
	pfmlib_options_t pfmlib_options;
	unsigned int num_counters;
	size_t len;
	char *name;

	pfm_get_max_event_name_len(&len);
	name = malloc(len+1);
	if (!name)
		fatal_error("cannot allocate event name buffer\n");

	pfm_get_num_counters(&num_counters);

	/*
	 * pass options to library (optional)
	 */
	memset(&pfmlib_options, 0, sizeof(pfmlib_options));
#if 0
	pfmlib_options.pfm_debug   = 0; /* set to 1 for debug */
	pfmlib_options.pfm_verbose = 1; /* set to 1 for verbose */
#endif
	pfm_set_options(&pfmlib_options);

	memset(pd, 0, sizeof(pd[0])*NUM_PMDS);
	memset(pc, 0, sizeof(pc[0])*NUM_PMCS);
	memset(&ctx, 0, sizeof(ctx));

	/*
	 * prepare parameters to library.
	 */
	memset(inp,0, sizeof(*inp));
	memset(outp,0, sizeof(*outp));

	/* select the events to measure */
	p = &(event_names[0]);
	for (i=0; *p ; i++, p++) {
		ret = pfm_find_full_event(*p, &(inp->pfp_events[i]));
		if (ret != PFMLIB_SUCCESS)
			fatal_error("event %s: %s\n", *p, pfm_strerror(ret));
	}

	/*
	 * set the default privilege mode for all counters:
	 * 	PFM_PLM3 : user level only
	 */
	inp->pfp_dfl_plm   = PFM_PLM3;

	if (i > num_counters) {
		i = num_counters;
		printf("too many events provided (max=%d events), using first %d event(s)\n", num_counters, i);
	}

	/*
	 * how many counters we use
	 */
	inp->pfp_event_count = i;

	/*
	 * now create a new context, per process context.
	 * This just creates a new context with some initial state, it is not
	 * active nor attached to any process.
	 */
	ctx_fd = pfm_create_context(&ctx, NULL, NULL, 0);
	if (ctx_fd == -1) {
		if (errno == ENOSYS)
			fatal_error("Your kernel does not have performance monitoring support!\n");
		fatal_error("Can't create PFM context %s\n", strerror(errno));
	}

	/*
	 * build the pfp_unavail_pmcs bitmask by looking
	 * at what perfmon has available. It is not always
	 * the case that all PMU registers are actually available
	 * to applications. For instance, on IA-32 platforms, some
	 * registers may be reserved for the NMI watchdog timer.
	 *
	 * With this bitmap, the library knows which registers NOT to
	 * use. Of source, it is possible that no valid assignement may
	 * be possible if certina PMU registers  are not available.
	 */
	detect_unavail_pmcs(ctx_fd, &(inp->pfp_unavail_pmcs));

	/*
	 * let the library figure out the values for the PMCS
	 */
	if ((ret=pfm_dispatch_events(inp, NULL, outp, NULL)) != PFMLIB_SUCCESS)
		fatal_error("cannot configure events: %s\n", pfm_strerror(ret));

	/*
	 * Now prepare the argument to initialize the PMDs and PMCS.
	 * We use pfp_pmc_count to determine the number of PMC to intialize.
	 * We use pfp_pmd_count to determine the number of PMD to initialize.
	 * Some events/features may cause extra PMCs to be used, leading to:
	 * 	- pfp_pmc_count may be >= pfp_event_count
	 * 	- pfp_pmd_count may be >= pfp_event_count
	 */
	for (i=0; i < outp->pfp_pmc_count; i++) {
		pc[i].reg_num   = outp->pfp_pmcs[i].reg_num;
		pc[i].reg_value = outp->pfp_pmcs[i].reg_value;
	}

	for (i=0; i < outp->pfp_pmd_count; i++) {
		pd[i].reg_num   = outp->pfp_pmds[i].reg_num;
	}

	/*
	 * Now program the registers
	 */
	if (pfm_write_pmcs(ctx_fd, pc, outp->pfp_pmc_count))
		fatal_error("pfm_write_pmcs error errno %d\n",errno);

	if (pfm_write_pmds(ctx_fd, pd, outp->pfp_pmd_count))
		fatal_error("pfm_write_pmds error errno %d\n",errno);

	return ctx_fd;
}

int run_experiment(int slave)
{
	int ret, j=slave;
	int fail=0;
	/* figure out arguments passed to run */
	double computed_err=0.0;
	int events_counted=0;
	pfmlib_input_param_t inp;
	pfmlib_output_param_t outp;
	pfarg_pmd_t pd[NUM_PMDS];
	pfarg_pmc_t pc[NUM_PMCS];
	pfarg_load_t load_args;
	int ctx_fd;
	int pmd_count;

	/*
	 * Initialize pfm library (required before we can use it)
	 */
	ret = pfm_initialize();
	if (ret != PFMLIB_SUCCESS)
		fatal_error("Cannot initialize library: %s\n", pfm_strerror(ret));

	{
		/* setup pfmon */
		ctx_fd=setup_pfmon(&inp, &outp, pd, pc);

		/* now we load (i.e., attach) the context to ourself */
		memset(&load_args, 0, sizeof(load_args));
		load_args.load_pid = getpid();
		if (pfm_load_context(ctx_fd, &load_args))
			fatal_error("pfm_load_context error errno %d\n",errno);

		/* start pfmon */
		if (pfm_start(ctx_fd, NULL))
			fatal_error("pfm_start error errno %d\n",errno);

		/* actually get some samples */
		trigger_event(count);

		/* stop */
		if (pfm_stop(ctx_fd))
			fatal_error("pfm_stop error errno %d\n",errno);

		/* read value */
		if (pfm_read_pmds(ctx_fd, pd, inp.pfp_event_count))
			fatal_error( "pfm_read_pmds error errno %d\n",errno);

		/* remove pfmon */
		close(ctx_fd);
		/* compute difference between actual and expected */
		events_counted = pd[0].reg_value;

		/* check the measurements */
		computed_err = fabs((double) (events_counted - count))/count;
		if (computed_err > allowed_error) {
		  fprintf(stdout, "FAIL on slave %d computed error is %f\n",
			  j, computed_err);
		  fail = 1;
		}
	}
	if ( fail==0) {
		return EXIT_SUCCESS;
	} else {
		return EXIT_FAILURE;
	}
}

int main(int argc, char *argv[])
{
	int j;
	pid_t *pids_array;
	int fail=0;

	parse_args(argc, argv);

	pids_array=calloc(iterations, sizeof(pid_t));

	for (j=0; j<iterations; j++)
	{
		pid_t pid = fork();
		if (pid>=0) {
			if (pid==0) {
				exit(run_experiment(j));
			} else {
				pids_array[j] = pid;
			}
		} else {
			/* problem forking */
			fail = 1;
		}
	}
	/* wait for each thread to complete */
	for (j=0; j<iterations; j++)
	{
		int status;
		pid_t pid = wait3(&status, WNOHANG, NULL);
		if (WEXITSTATUS(status) != EXIT_SUCCESS)
			fail = 1;
	}

	if ( fail==0) {
		fprintf(stdout, "PASS\n");
		return EXIT_SUCCESS;
	} else {
		return EXIT_FAILURE;
	}
}
