#include <iostream>
#include <petscksp.h>
#include <petscvec.h>

// Tiny code to read a matrix and solution vector,
// create the constant null space and test it against the matrix.

// Using GCC compiler it easy compiles with:
// g++ -o testNullSpace -I/path/to/mpi/include/ -I/path/to/petsc/include testNullSpace.cpp -L/path/to/petsc/lib/ -lpetsc_real

int main(int argc, char *argv[])
{
    std::string help = "None";
    PetscInitialize(&argc, &argv, 0, help.data());

    // Viewer for reading mat and vec for nullspace
    PetscViewer v;
    PetscViewerCreate(PETSC_COMM_WORLD,&v);
    PetscViewerSetType(v,PETSCVIEWERBINARY);
    PetscViewerFileSetMode(v,FILE_MODE_READ);

    // Read matrix
    Mat matrix;
    MatCreate(PETSC_COMM_WORLD, &matrix);
    PetscViewerFileSetName(v,"./A.txt");
    MatLoad(matrix, v);

    // Read rhs vec and build the solution vec by vec duplication
    Vec rhs;
    VecCreate(PETSC_COMM_WORLD, &rhs);
    PetscViewerFileSetName(v,"./rhs.txt");
    VecLoad(rhs, v);

    Vec solution;
    VecCreate(PETSC_COMM_WORLD, &solution);
    VecDuplicate(rhs, &solution);
    VecSet(solution, 0.0);

    // Create the nullspace, by setting the constant (true) or by providing the constant vector (false)
    MatNullSpace nullspace;
    MatNullSpaceCreate(PETSC_COMM_WORLD, PETSC_TRUE, 0, nullptr, &nullspace);
    // Set the nullspace
    MatSetNullSpace(matrix, nullspace);

    // Test the null space
    PetscBool isNullSpaceValid;
    MatNullSpaceTest(nullspace, matrix, &isNullSpaceValid);
    if (!isNullSpaceValid) {
        std::cout << "Null space is not valid!" << std::endl;
    } else {
        std::cout << "Null space is valid!" << std::endl;
    }

    // Create ksp and set operators
    KSP ksp;
    KSPCreate(PETSC_COMM_WORLD, &ksp);
    KSPSetOperators(ksp, matrix, matrix);
    KSPSetType(ksp, KSPFGMRES);
    KSPGMRESSetRestart(ksp, 1000);
    KSPSetFromOptions(ksp);
    KSPSetUp(ksp);

    //Solve
    KSPSolve(ksp, rhs, solution);

    // Print solution
    PetscViewer vv;
    PetscViewerCreate(PETSC_COMM_WORLD,&vv);
    PetscViewerSetType(vv,PETSCVIEWERASCII);
    PetscViewerFileSetMode(vv,FILE_MODE_WRITE);
    PC pc;
    KSPGetPC(ksp, &pc);
    PCType preconditionerType;
    PCGetType(pc, &preconditionerType);
    std::stringstream ss;
    ss << "./solution_" << preconditionerType << ".txt";
    PetscViewerFileSetName(vv,ss.str().c_str());
    VecView(solution, vv);
    double norm1, norm2, mean;
    VecNorm(solution, NORM_1, &norm1);
    VecNorm(solution, NORM_2, &norm2);
    VecSum(solution, &mean);
    PetscInt size;
    VecGetSize(solution, &size);
    std::cout << "Solution norm 1 = " << norm1 << std::endl;
    std::cout << "Solution norm 2 = " << norm2 << std::endl;
    std::cout << "Solution mean = " << mean / size << std::endl;

    // Free memory by destroying PETSc structure
    PetscViewerDestroy(&v);
    PetscViewerDestroy(&v);
    MatNullSpaceDestroy(&nullspace);
    MatDestroy(&matrix);
    VecDestroy(&rhs);
    VecDestroy(&solution);

    PetscFinalize();
}
