package com.polarislabs.pgpool;

import com.mchange.v2.c3p0.ComboPooledDataSource;
import java.beans.PropertyVetoException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;

/**
 * Creates a workload which cause PGPool to send executes to nodes that were not
 * bound when running in MASTER-SLAVE && LOAD_BALANCE
 */
public class PoolBuster {
    public static void main(String[] args) throws PropertyVetoException, SQLException, InterruptedException {
        if (args.length < 3) {
            System.out.println("Usage: PoolBuster <jdbc url> <username> <password>");
        }

        // Create a C3P0 connection pool
        ComboPooledDataSource poolDs = new ComboPooledDataSource();
        poolDs.setDriverClass("org.postgresql.Driver");
        poolDs.setJdbcUrl(args[0]);
        poolDs.setUser(args[1]);
        poolDs.setPassword(args[2]);

        poolDs.setMinPoolSize(5);
        poolDs.setAcquireIncrement(5);
        poolDs.setMaxPoolSize(25);
        poolDs.setMaxStatementsPerConnection(10);

        Connection connection = poolDs.getConnection();

        Statement s = connection.createStatement();
        s.execute("DROP TABLE IF EXISTS public.test");
        s.close();
        
        s = connection.createStatement();
        s.execute("CREATE TABLE public.test (id serial, value int4)");
        s.close();

        connection.close();

        // Create a thread that inserts.
        Thread inserter = new Thread(new Inserter(poolDs));
        Thread selector = new Thread(new Selector(poolDs));
        
        inserter.start();
        selector.start();
        
        inserter.join();
        selector.join();
    }

    private static class Inserter implements Runnable {
        ComboPooledDataSource pool;

        public Inserter(ComboPooledDataSource pool) {
            this.pool = pool;
        }

        public void run() {
            for (int i = 0; i < 3000; i++) {
                try {
                    Connection c = pool.getConnection();

                    PreparedStatement ps = c.prepareStatement("INSERT INTO public.test (value) VALUES(?);");
                    ps.setInt(1, i);
                    ps.execute();

                    ps.close();
                    c.close();
                } catch (SQLException ex) {
                    ex.printStackTrace();
                }
            }
        }
    }

    private static class Selector implements Runnable {
        ComboPooledDataSource pool;

        public Selector(ComboPooledDataSource pool) {
            this.pool = pool;
        }

        public void run() {
            for (int i = 0; i < 3000; i++) {
                try {
                    Connection c = pool.getConnection();
                    c.setAutoCommit(false);

                    PreparedStatement txStart = c.prepareStatement("BEGIN");
                    PreparedStatement txCommit = c.prepareStatement("COMMIT");

                    PreparedStatement select = c.prepareStatement("SELECT * FROM public.test WHERE id = ?");
                    
                    // Run this query outside of a TX until it gets compiled server-side.
                    // This query may or may not be sent to the standby.
                    for (int j = 0; j <= 5; j++) {
                        select.setInt(1, i);
                        ResultSet rs = select.executeQuery();
                        rs.close();
                    }

                    // Start a transaction. If the select statement was sent to
                    // the standby, then the next invocation should fail.
                    txStart.execute();

                    select.setInt(1, i);
                    ResultSet rs = select.executeQuery();
                    rs.close();

                    txCommit.execute();

                    select.close();
                    txStart.close();
                    txCommit.close();
                    c.close();
                } catch (SQLException ex) {
                    ex.printStackTrace();
                }
            }
        }
    }
}
