/*
 * cursor.c
 *
 * Test bed for hold-able cursor bug - a row is emitted more than once.
 */
#include <stdio.h>
#include <stdlib.h>
#include "libpq-fe.h"

/* Max obj_id is this value */
#define		MAXID			(10000000)

static void
exit_nicely(PGconn *conn)
{
	PQfinish(conn);
	exit(1);
}

int
main(int argc, char **argv)
{
	const char *conninfo;
	PGconn	   *conn;
	PGresult   *res;
	int			i;
	int			*seenid;   
	int			id;


	seenid = (int *)malloc(MAXID * sizeof(int));
	if (seenid == NULL)
	{
		fprintf(stderr, "Failed to malloc seenid buffer");
		exit(1);
	}
		
	/*
	 * If the user supplies a parameter on the command line, use it as the
	 * conninfo string; otherwise default to setting dbname=postgres and using
	 * environment variables or defaults for all other connection parameters.
	 */
	if (argc > 1)
		conninfo = argv[1];
	else
		conninfo = "dbname = sort";

	/* Make a connection to the database */
	conn = PQconnectdb(conninfo);

	/* Check to see that the backend connection was successfully made */
	if (PQstatus(conn) != CONNECTION_OK)
	{
		fprintf(stderr, "Connection to database failed: %s",
				PQerrorMessage(conn));
		exit_nicely(conn);
	}

	/*
	 * initialize the seen array
	 */
	for (i = 0; i < MAXID; i++) 
	{
		seenid[i] = 0;
	}

	/* Start a transaction block */
	res = PQexec(conn, "BEGIN");
	if (PQresultStatus(res) != PGRES_COMMAND_OK)
	{
		fprintf(stderr, "BEGIN command failed: %s", PQerrorMessage(conn));
		PQclear(res);
		exit_nicely(conn);
	}

	/*
	 * Should PQclear PGresult whenever it is no longer needed to avoid memory
	 * leaks
	 */
	PQclear(res);

	/*
	 * Fetch rows from pg_database, the system catalog of databases
	 */
	res = PQexec(conn, "DECLARE myportal CURSOR WITH HOLD FOR SELECT * FROM obj");
	if (PQresultStatus(res) != PGRES_COMMAND_OK)
	{
		fprintf(stderr, "DECLARE CURSOR failed: %s", PQerrorMessage(conn));
		PQclear(res);
		exit_nicely(conn);
	}
	PQclear(res);

	while (1)
	{
		res = PQexec(conn, "FETCH 1000 IN myportal");
		if (PQresultStatus(res) != PGRES_TUPLES_OK ||
		    PQntuples(res) == 0)
		{
			fprintf(stderr, "FETCH failed or got no rows...: %s", PQerrorMessage(conn));
			PQclear(res);
			exit_nicely(conn);
		}


		/* next, print out the 1st field for each row */
		for (i = 0; i < PQntuples(res); i++)
		{
			id = atol(PQgetvalue(res, i, 0));
			(seenid[id])++;
			//printf("%d\n", id);
		
			if (seenid[id] > 1)
			{
				fprintf(stderr, "seen this id %d before - aborting\n", id);
				PQclear(res);
				exit_nicely(conn);
			}
		}

		PQclear(res);

		/* Commit & restart a transaction */
		res = PQexec(conn, "COMMIT");
		PQclear(res);
		res = PQexec(conn, "BEGIN");
		PQclear(res);
	}
	

	/* close the portal ... we don't bother to check for errors ... */
	res = PQexec(conn, "CLOSE myportal");
	PQclear(res);

	/* end the transaction */
	res = PQexec(conn, "END");
	PQclear(res);

	/* close the connection to the database and cleanup */
	PQfinish(conn);
	free(seenid);

	return 0;
}
