#!/usr/local/bin/perl

use strict;

# Perl routine to generate C/SQL for various operators for postgres
my $debug = 1;

my $name = 'uint_ex';
my $so_name = 'uint';

my @type_order = ();
my %type_order = ();	# Hash for type ordering
my %type_info = ();
my %func_done = ();	# Track which C functions have been generated
my %func_sql = ();	# Track which sql functions have been generated
my %op_info = ();	# Operator information

my $type;
while ($_=<DATA>) {
  next if /^\s*\#/;			# Ignore comments;
  /^type\s+(\w+)/ && ($type = $1) && (push @type_order, $1);	# new type
  next unless $type;
  # Information and conversion
  /^\s+(\w+):\s+/ &&
    do {
      $type_info{$type}{$1} = [split ' ', $'];
      next;
    };
  /^\s+([OBLR])\[(\w+)\]\s+(\w+)\(([^\)]+)\): / && 
    do {
      my $fun = $3;
      $type_info{$type}{$fun}{optype} = $1;
      $type_info{$type}{$fun}{r} = $2;
      $type_info{$type}{$fun}{op} = $4;
      $type_info{$type}{$fun}{v} = [split ' ', $'];
      next;
    };
  # Type information 
  /^op (\S+) \{/ && 
    do {
      my $op = $1;
      while (($_=<DATA>) && !/^\}/) {
	/(\w+):\s*(\S*)/ && ($op_info{$op}{$1} = $2);
      }
    };
}

# Print out all the types
warn join(' ',@type_order),"\n" if $debug;
{ my $i=0;
  %type_order = map {$_=>$i++} @type_order;
}

open C, ">$name.c" || die "Couldn't open $name.c\n";
open H, ">$name.h" || die "Couldn't open $name.h\n";
open SQL, ">$name.source" || die "Couldn't open $name.source\n";


# Go through all the types and generate the functions
foreach my $type (@type_order) {
  # Any conversion functions?
  if (exists $type_info{$type}{conv}) {
    foreach my $type2 (@{$type_info{$type}{conv}}) {
      do_conv_func($type,$type2);
      do_conv_func($type2,$type);
      do_conv_sql($type,$type2);
      do_conv_sql($type2,$type);
    }
  }
  # Do all the operators
  while (my ($f,$h) = each %{$type_info{$type}}) {
    next unless ref $h eq 'HASH' && $h->{op};
    foreach my $type2 (@{$h->{v}}) {
      if ($h->{optype} =~ /[OB]/) {
	do_op_func($f,$h,$type,$type2);
	do_op_sql($f,$h,$type,$type2);
	if ($h->{optype} eq 'B') {
	  do_op_func($f,$h,$type2,$type);
	  do_op_sql($f,$h,$type2,$type);
	}
      } elsif ($h->{optype} eq 'L') {
      }
    }
  }
}

sub do_conv_func {
  my ($from,$to) = @_;
  # Return if we have done this before
  return undef if $func_done{conv}{$from}{$to};
  $func_done{conv}{$from}{$to} = 1;
  my $ft = $type_info{$from}{ctype}[0];
  my $tt = $type_info{$to}{ctype}[0];
  my $uft = uc $ft; 
  my $utt = uc $tt;
  print H "Datum $from\_to_$to(PG_FUNCTION_ARGS);\n";
  print C <<FUNC
    
PG_FUNCTION_INFO_V1($from\_to_$to);

Datum
$from\_to_$to(PG_FUNCTION_ARGS)
\{
	$ft		arg1 = PG_GETARG_$uft(0);
FUNC
  ;

  if (exists $type_info{$to}{range}) {
    print C <<FUNC
	if (arg1 < $type_info{$to}{range}[0])
		elog(ERROR, "$from\_to_$to: '$type_info{$from}{cform}[0]' causes $to underflow", arg1);
	if (arg1 > $type_info{$to}{range}[1])
		elog(ERROR, "$from\_to_$to: '$type_info{$from}{cform}[0]' causes $to overflow", arg1);
FUNC
  ;
  }
  print C <<FUNC
	PG_RETURN_$utt(($tt) arg1);
}
FUNC
  ;
}

sub do_conv_sql {
  my ($from,$to) = @_;
  # Return if we have done this before
  return undef if $func_sql{conv}{$from}{$to};
  $func_sql{conv}{$from}{$to} = 1;
  my $func = "$from\_to_$to";
  print SQL <<FUNC
CREATE FUNCTION $to($from) RETURNS $to
  AS '_OBJWD_/$so_name\_DLSUFFIX_', '$func'
  LANGUAGE 'c';
FUNC
  ;
}

sub do_op_func {
  my ($func, $h, $left, $right) = @_;

  next if $func_done{$left}{$right}{$func};
  $func_done{$left}{$right}{$func} = 1;

  # Get C-types for left and right
  my $leftc = $type_info{$left}{ctype}[0];
  my $rightc = $type_info{$right}{ctype}[0];
  # Postgres UC names for these types
  my ($leftp, $rightp) = (uc $leftc, uc $rightc);
  # Figure out the result type
  my $result = $h->{r};
  if ($h->{r} eq 'TYPE') {
    $result = $type_order{$left} > $type_order{$right} ? $left : $right;
  }
  my $resultp = uc $type_info{$result}{ctype}[0];
  print H "Datum $left\_$func\_$right(PG_FUNCTION_ARGS);\n";
  print C <<FUNC

PG_FUNCTION_INFO_V1($left\_$func\_$right);

Datum
$left\_$func\_$right(PG_FUNCTION_ARGS)
{
	$leftc		arg1 = PG_GETARG_$leftp(0);
	$rightc		arg2 = PG_GETARG_$rightp(1);
	PG_RETURN_$resultp(arg1 $h->{op} arg2);
}
FUNC
  ;
}

sub do_op_sql {
  my ($func, $h, $left, $right) = @_;

  next if $func_sql{$left}{$right}{$func};
  $func_sql{$left}{$right}{$func} = 1;

  # Get C-types for left and right
  my $leftc = $type_info{$left}{ctype}[0];
  my $rightc = $type_info{$right}{ctype}[0];
  # Postgres UC names for these types
  my ($leftp, $rightp) = (uc $leftc, uc $rightc);
  # Figure out the result type
  my $result = $h->{r};
  if ($h->{r} eq 'TYPE') {
    $result = $type_order{$left} > $type_order{$right} ? $left : $right;
  }
  my $resultp = uc $type_info{$result}{ctype}[0];

  print SQL<<FUNC

CREATE FUNCTION $left\_$func\_$right($left,$right) RETURNS $result
  AS '_OBJWD_/$so_name\_DLSUFFIX_'
  LANGUAGE 'c';
FUNC
  ;
  my $op = "
CREATE OPERATOR $h->{op} (
  leftarg = $left,
  rightarg = $right,
  procedure = $left\_$func\_$right";
  
  # What else do we need to add to this operator?
  if (my $o=$op_info{$h->{op}}) {
    while (my ($left,$right) = each %$o) {
      $op .= ",\n  $left";
      $op .= " = $right" if $right;
    }
  }

  $op .= "\n);\n";

  print SQL $op;
}

__END__

# The data section defines types/operators and the relationships between them

type bool {
 ctype: bool
 cform: %d
}

type int2 {
 ctype: int16
 cform: %d
 range: SHRT_MIN SHRT_MAX
}

type uint2 {
 ctype: uint16
 cform: %d
 range: MIN_UINT16 MAX_UINT16
 conv: int2 uint2 int4 uint4
 O[bool] lt(<): int2 uint2 int4 uint4 
 O[bool] le(<=): int2 uint2 int4 uint4 
 O[bool] eq(=): int2 uint2 int4 uint4 
 O[bool] ge(>=): int2 uint2 int4 uint4 
 O[bool] gt(>): int2 uint2 int4 uint4 
 B[TYPE] plus(+): int2 uint2 int4 uint4 
 B[TYPE] minus(-): int2 uint2 int4 uint4 
 B[TYPE] mul(*): int2 uint2 int4 uint4 
 B[TYPE] div(/): int2 uint2 int4 uint4 
 B[TYPE] mod(%): int2 uint2 int4 uint4
 O[uint2] and(&): uint2
 O[uint2] or(|): uint2
 O[uint2] xor(|): uint2
 L[uint2] not(~):
 O[uint2] shift_left(<<): int4
 O[uint2] shift_right(<<): int4
}

type int4 {
 ctype: int32
 cform: %d
 range: INT_MIN INT_MAX
};


# Definition of operators
# [Ordered,Binary both ways,Left unary, Right unary]
# Type: fixed type or TYPE = greater type in ordering of types
# Name of function
# Operator
# Types to generate functions for

type uint4 {
 ctype: uint32
 cform: %u
 range: MIN_UINT32 MAX_UINT32
 conv: int2 uint2 int4 int8 float8
 O[bool] lt(<): int2 uint2 int4 uint4 int8 float8
 O[bool] le(<=): int2 uint2 int4 uint4 int8 float8
 O[bool] eq(=): int2 uint2 int4 uint4 int8 float8
 O[bool] ge(>=): int2 uint2 int4 uint4 int8 float8
 O[bool] gt(>): int2 uint2 int4 uint4 int8 float8
 B[TYPE] plus(+): int2 uint2 int4 uint4 int8 float8
 B[TYPE] minus(-): int2 uint2 int4 uint4 int8 float8
 B[TYPE] mul(*): int2 uint2 int4 uint4 int8 float8
 B[TYPE] div(/): int2 uint2 int4 uint4 int8 float8
 B[TYPE] mod(%): int2 uint2 int4 uint4 int8
 O[uint4] and(&): uint4
 O[uint4] or(|): uint4
 O[uint4] xor(|): uint4
 L[uint4] not(~):
 O[uint4] shift_left(<<): int4
 O[uint4] shift_right(<<): int4
}

type int8 {
  ctype: int64
  cform: %ld
}

type float8 {
  ctype: float8
  cform: %f
}


# The remaineder provides some type information for operators
op < {
      negator: >=
      commutator: >
      restrict: scalarltsel
      join: scalarltjoinsel
}

op <= {
      negator: >
      commutator: >=
      restrict: scalarltsel
      join: scalarltjoinsel
}

op = {
      negator: <>
      commutator: =
      restrict: eqsel
      join: eqjoinsel
      hashes:
}

op <> {
      negator: =
      commutator: <>
      restrict: neqsel
      join: neqjoinsel
}

op >= {
      negator: <
      commutator: <=
      restrict: scalargtsel
      join: scalargtjoinsel
}

op > {
      negator: <=
      commutator: <
      restrict: scalargtsel
      join: scalargtjoinsel
}

op * {
      commutator: *
}

op + {
      commutator: +
}

op - {
}

op / {
}

op % {
}

op | {
      commutator: |
}

op # {
      commutator: #
}

op & {
      commutator: &
}



