/*
 * has_priv.c
 *
 * Check for user privileges
 *
 * Copyright (c) Joseph Conway <joe.conway@mail.com>, 2001;
 * Licence: BSD
 *
 */

#include "has_priv.h"

/*
 * has_privilege
 *
 * Check user privileges on a relation
 * Returns bool
 *
 */

PG_FUNCTION_INFO_V1(has_privilege);
Datum
has_privilege(PG_FUNCTION_ARGS)
{
	text		*username_text;
	text		*relname_text;
	text		*priv_type_text;
	char		*relname;
	char		*username;
	char		*priv_type;
	AclMode		mode;
	int32		result;
	HeapTuple	tuple;
	Oid			userid;

	if (PG_ARGISNULL(0) || PG_ARGISNULL(1) || PG_ARGISNULL(2))
	{
		elog(ERROR, "has_privilege: NULL arguments are not permitted");
	}

	username_text = PG_GETARG_TEXT_P(0);
	relname_text = PG_GETARG_TEXT_P(1);
	priv_type_text = PG_GETARG_TEXT_P(2);

	/* 
	 * Convert 'text' pattern to null-terminated string
	 */

	relname = DatumGetCString(DirectFunctionCall1(textout, PointerGetDatum(relname_text)));
	username = DatumGetCString(DirectFunctionCall1(textout, PointerGetDatum(username_text)));
	priv_type = str_toupper(DatumGetCString(DirectFunctionCall1(textout, PointerGetDatum(priv_type_text))));

	/*
	 * Lookup userid based on username
	 */

	tuple = SearchSysCache(SHADOWNAME, NameGetDatum(username), 0, 0, 0);
	if (!HeapTupleIsValid(tuple)) {
		elog(ERROR, "has_privilege: invalid user name %s", (char *) username);
	}

	userid = ((Form_pg_shadow) GETSTRUCT(tuple))->usesysid;

	ReleaseSysCache(tuple);

	/*
	 * Verify relname is valid
	 */

	tuple = SearchSysCache(RELNAME, NameGetDatum(relname), 0, 0, 0);
	if (!HeapTupleIsValid(tuple)) {
		elog(ERROR, "has_privilege: invalid relation name %s", (char *) relname);
	}

	ReleaseSysCache(tuple);


	/*
	 * select = r = (ACL_RD)
	 * update = w (ACL_WR)
	 * delete = w (ACL_WR)
	 * rule = R (ACL_RU)
	 * insert = a (ACL_AP) or w (ACL_WR)
	 *
	 */


	if (!strcmp(priv_type, PRIV_INSERT) == 0) {

		if (strcmp(priv_type, PRIV_SELECT) == 0) {

			mode = (AclMode) ACL_RD;

		} else if (strcmp(priv_type, PRIV_UPDATE) == 0) {

			mode = (AclMode) ACL_WR;

		} else if (strcmp(priv_type, PRIV_DELETE) == 0) {

			mode = (AclMode) ACL_WR;

		} else if (strcmp(priv_type, PRIV_RULE) == 0) {

			mode = (AclMode) ACL_RU;

		} else {

			mode = (AclMode) ACL_NO;
			elog(ERROR, "has_privilege: invalid privilege type %s", (char *) priv_type);

		}

		result = pg_aclcheck(relname, userid, mode);

	} else {

		mode = (AclMode) ACL_WR;
		result = pg_aclcheck(relname, userid, mode);

		/*
		 * if write is granted,
		 * no need to check append
		 *
		 */

		if (result != 1) {
			mode = (AclMode) ACL_WR;
			result = pg_aclcheck(relname, userid, mode);
		}

	}

	if (result == 1) {
		PG_RETURN_BOOL(FALSE);
	} else {
		PG_RETURN_BOOL(TRUE);
	}
}


/*
 * Convert string to upper-string
 * Original from backend/utils/adt/formatting.c
 *
 */
static char *
str_toupper(char *buff)
{
	char	   *p_buff = buff;

	while (*p_buff)
	{
		*p_buff = toupper((unsigned char) *p_buff);
		++p_buff;
	}
	return buff;
}
