#include "postgres.h"

#include "access/gist.h"
#include "access/itup.h"
#include "access/nbtree.h"

#include "utils/palloc.h"
#include "utils/geo_decls.h"
#include "utils/elog.h"

typedef Datum (*RDF)(PG_FUNCTION_ARGS);
typedef char * (*BINARY_UNION)(char*, char*, int*);
typedef float (*SIZE_BOX)(char*);

#define WISH_F(a,b,c) (double)( -(double)(((a)-(b))*((a)-(b))*((a)-(b)))*(c) )

typedef struct intkey {
	int4	lower;
	int4	upper;
} INT4KEY; 

typedef struct tskey {
        Timestamp       lower;
        Timestamp       upper;
} TSKEY;

/*
** int4key in/out
*/
PG_FUNCTION_INFO_V1(int4key_in);
PG_FUNCTION_INFO_V1(int4key_out);
Datum	int4key_in(PG_FUNCTION_ARGS);
Datum	int4key_out(PG_FUNCTION_ARGS);

/*
** tskey in/out
*/
PG_FUNCTION_INFO_V1(tskey_in);
PG_FUNCTION_INFO_V1(tskey_out);
Datum	tskey_in(PG_FUNCTION_ARGS);
Datum	tskey_out(PG_FUNCTION_ARGS);

/*
** int4 ops 
*/
PG_FUNCTION_INFO_V1(gint4_compress);
PG_FUNCTION_INFO_V1(gint4_union);
PG_FUNCTION_INFO_V1(gint4_picksplit);
PG_FUNCTION_INFO_V1(gint4_consistent);
PG_FUNCTION_INFO_V1(gint4_penalty);
PG_FUNCTION_INFO_V1(gint4_same);

Datum	gint4_compress(PG_FUNCTION_ARGS);
Datum	gint4_union(PG_FUNCTION_ARGS);
Datum	gint4_picksplit(PG_FUNCTION_ARGS);
Datum	gint4_consistent(PG_FUNCTION_ARGS);
Datum	gint4_penalty(PG_FUNCTION_ARGS);
Datum	gint4_same(PG_FUNCTION_ARGS);

static char * gint4_binary_union(char *r1, char *r2, int *sizep);
static float size_int4key( char *pk );

PG_FUNCTION_INFO_V1(gint4_inter);
Datum gint4_inter(PG_FUNCTION_ARGS);

/*
** timestamp ops
*/
PG_FUNCTION_INFO_V1(gts_compress);
PG_FUNCTION_INFO_V1(gts_union);
PG_FUNCTION_INFO_V1(gts_picksplit);
PG_FUNCTION_INFO_V1(gts_consistent);
PG_FUNCTION_INFO_V1(gts_penalty);
PG_FUNCTION_INFO_V1(gts_same);

Datum   gts_compress(PG_FUNCTION_ARGS);
Datum   gts_union(PG_FUNCTION_ARGS);
Datum   gts_picksplit(PG_FUNCTION_ARGS);
Datum   gts_consistent(PG_FUNCTION_ARGS);
Datum   gts_penalty(PG_FUNCTION_ARGS);
Datum   gts_same(PG_FUNCTION_ARGS);

static char * gts_binary_union(char *r1, char *r2, int *sizep);
static float size_tskey( char *pk );

PG_FUNCTION_INFO_V1(gts_inter);
Datum gts_inter(PG_FUNCTION_ARGS);

/* define for comparison */
#define TSGE( ts1, ts2 ) (DatumGetBool(DirectFunctionCall2( \
        timestamp_ge, \
        PointerGetDatum( ts1 ), \
        PointerGetDatum( ts2 ) \
)))
#define TSGT( ts1, ts2 ) (DatumGetBool(DirectFunctionCall2( \
        timestamp_gt, \
        PointerGetDatum( ts1 ), \
        PointerGetDatum( ts2 ) \
)))
#define TSEQ( ts1, ts2 ) (DatumGetBool(DirectFunctionCall2( \
        timestamp_eq, \
        PointerGetDatum( ts1 ), \
        PointerGetDatum( ts2 ) \
)))
#define TSLT( ts1, ts2 ) (DatumGetBool(DirectFunctionCall2( \
        timestamp_lt, \
        PointerGetDatum( ts1 ), \
        PointerGetDatum( ts2 ) \
)))
#define TSLE( ts1, ts2 ) (DatumGetBool(DirectFunctionCall2( \
        timestamp_le, \
        PointerGetDatum( ts1 ), \
        PointerGetDatum( ts2 ) \
)))

/*
** Common btree-function (for all ops)
*/
static char* btree_union(bytea *entryvec, int *sizep, BINARY_UNION bu);
static float * btree_penalty(GISTENTRY *origentry, GISTENTRY *newentry, 
		float *result, BINARY_UNION bu, SIZE_BOX sb);
static GIST_SPLITVEC * btree_picksplit(bytea *entryvec, GIST_SPLITVEC *v, 
	int keylen, BINARY_UNION bu, RDF interop, SIZE_BOX sb);

PG_FUNCTION_INFO_V1(btree_decompress);
Datum btree_decompress(PG_FUNCTION_ARGS);

/**************************************************
 * int4 ops
 **************************************************/

Datum
gint4_compress(PG_FUNCTION_ARGS)
{
	GISTENTRY *entry=(GISTENTRY*)PG_GETARG_POINTER(0);
	GISTENTRY *retval;

	if ( entry->leafkey) {
		INT4KEY *r = palloc(sizeof(INT4KEY));
		retval = palloc(sizeof(GISTENTRY));
		r->lower = r->upper = (entry->key);
		
		gistentryinit(*retval, PointerGetDatum(r), entry->rel, entry->page,
			entry->offset, sizeof(INT4KEY),FALSE);

	} else {
		retval = entry;
	}
	PG_RETURN_POINTER( retval );
}

Datum 
gint4_consistent(PG_FUNCTION_ARGS)
{
    GISTENTRY *entry = (GISTENTRY*) PG_GETARG_POINTER(0);
    int4 query       = PG_GETARG_INT32(1);
    INT4KEY *kkk= (INT4KEY *)DatumGetPointer(entry->key);
    StrategyNumber strategy = (StrategyNumber) PG_GETARG_UINT16(2);
    bool retval;

    switch(strategy) {
	case BTLessEqualStrategyNumber:
		retval = ( query >= kkk->lower );
		break;
	case BTLessStrategyNumber:
		if (GIST_LEAF(entry)) 
			retval = ( query >  kkk->lower  );
		else 
			retval = ( query >= kkk->lower ); 
		break;
	case BTEqualStrategyNumber:
    		/* in leaf page kkk->lower always = kkk->upper */
		if (GIST_LEAF(entry)) 
			retval = ( query == kkk->lower );
		else 
			retval = ( kkk->lower <= query && query <= kkk->upper ); 
		break;
	case BTGreaterStrategyNumber:
		if (GIST_LEAF(entry)) 
			retval = ( query <  kkk->upper  );
		else 
			retval = ( query <= kkk->upper ); 
		break;
	case BTGreaterEqualStrategyNumber:
		retval = ( query <= kkk->upper );
		break;
	default:
		retval = FALSE;
    }
    PG_RETURN_BOOL(retval);
}

Datum
gint4_union(PG_FUNCTION_ARGS)
{
    PG_RETURN_POINTER( 
	btree_union(
		(bytea*) PG_GETARG_POINTER(0),
		(int*) PG_GETARG_POINTER(1),
		gint4_binary_union
	)
   );
}

Datum
gint4_penalty(PG_FUNCTION_ARGS)
{
    PG_RETURN_POINTER( btree_penalty(
	(GISTENTRY*) PG_GETARG_POINTER(0),
	(GISTENTRY*) PG_GETARG_POINTER(1),
	(float*) PG_GETARG_POINTER(2),
	gint4_binary_union,
	size_int4key
    ) );
}

Datum
gint4_picksplit(PG_FUNCTION_ARGS)
{
    PG_RETURN_POINTER( btree_picksplit(
	(bytea*)PG_GETARG_POINTER(0),
	(GIST_SPLITVEC*)PG_GETARG_POINTER(1),
	sizeof(INT4KEY),
	gint4_binary_union,
	gint4_inter,
	size_int4key
    ) );
}

Datum
gint4_same(PG_FUNCTION_ARGS)
{
  INT4KEY *b1 = (INT4KEY*) PG_GETARG_POINTER(0);
  INT4KEY *b2 = (INT4KEY*) PG_GETARG_POINTER(1);
  bool *result = (bool*) PG_GETARG_POINTER(2);

  *result = ( b1->lower == b2->lower && b1->upper == b2->upper ) ? TRUE : FALSE;
  PG_RETURN_POINTER(result);
}

/* 
** SUPPORT ROUTINES for int4gons
*/
Datum 
gint4_inter(PG_FUNCTION_ARGS) {
  	INT4KEY *b1 = (INT4KEY*) PG_GETARG_POINTER(0);
  	INT4KEY *b2 = (INT4KEY*) PG_GETARG_POINTER(1);
	INT4KEY *interd = palloc( sizeof(INT4KEY));

	if ( b1->lower > b2->upper || b1->upper < b2->lower ) {
		PG_RETURN_POINTER( NULL );
	}

	interd->lower = ( b1->lower > b2->lower ) ? b1->lower : b2->lower;
	interd->upper = ( b1->upper > b2->upper ) ? b2->upper : b1->upper;

	PG_RETURN_POINTER( interd );
}

static char *
gint4_binary_union(char *r1, char *r2, int *sizep)
{
    INT4KEY *b1 = (INT4KEY*) r1; 
    INT4KEY *b2 = (INT4KEY*) r2;
    INT4KEY *uniond = palloc( sizeof(INT4KEY));

	uniond->lower = ( b1->lower > b2->lower ) ?
		b2->lower : b1->lower;
	uniond->upper = ( b1->upper > b2->upper ) ?
		b1->upper : b2->upper;
    	*sizep = sizeof(int4);
    return (char*)uniond;
}


static float 
size_int4key( char *pk ) {
    if ( pk ) {
	return (float)( ((INT4KEY*)pk)->upper - ((INT4KEY*)pk)->lower );
    } else
	return 0.0;
}

/**************************************************
 * timestamp ops
 **************************************************/  

Datum
gts_compress(PG_FUNCTION_ARGS)
{
	GISTENTRY *entry=(GISTENTRY*)PG_GETARG_POINTER(0);
	GISTENTRY *retval;

	if ( entry->leafkey) {
		TSKEY *r = (TSKEY *)palloc( sizeof(TSKEY) );
		retval = palloc(sizeof(GISTENTRY));
		if ( entry->key ) {
			r->lower = r->upper = *(Timestamp*)(entry->key);

			gistentryinit(*retval, PointerGetDatum(r), entry->rel, entry->page,
				entry->offset, sizeof(TSKEY),FALSE);

		} else {
			gistentryinit(*retval, NULL, entry->rel, entry->page,
				entry->offset, 0,FALSE);
		} 
	} else {
		retval = entry;
	}
	PG_RETURN_POINTER( retval );
}

Datum 
gts_consistent(PG_FUNCTION_ARGS)
{
    GISTENTRY *entry = (GISTENTRY*) PG_GETARG_POINTER(0);
    Timestamp *query       = (Timestamp *)PG_GETARG_POINTER(1);
    StrategyNumber strategy = (StrategyNumber) PG_GETARG_UINT16(2);
    bool retval;
    TSKEY *key;
    /*
    ** if entry is not leaf, use gbox_internal_consistent,
    ** else use gbox_leaf_consistent
    */
    if ( ! entry->key )
	return FALSE;
    key = (TSKEY*) DatumGetPointer(entry->key);

    switch(strategy) {
	case BTLessEqualStrategyNumber:
		retval = TSGE( query, &(key->lower) );
		break;
	case BTLessStrategyNumber:
		if (GIST_LEAF(entry))
			retval = TSGT( query, &(key->lower) );
		else 
			retval = TSGE( query, &(key->lower) );
		break;
	case BTEqualStrategyNumber:
    		/* in leaf page key->lower always = key->upper */
		if (GIST_LEAF(entry)) 
			retval = TSEQ( query, &(key->lower));
		else 
			retval = ( TSLE( &(key->lower), query ) && TSLE( query, &(key->upper) ) ); 
		break;
	case BTGreaterStrategyNumber:
		if (GIST_LEAF(entry))
			retval = TSLT( query, &(key->upper) );
		else 
			retval = TSLE( query, &(key->upper) );
		break;
	case BTGreaterEqualStrategyNumber:
		retval = TSLE( query, &(key->upper) );
		break;
	default:
		retval = FALSE;
    }
    PG_RETURN_BOOL(retval);
}

Datum
gts_union(PG_FUNCTION_ARGS)
{
    PG_RETURN_POINTER( 
	btree_union(
		(bytea*) PG_GETARG_POINTER(0),
		(int*) PG_GETARG_POINTER(1),
		gts_binary_union
	)
   );
}

Datum
gts_penalty(PG_FUNCTION_ARGS)
{
    PG_RETURN_POINTER( btree_penalty(
	(GISTENTRY*) PG_GETARG_POINTER(0),
	(GISTENTRY*) PG_GETARG_POINTER(1),
	(float*) PG_GETARG_POINTER(2),
	gts_binary_union,
	size_tskey
    ) );
}

Datum
gts_picksplit(PG_FUNCTION_ARGS)
{
    PG_RETURN_POINTER( btree_picksplit(
	(bytea*)PG_GETARG_POINTER(0),
	(GIST_SPLITVEC*)PG_GETARG_POINTER(1),
	sizeof(TSKEY),
	gts_binary_union,
	gts_inter,
	size_tskey
    ) );
}

Datum
gts_same(PG_FUNCTION_ARGS)
{
  TSKEY *b1 = (TSKEY*) PG_GETARG_POINTER(0);
  TSKEY *b2 = (TSKEY*) PG_GETARG_POINTER(1);

  bool *result = (bool*) PG_GETARG_POINTER(2);
  if ( b1 && b2 )
	*result = ( TSEQ( &(b1->lower), &(b2->lower) )  && TSEQ( &(b1->upper), &(b2->upper) ) ) ? TRUE : FALSE;
  else
	*result = ( b1==NULL && b2==NULL ) ? TRUE : FALSE; 
  PG_RETURN_POINTER(result);
}

/* 
** SUPPORT ROUTINES for tsgons
*/
Datum 
gts_inter(PG_FUNCTION_ARGS) {
  	TSKEY *b1 = (TSKEY*) PG_GETARG_POINTER(0);
  	TSKEY *b2 = (TSKEY*) PG_GETARG_POINTER(1);
	TSKEY *interd;

	if ( TSGT( &(b1->lower), &(b2->upper) ) || TSLT( &(b1->upper), &(b2->lower) ) ) {
		PG_RETURN_POINTER( NULL );
	}

	interd = (TSKEY *) palloc(sizeof(TSKEY));
	
	interd->lower = ( TSGT( &(b1->lower), &(b2->lower) ) ) ? b1->lower : b2->lower;
	interd->upper = ( TSGT( &(b1->upper), &(b2->upper) ) ) ? b2->upper : b1->upper;
	
	PG_RETURN_POINTER( interd );
}

static char *
gts_binary_union(char *r1, char *r2, int *sizep)
{
    TSKEY *retval;

    if ( ! (r1 && r2) ) {
	if ( r1 ) {
		retval = (TSKEY*)palloc( sizeof(TSKEY) );
		memcpy( (void*)retval, (void*)r1, sizeof(TSKEY) );
    		*sizep = sizeof(TSKEY);
	} else if ( r2 ) {
		retval = (TSKEY*)palloc( sizeof(TSKEY) );
		memcpy( (void*)retval, (void*)r2, sizeof(TSKEY) );
    		*sizep = sizeof(TSKEY);
	} else {
		*sizep = 0;
		retval = NULL;
	} 
    } else {	
	retval = (TSKEY *) palloc(sizeof(TSKEY));
	retval->lower = ( TSGT( &(((TSKEY*)r1)->lower), &(((TSKEY*)r2)->lower) ) ) ? 
		((TSKEY*)r2)->lower : ((TSKEY*)r1)->lower;
	retval->upper = ( TSGT( &(((TSKEY*)r1)->lower), &(((TSKEY*)r2)->lower) ) ) ?
		((TSKEY*)r1)->upper : ((TSKEY*)r2)->upper;
    	*sizep = sizeof(TSKEY);
    }
    return (char*)retval;
}

static float 
size_tskey( char *pk ) {
    if ( pk )
	return (float)( ((TSKEY*)pk)->upper - ((TSKEY*)pk)->lower );
    else
	return 0.0;
}

/**************************************************
 * Common btree-function (for all ops)
 **************************************************/

static char* 
btree_union(bytea *entryvec, int *sizep, BINARY_UNION bu)
{
    int numranges, i;
    char *out, *tmp;

    numranges = (VARSIZE(entryvec) - VARHDRSZ)/sizeof(GISTENTRY); 
    tmp = (char *)(((GISTENTRY *)(VARDATA(entryvec)))[0]).key;
    out = NULL;

    for (i = 1; i < numranges; i++) {
	out = (*bu)(tmp, (char *)
				 (((GISTENTRY *)(VARDATA(entryvec)))[i]).key,
				 sizep);
	if (i > 1 && tmp) pfree(tmp);

	tmp = out;
    }

    return(out);
}

static float *
btree_penalty(GISTENTRY *origentry, GISTENTRY *newentry, float *result, BINARY_UNION bu, SIZE_BOX sb)
{
    char * ud;
    float tmp1;
    int sizep;
   
    ud = (*bu)( (char*)origentry->key, (char*)newentry->key, &sizep );
    tmp1 = (*sb)( ud ); 
    if (ud) pfree(ud);

    *result = tmp1 - (*sb)( (char*)(origentry->key) );
    return(result);
}

/*
** The GiST PickSplit method
** We use Guttman's int4 time split algorithm 
*/
static GIST_SPLITVEC *
btree_picksplit(bytea *entryvec, GIST_SPLITVEC *v, int keylen, BINARY_UNION bu, RDF interop, SIZE_BOX sb)
{
    OffsetNumber i, j;
    char *datum_alpha, *datum_beta;
    char *datum_l, *datum_r;
    char *union_d, *union_dl, *union_dr;
    char *inter_d;
    bool firsttime;
    float size_alpha, size_beta, size_union, size_inter;
    float size_waste, waste;
    float size_l, size_r;
    int nbytes;
    int sizep;
    OffsetNumber seed_1 = 0, seed_2 = 0;
    OffsetNumber *left, *right;
    OffsetNumber maxoff;

    maxoff = ((VARSIZE(entryvec) - VARHDRSZ)/sizeof(GISTENTRY)) - 2;
    nbytes =  (maxoff + 2) * sizeof(OffsetNumber);
    v->spl_left = (OffsetNumber *) palloc(nbytes);
    v->spl_right = (OffsetNumber *) palloc(nbytes);
    
    firsttime = true;
    waste = 0.0;
    
    for (i = FirstOffsetNumber; i < maxoff; i = OffsetNumberNext(i)) {
	datum_alpha = (char *)(((GISTENTRY *)(VARDATA(entryvec)))[i].key);
	for (j = OffsetNumberNext(i); j <= maxoff; j = OffsetNumberNext(j)) {
	    datum_beta = (char *)(((GISTENTRY *)(VARDATA(entryvec)))[j].key);
	    
	    /* compute the wasted space by unioning these guys */
	    /* size_waste = size_union - size_inter; */	
	    union_d = (*bu)( datum_alpha, datum_beta, &sizep );
	    if ( union_d ) {
		size_union = (*sb)(union_d);
		pfree(union_d);
	    } else
		size_union = 0.0;

	    if ( datum_alpha && datum_beta ) {
    	    	inter_d = DatumGetPointer(DirectFunctionCall2(
			interop,
			PointerGetDatum( datum_alpha ),
			PointerGetDatum( datum_beta )) );
		if ( inter_d ) {
			size_inter = (*sb)(inter_d);
			pfree(inter_d);
		} else 
			size_inter = 0.0;
	    } else
		size_inter = 0.0;

	    size_waste = size_union - size_inter;
	    
	    /*
	     *  are these a more promising split that what we've
	     *  already seen?
	     */
	    
	    if (size_waste > waste || firsttime) {
		waste = size_waste;
		seed_1 = i;
		seed_2 = j;
		firsttime = false;
	    }
	}
    }
    
    left = v->spl_left;
    v->spl_nleft = 0;
    right = v->spl_right;
    v->spl_nright = 0;
  
	datum_l = (char*) palloc( keylen );
	memcpy( (void*)datum_l, (void*)(((GISTENTRY *)(VARDATA(entryvec)))[seed_1].key ), keylen );
    size_l  = (*sb)( datum_l ); 
	datum_r = (char*) palloc( keylen );
	memcpy( (void*)datum_r, (void*)(((GISTENTRY *)(VARDATA(entryvec)))[seed_2].key ), keylen );
    size_r  = (*sb)( datum_r ); 
    
    /*
     *  Now split up the regions between the two seeds.  An important
     *  property of this split algorithm is that the split vector v
     *  has the indices of items to be split in order in its left and
     *  right vectors.  We exploit this property by doing a merge in
     *  the code that actually splits the page.
     *
     *  For efficiency, we also place the new index tuple in this loop.
     *  This is handled at the very end, when we have placed all the
     *  existing tuples and i == maxoff + 1.
     */
    
    maxoff = OffsetNumberNext(maxoff);
    for (i = FirstOffsetNumber; i <= maxoff; i = OffsetNumberNext(i)) {
	
	/*
	 *  If we've already decided where to place this item, just
	 *  put it on the right list.  Otherwise, we need to figure
	 *  out which page needs the least enlargement in order to
	 *  store the item.
	 */
	
	if (i == seed_1) {
	    *left++ = i;
	    v->spl_nleft++;
	    continue;
	} else if (i == seed_2) {
	    *right++ = i;
	    v->spl_nright++;
	    continue;
	}
	
	/* okay, which page needs least enlargement? */ 
	datum_alpha = (char *)(((GISTENTRY *)(VARDATA(entryvec)))[i].key);
	union_dl = (*bu)( datum_l, datum_alpha, &sizep );
	union_dr = (*bu)( datum_r, datum_alpha, &sizep );
	size_alpha = (*sb)( union_dl ); 
	size_beta  = (*sb)( union_dr ); 
	
	/* pick which page to add it to */
	if (size_alpha - size_l < size_beta - size_r + WISH_F(v->spl_nleft, v->spl_nright, 1) ) {
	    pfree(datum_l);
	    pfree(union_dr);
	    datum_l = union_dl;
	    size_l = size_alpha;
	    *left++ = i;
	    v->spl_nleft++;
	} else {
	    pfree(datum_r);
	    pfree(union_dl);
	    datum_r = union_dr;
	    size_r = size_alpha;
	    *right++ = i;
	    v->spl_nright++;
	}
    }
    *left = *right = FirstOffsetNumber;	/* sentinel value, see dosplit() */
    
    v->spl_ldatum = (Datum)datum_l;
    v->spl_rdatum = (Datum)datum_r;
    return( v );
}

/*
** GiST DeCompress methods
** do not do anything.
*/
Datum
btree_decompress(PG_FUNCTION_ARGS)
{
    PG_RETURN_POINTER(PG_GETARG_POINTER(0));
}


/**************************************************
 * In/Out for keys, not really needed
 **************************************************/
Datum	
int4key_in(PG_FUNCTION_ARGS) {
	INT4KEY *key = palloc(sizeof(INT4KEY));

	if ( sscanf( PG_GETARG_POINTER(0), "%d|%d", &(key->lower), &(key->upper)) != 2 ) 
		elog(ERROR, "Error in input format");
 
	PG_RETURN_POINTER( key );
}

Datum	int4key_out(PG_FUNCTION_ARGS) {
	INT4KEY *key = (INT4KEY *) PG_GETARG_POINTER(0);
	char *str=palloc(sizeof(char)*22);
	sprintf(str,"%d|%d", key->lower, key->upper);
	PG_RETURN_POINTER( str ); 
}

Datum
tskey_in(PG_FUNCTION_ARGS) {
	elog(ERROR, "Not implemented");
	PG_RETURN_POINTER( NULL );
}

Datum
tskey_out(PG_FUNCTION_ARGS) {
	elog(ERROR, "Not implemented");
	PG_RETURN_POINTER( NULL );
}
