/*-------------------------------------------------------------------------
 *
 * json.c
 *
 *
 * Copyright (c) 2008-2009, PostgreSQL Global Developent Group
 *
 * IDENTIFICATION
 *	  $PostgreSQL: pgsql/contrib/json
 *
 * JSON support - comfortable function sample
 *-------------------------------------------------------------------------
 */
#include "postgres.h"
#include "fmgr.h"

#include "catalog/namespace.h"
#include "catalog/pg_type.h"
#include "lib/stringinfo.h"
#include "nodes/makefuncs.h"
#include "nodes/nodeFuncs.h"
#include "parser/parse_coerce.h"
#include "parser/parse_expr.h"
#include "parser/parse_func.h"
#include "parser/parse_target.h"
#include "parser/parse_type.h"
#include "utils/array.h"
#include "utils/builtins.h"
#include "utils/lsyscache.h"
#include "utils/syscache.h"

#include <string.h>

Node * transformJSON(ParseState *pstate, Node *expr);

/* Saved hook value  */
static ParseExprTransform_hook_type	prev_transformExpr = NULL;

void	_PG_init(void);
void	_PG_fini(void);

PG_FUNCTION_INFO_V1(jsonlib_json_object);
PG_FUNCTION_INFO_V1(jsonlib_json_array);
PG_FUNCTION_INFO_V1(jsonlib_json_members);
PG_FUNCTION_INFO_V1(jsonlib_jsonin);

Datum jsonlib_json_object(PG_FUNCTION_ARGS);
Datum jsonlib_json_array(PG_FUNCTION_ARGS);
Datum jsonlib_json_members(PG_FUNCTION_ARGS);
Datum jsonlib_jsonin(PG_FUNCTION_ARGS);

static void loadDynamicFid(void);

static Oid jsonlib_json_object_oid = InvalidOid;
static Oid jsonlib_json_members_oid = InvalidOid;
static Oid json_type_oid = InvalidOid;

PG_MODULE_MAGIC;


/*
 * Convert C string to JSON string
 */
static char *
json_string(char *str)
{
	char	len = strlen(str);
	char		*result, *wc;
	
	wc = result = palloc(len * 2 + 1);
	while (*str != '\0')
	{
		char	c = *str++;
		
		switch (c)
		{
			case '\t':
				*wc++ = '\\';
				*wc++ = 't';
				break;
			case '\b':
				*wc++ = '\\';
				*wc++ = 'b';
				break;
			case '\n':
				*wc++ = '\\';
				*wc++ = 'n';
				break;
			case '\r':
				*wc++ = '\\';
				*wc++ = 'r';
				break;
			case '\\':
				*wc++ = '\\';
				*wc++ = '\\';
				break;
			case '"':
				*wc++ = '\\';
				*wc++ = '"';
				break;
			default:
				*wc++ = c;
		}
	}
	*wc = '\0';
	return result;
}

/*
 * Don't allow enter json value directly. Should be replaced by real 
 * json parser in future.
 */
Datum
jsonlib_jsonin(PG_FUNCTION_ARGS)
{
	ereport(ERROR,
			(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
			 errmsg("cannot accept a value of type json")));
	PG_RETURN_VOID();
}


Datum
jsonlib_json_members(PG_FUNCTION_ARGS)
{
	int i;
	Oid	valtype;
	StringInfoData buf;
	text *result;
	
	loadDynamicFid();
	
	initStringInfo(&buf);
	for (i = 0; i < fcinfo->nargs;i++)
	{
		Oid	typoutput;
		bool		typIsVarlena;
		bool		isnull;
		Datum value;
		TYPCATEGORY typcat;
		isnull = PG_ARGISNULL(i);
		
		if (i > 0)
			appendStringInfoChar(&buf,',');
	
		if (PG_ARGISNULL(i))
			elog(ERROR, "property name cannot be empty");
		valtype = get_fn_expr_argtype(fcinfo->flinfo, i);
		
		if (valtype == TEXTOID)
		{
			appendStringInfo(&buf, "\"%s\":", json_string(TextDatumGetCString(PG_GETARG_TEXT_P(i))));
		}
		else
			elog(ERROR, "property type isn't text %d", valtype);

		i++;
		if (i >= fcinfo->nargs)
			elog(ERROR, "missing propert value");
			
		isnull = PG_ARGISNULL(i);
		
		if (!isnull)
		{
			valtype = get_fn_expr_argtype(fcinfo->flinfo, i);
			if (!OidIsValid(valtype))
				elog(ERROR, "could not determine data type of input");
				
			typcat = TypeCategory(valtype);
		
			getTypeOutputInfo(valtype, &typoutput, &typIsVarlena);
		
			value = PG_GETARG_DATUM(i);
			if (valtype == json_type_oid || typcat == 'N')
				appendStringInfoString(&buf, OidOutputFunctionCall(typoutput, value));
			else if (typcat == 'B')
			{
				bool	bool_value = PG_GETARG_BOOL(i);
				appendStringInfoString(&buf, bool_value ? "true" : "false");
			}
			else
				appendStringInfo(&buf, "\"%s\"", json_string(OidOutputFunctionCall(typoutput, value)));
		}
		else
			appendStringInfoString(&buf, "null");
	}
	
	result = DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum(buf.data)));
	
	PG_RETURN_TEXT_P(result);
}

Datum
jsonlib_json_array(PG_FUNCTION_ARGS)
{
	int i;
	Oid	valtype;
	StringInfoData buf;
	text *result;
	
	loadDynamicFid();
	
	initStringInfo(&buf);
	appendStringInfoChar(&buf, '[');
	for (i = 0; i < fcinfo->nargs; i++)
	{
		Oid	typoutput;
		bool		typIsVarlena;
		bool		isnull;
		Datum value;
		TYPCATEGORY typcat;
		isnull = PG_ARGISNULL(i);
		
		if (i > 0)
			appendStringInfoChar(&buf,',');
	
		if (!isnull)
		{
			valtype = get_fn_expr_argtype(fcinfo->flinfo, i);
			if (!OidIsValid(valtype))
				elog(ERROR, "could not determine data type of input");
				
			typcat = TypeCategory(valtype);
		
			getTypeOutputInfo(valtype, &typoutput, &typIsVarlena);
		
			value = PG_GETARG_DATUM(i);
			if (valtype == json_type_oid || typcat == 'N')
				appendStringInfoString(&buf, OidOutputFunctionCall(typoutput, value));
			else if (typcat == 'B')
			{
				bool	bool_value = PG_GETARG_BOOL(i);
				appendStringInfoString(&buf, bool_value ? "true" : "false");
			}
			else
				appendStringInfo(&buf, "\"%s\"", json_string(OidOutputFunctionCall(typoutput, value)));
		}
		else
			appendStringInfoString(&buf, "null");
		
	}

	appendStringInfoChar(&buf, ']');
	
	result = DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum(buf.data)));
	PG_RETURN_TEXT_P(result);
}


Datum 
jsonlib_json_object(PG_FUNCTION_ARGS)
{
	char	*str[100];
	
	Datum *elems;
	int		nelems;
	int i;
	Oid	valtype;
	StringInfoData buf;
	text *result;
	
	deconstruct_array(PG_GETARG_ARRAYTYPE_P(0),
				TEXTOID, -1, false, 'i',
				&elems, NULL, &nelems);

	initStringInfo(&buf);
	appendStringInfoChar(&buf, '{');
	for (i = 0; i < nelems; i++)
	{
		Oid	typoutput;
		bool		typIsVarlena;
		bool		isnull;
		Datum value;
		TYPCATEGORY typcat;
		isnull = PG_ARGISNULL(i+1);
		
		if (i > 0)
			appendStringInfoChar(&buf,',');
	
		str[i] = TextDatumGetCString(elems[i]);
		
		/* skip json_members label */
		if (strcmp(str[i], "json_members") != 0)
			appendStringInfo(&buf, "\"%s\":", json_string(str[i]));
		
		if (!isnull)
		{
			valtype = get_fn_expr_argtype(fcinfo->flinfo, i+1);
			if (!OidIsValid(valtype))
				elog(ERROR, "could not determine data type of input");
				
			typcat = TypeCategory(valtype);
		
			getTypeOutputInfo(valtype, &typoutput, &typIsVarlena);
		
			value = PG_GETARG_DATUM(i+1);
			
			if (valtype == json_type_oid || typcat == 'N')
				appendStringInfoString(&buf, OidOutputFunctionCall(typoutput, value));
			else if (typcat == 'B')
			{
				bool	bool_value = PG_GETARG_BOOL(i+1);
				appendStringInfoString(&buf, bool_value ? "true" : "false");
			}
			else
				appendStringInfo(&buf, "\"%s\"", json_string(OidOutputFunctionCall(typoutput, value)));
		}
		else
			appendStringInfoString(&buf, "null");
	}

	appendStringInfoChar(&buf, '}');

	result = DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum(buf.data)));
	PG_RETURN_TEXT_P(result);
}



void
_PG_init(void)
{
	/* Install hooks. */
	prev_transformExpr = ParseExprTransform_hook;
	ParseExprTransform_hook = transformJSON;
}

/*
 * Module unload callback
 */
void
_PG_fini(void)
{
	/* Uninstall hooks. */
	ParseExprTransform_hook = prev_transformExpr;
	jsonlib_json_object_oid = InvalidOid;
	jsonlib_json_members_oid = InvalidOid;
	json_type_oid = InvalidOid;
}


static void 
loadDynamicFid()
{
	Oid	argtypes[2];
	Oid		*true_typeids;
	int		nvargs;
	Oid		rettype;
	bool		retset;
	
	if (jsonlib_json_object_oid == InvalidOid)
	{
		HeapTuple	typeTup;
	
		argtypes[0] = TEXTARRAYOID;
		argtypes[1] = TEXTOID;
		if (func_get_detail(list_make2(makeString("json"), makeString("__json_object")),
						    NIL, NIL, 2, argtypes, false, true,
						    &jsonlib_json_object_oid, &rettype, 
						    &retset, &nvargs, &true_typeids, NULL, NULL, NULL) != FUNCDETAIL_NORMAL)
			ereport(ERROR,
				    (errcode(ERRCODE_INTERNAL_ERROR),
				     errmsg("cannot find __json_object function"),
				     errhint("Please, try to import registering sql script for json module - \"json.sql\".")));
		
		argtypes[0] = TEXTOID;
		if (func_get_detail(list_make2(makeString("json"), makeString("json_members")),
						    NIL, NIL, 1, argtypes, false, true,
						    &jsonlib_json_members_oid, &rettype, 
						    &retset, &nvargs, &true_typeids, NULL, NULL, NULL) != FUNCDETAIL_NORMAL)
			ereport(ERROR,
				    (errcode(ERRCODE_INTERNAL_ERROR),
				     errmsg("cannot find __json_object function"),
				     errhint("Please, try to import registering sql script for json module - \"json.sql\".")));
		
		
		typeTup = LookupTypeName(NULL, makeTypeName("json"), NULL);
		if (typeTup)
		{
			json_type_oid = HeapTupleGetOid(typeTup);
			ReleaseSysCache(typeTup);
		}
		else
			ereport(ERROR,
				    (errcode(ERRCODE_INTERNAL_ERROR),
				     errmsg("cannot find json domain type"),
				     errhint("Please, try to import registering sql script for json module - \"json.sql\".")));
	}
	Assert(jsonlib_json_object_oid != InvalidOid);
	Assert(jsonlib_json_members_oid != InvalidOid);
	Assert(json_type_oid != InvalidOid);
}

/*
 * Basic rule - don't duplicate code, 
 *    transformation have to be transitive!!
 * 
 */
Node *
transformJSON(ParseState *pstate, Node *expr)
{
	/* 
	 * a) collect labels and column names, store its to first param
	 * b) coerce data to basic JSON types
	 */
	if (IsA(expr, FuncCall) )
	{
		FuncCall   *fn = (FuncCall *) expr;
		char		*schemaname;
		char		*funcname;
		ListCell		*args;
		List			*modargs = NIL;
		Datum		*labels;
		
		
		DeconstructQualifiedName(fn->funcname, &schemaname, &funcname);
		if (schemaname != NULL && strcmp(schemaname, "json") != 0)
			goto not_json_func;
	
		if (strcmp(funcname, "json_object") == 0)
		{
			int			i = 0;
			int nlabels = list_length(fn->args);
			
			loadDynamicFid();
			labels = (Datum *) palloc0(nlabels * sizeof(Datum));
			
			foreach(args, fn->args)
			{
				Node *nd = lfirst(args);
				Node	*tnode;
			
				if (IsA(nd, ArgExpr))
				{
					labels[i] = CStringGetTextDatum(((ArgExpr *) nd)->name);
					tnode = transformExpr(pstate, ((ArgExpr *) nd)->expr);
				}
				else
				{
					char	*label = FigureColname(nd);
				
					/* hack now */
					if (strcmp(label, "?column?") == 0)
					ereport(ERROR,
							(errcode(ERRCODE_SYNTAX_ERROR),
							 errmsg("unnamed JSON attribute must be a column reference"),
							 parser_errposition(pstate, exprLocation(nd))));
					
					labels[i] = CStringGetTextDatum(label);
						
					tnode = transformExpr(pstate, nd);
				}
				modargs = lappend(modargs, tnode);
				i++;
			}
			
			/*
			 * labels are transported as serialised list
			 */
			modargs = lcons(makeConst(TEXTARRAYOID, -1, -1, 
								    PointerGetDatum(construct_array(labels, nlabels, TEXTOID, -1, false, 'i')), false, false),
								modargs);
			
			/* 
			 * because I don't wont to fake parameters system tables,
			 * this transformation isnt't transitive.
			 */
			return (Node *)  makeFuncExpr(jsonlib_json_object_oid,
						    json_type_oid, modargs, COERCE_EXPLICIT_CALL);
		}
		/* cast every odd parameter to text */
		if (strcmp(funcname, "json_members") == 0)
		{
			int		i = 0;
			List			*targs = NIL;
			
			loadDynamicFid();
			
			foreach(args, fn->args)
			{
				Node *arg = (Node *) lfirst(args);
				arg = transformExpr(pstate, arg);
			
				if (i++ % 2 == 0)
					arg = coerce_to_specific_type(pstate, arg, TEXTOID, "json property name");

				targs = lappend(targs, arg);
			}
			
			return (Node *) makeFuncExpr(jsonlib_json_members_oid,
									    json_type_oid, targs, COERCE_EXPLICIT_CALL);
			
		}
	}

not_json_func:
	if (prev_transformExpr)
		return prev_transformExpr(pstate, expr);
	else
		return standard_transformExpr(pstate, expr);
}
