/*-------------------------------------------------------------------------
 *
 * null.c
 *
 *
 * Copyright (c) 2008-2009, PostgreSQL Global Developent Group
 *
 * IDENTIFICATION
 *	  $PostgreSQL: pgsql/contrib/null
 *
 *-------------------------------------------------------------------------
 *
 * Rules:
 *   a)   X || X || NULL --> XX
 *   b)   NULL || NULL   --> NULL
 *   c)   ''             --> NULL
 *   d)   substring('abcd' FROM 10 FOR 1) --> NULL
 *   e)   '' IS NULL     --> true
 *
 */
#include "postgres.h"
#include "fmgr.h"

#include "catalog/namespace.h"
#include "catalog/pg_type.h"
#include "nodes/makefuncs.h"
#include "parser/parse_coerce.h"
#include "parser/parse_expr.h"
#include "parser/parse_func.h"


PG_MODULE_MAGIC;

/*
 * These functions are used for dynamic empty string replacing 
 */
static Oid ces1 = InvalidOid;
static Oid ces2 = InvalidOid;
static Oid cces = InvalidOid;

/* Saved hook value  */
static ParseExprTransform_hook_type	prev_transformExpr = NULL;

void	_PG_init(void);
void	_PG_fini(void);

static Node * transformEmptyStr(ParseState *pstate, Node *expr);
static Node * makeStringConst(char *str, int location);

PG_FUNCTION_INFO_V1(__orafce_replace_empty_string1);
PG_FUNCTION_INFO_V1(__orafce_replace_empty_string2);
PG_FUNCTION_INFO_V1(__orafce_replace_empty_cstring);

Datum __orafce_replace_empty_string1(PG_FUNCTION_ARGS);
Datum __orafce_replace_empty_string2(PG_FUNCTION_ARGS);
Datum __orafce_replace_empty_cstring(PG_FUNCTION_ARGS);


static Node *
makeStringConst(char *str, int location)
{
	A_Const *n = makeNode(A_Const);

	n->val.type = T_String;
	n->val.val.str = str;
	n->location = location;

	return (Node *)n;
}

static Node *
wrapNode(Oid ces, Node *node, Oid resulttype, int location)
{
	FuncExpr *r = (FuncExpr *) makeNode(FuncExpr);
	
	Assert(ces != InvalidOid);
	
	r->funcid = ces;
	r->funcresulttype = resulttype;
	r->funcretset = false;
	r->funcformat = COERCE_IMPLICIT_CAST;
	r->args = list_make1(node);
	r->location = location;
	
	return (Node *) r;
}

static Node *
wrapNodeCast(ParseState *pstate, Node *node, Oid resulttype, int location)
{
	FuncExpr *r;
	Node *param = coerce_to_target_type(pstate, node, resulttype,
								    TEXTOID, -1,
								    COERCION_IMPLICIT, COERCE_IMPLICIT_CAST,
								    location);
					

	/* when we cannot cast, returns original node */
	if (param == NULL)
		return node;
		
	Assert(ces2 != InvalidOid);

	r = (FuncExpr *) makeNode(FuncExpr);
	r->funcid = ces2;
	r->funcresulttype = resulttype;
	r->funcretset = false;
	r->funcformat = COERCE_IMPLICIT_CAST;
	/* I don't know if copy is necessary */
	r->args = list_make2(param, copyObject(node));
	r->location = location;
	
	return (Node *) r;
}

/* 
 * Private polymorphic function 
 *   First parameter should be binary compatible with text
 */
Datum
__orafce_replace_empty_string1(PG_FUNCTION_ARGS)
{
	text *result = PG_GETARG_TEXT_P(0);
	
	if (VARSIZE(result) == VARHDRSZ)
	{
		PG_RETURN_NULL();
	}
	PG_RETURN_DATUM(PG_GETARG_DATUM(0));
}

Datum
__orafce_replace_empty_string2(PG_FUNCTION_ARGS)
{
	text *constr = PG_GETARG_TEXT_P(0);
	
	if (VARSIZE(constr) == VARHDRSZ)
	{
		PG_RETURN_NULL();
	}
	PG_RETURN_DATUM(PG_GETARG_DATUM(1));
}

Datum
__orafce_replace_empty_cstring(PG_FUNCTION_ARGS)
{
	char	*str = PG_GETARG_CSTRING(0);
	
	if (strcmp(str, "") == 0)
	{
		PG_RETURN_NULL();
	}
	PG_RETURN_DATUM(PG_GETARG_DATUM(0));
}

static void
loadCESFunctionOids(Oid *ces1, Oid *ces2, Oid *cces)
{
	Oid	argtypes[2];
	Oid		*true_typeids;
	int		nvargs;
	Oid		rettype;
	bool		retset;
	
	if (*ces1 == InvalidOid)
	{
		argtypes[0] = TEXTOID;
		if (func_get_detail(list_make1(makeString("__orafce_replace_empty_string1")),
						    NIL, NIL, 1, argtypes, false, true,
						    ces1, &rettype, 
						    &retset, &nvargs, &true_typeids, NULL, NULL, NULL) != FUNCDETAIL_NORMAL)
			ereport(ERROR,
				    (errcode(ERRCODE_INTERNAL_ERROR),
				     errmsg("cannot find __orafce_replace_empty_string1 function"),
				     errhint("Please, try to import registering sql script for null module - \"null.sql\".")));
		
		argtypes[0] = TEXTOID;
		argtypes[1] = TEXTOID;
		if (func_get_detail(list_make1(makeString("__orafce_replace_empty_string2")),
						    NIL, NIL, 2, argtypes, false, true,
						    ces2, &rettype, 
						    &retset, &nvargs, &true_typeids, NULL, NULL, NULL) != FUNCDETAIL_NORMAL)
			ereport(ERROR,
				    (errcode(ERRCODE_INTERNAL_ERROR),
				     errmsg("cannot find __orafce_replace_empty_string2 function"),
				     errhint("Please, try to import registering sql script for null module - \"null.sql\".")));
					
		argtypes[0] = CSTRINGOID;
		if (func_get_detail(list_make1(makeString("__orafce_replace_empty_cstring")),
						    NIL, NIL, 1, argtypes, false, true,
						    cces, &rettype, 
						    &retset, &nvargs, &true_typeids, NULL, NULL, NULL) != FUNCDETAIL_NORMAL)
			ereport(ERROR,
				    (errcode(ERRCODE_INTERNAL_ERROR),
				     errmsg("cannot find __orafce_replace_empty_cstring function"),
				     errhint("Please, try to import registering sql script for null module - \"null.sql\".")));
		
	}
	
	Assert(*ces1 != InvalidOid);
	Assert(*ces2 != InvalidOid);
	Assert(*cces != InvalidOid);
}

/*
 * Module load callback - we cannot initialise ces in this moment,
 * because we cannot find function in pg_proc before registration.
 */
void
_PG_init(void)
{
	/* Install hooks. */
	prev_transformExpr = ParseExprTransform_hook;
	ParseExprTransform_hook = transformEmptyStr;
	
	ces1 = ces2 = cces = InvalidOid;
}

/*
 * Module unload callback
 */
void
_PG_fini(void)
{
	/* Uninstall hooks. */
	ParseExprTransform_hook = prev_transformExpr;
}


/*
 * transform node to emulate Oracle emty string behave
 * premise: stored data doesn't contain any empty string!
 */
Node *
transformEmptyStr(ParseState *pstate, Node *expr)
{
	Node *result;
	
	/* This helps with NULL skipping || operator -> use coalesce */
	if (IsA(expr, A_Expr))
	{
		A_Expr *aexpr = (A_Expr *) expr;
		char	*schemaname;
		char	*funcname;
		
		if (aexpr->kind == AEXPR_OP)
		{
			DeconstructQualifiedName(aexpr->name, &schemaname, &funcname);
			if (strcmp(funcname, "||") == 0)
			{
				CoalesceExpr *clexpr;
				CoalesceExpr *crexpr;
				
				clexpr = (CoalesceExpr *) makeNode(CoalesceExpr);
				clexpr->location = aexpr->location;
				clexpr->args = list_make2(aexpr->lexpr, makeStringConst("", -1));
				
				crexpr = (CoalesceExpr *) makeNode(CoalesceExpr);
				crexpr->location = aexpr->location;
				crexpr->args = list_make2(aexpr->rexpr, makeStringConst("", -1));
				
				expr = (Node *) makeSimpleA_Expr(AEXPR_OP, "||", 
										(Node *) clexpr, 
										(Node *) crexpr, 
										aexpr->location);
			}
		}
	}
	/* this replace explicit empty string */
	else if (IsA(expr, A_Const))
	{
		A_Const *c = (A_Const *) expr;

		if (c->val.type == T_String && c->location != -1 && *c->val.val.str == '\0')
		{
			A_Const *n = makeNode(A_Const);
			n->val.type = T_Null;
			n->location = c->location;
			expr = (Node *) n;
		}
	}

	if (prev_transformExpr)
		result =  prev_transformExpr(pstate, expr);
	else
		result =  standard_transformExpr(pstate, expr);

	/* this check result of any string function or operator */
	if (IsA(result, Const))
	{
		Const *c = (Const *) result;
		TYPCATEGORY	typcategory;
		/* 
		 * duplicate and cast to text when const value is any string
		 */
		if (!c->constisnull && c->location != -1)
		{
			typcategory = TypeCategory(c->consttype);
			if (typcategory == 'S')
			{
				if (IsBinaryCoercible(c->consttype, TEXTOID))
				{
					text *str = DatumGetTextP(c->constvalue);
					if (VARSIZE(str) == VARHDRSZ)
						result = (Node *) makeNullConst(c->consttype, c->consttypmod);
				}
				else
				{
					loadCESFunctionOids(&ces1, &ces2, &cces);
					result = wrapNodeCast(pstate, (Node *) c, c->consttype, c->location);
				}
			}
			/* unknown is stored as cstring */
			else if (typcategory == 'X')
			{
				if (strcmp(DatumGetPointer(c->constvalue), "") == 0)
				{
					result = (Node *) makeNullConst(UNKNOWNOID, -1);
				}
			}
		}
	}
	else if (IsA(result, FuncExpr))
	{
		FuncExpr *fe = (FuncExpr *) result;
		
		/* ignore SRF function */
		if (!fe->funcretset && fe->funcid != ces1 && fe->funcid != ces2 && fe->funcid != cces)
		{
			TYPCATEGORY	typcategory = TypeCategory(fe->funcresulttype);
			
			if (typcategory == 'S')
			{
				if (IsBinaryCoercible(fe->funcresulttype, TEXTOID))
				{
					loadCESFunctionOids(&ces1, &ces2, &cces);
					result = wrapNode(ces1, (Node *) fe, fe->funcresulttype, fe->location);
				}
				else /* propably only char type and name type*/
				{
					loadCESFunctionOids(&ces1, &ces2, &cces);
					result = wrapNodeCast(pstate, (Node *) fe, fe->funcresulttype, fe->location);
				}
			}
			else if (typcategory == 'X')
			{
				loadCESFunctionOids(&ces1, &ces2, &cces);
				result = wrapNode(cces, (Node *) fe, fe->funcresulttype, fe->location);
			}
		}
	}
	else if (IsA(result, OpExpr))
	{
		OpExpr *op = (OpExpr *) result;
		
		/* ignore SRF operator */
		if (!op->opretset)
		{
			TYPCATEGORY	typcategory = TypeCategory(op->opresulttype);
			
			if (typcategory == 'S')
			{
				if (IsBinaryCoercible(op->opresulttype, TEXTOID))
				{
					loadCESFunctionOids(&ces1, &ces2, &cces);
					result = wrapNode(ces1, (Node *) op, op->opresulttype, op->location);
				}
				else
				{
					loadCESFunctionOids(&ces1, &ces2, &cces);
					result = wrapNodeCast(pstate, (Node *) op, op->opresulttype, op->location);
				}
			}
			else if (typcategory == 'X')
			{
				loadCESFunctionOids(&ces1, &ces2, &cces);
				result = wrapNode(cces, (Node *) op, op->opresulttype, op->location);
			}
		}
	}

	return result;
}

