BEGIN;
-- Add language if not exist
CREATE OR REPLACE LANGUAGE plpgsql;

-- Create my own type
CREATE TYPE changeset AS (
	colname varchar,
	oldvalue varchar,
	newvalue varchar
);

-- Create test table
CREATE TABLE foo (
	id serial PRIMARY KEY, 
	description text
);

-- Table with journal data
CREATE TABLE foo_journal (
	foo_id int,
	revision int,
	changes changeset[],	
	created timestamp without time zone DEFAULT CURRENT_TIMESTAMP,
	processed timestamp without time zone NULL,
	PRIMARY KEY(foo_id, revision)
);

-- Function adds revision to foo_journal table from triggers on foo table
CREATE FUNCTION add_revision(p_foo_id int, p_changes changeset[]) RETURNS void AS $$
DECLARE
	p_revision int;
BEGIN
	-- Check the changes length
	IF coalesce(array_length(p_changes, 1), 0) = 0 THEN
		RETURN;
	END IF;
	SELECT coalesce(max(revision), 0) + 1 INTO p_revision FROM foo_journal WHERE foo_id = p_foo_id;
	INSERT INTO foo_journal(foo_id, revision, changes) VALUES (p_foo_id, p_revision, p_changes);	
END; $$ LANGUAGE plpgsql;

-- Trigger function for log insert action
CREATE FUNCTION tg_foo_insert() RETURNS trigger AS $$
BEGIN
	PERFORM add_revision(NEW.id, array[ROW('description', NULL, NEW.description)::changeset]);
	RETURN NEW;
END; $$ LANGUAGE plpgsql;
CREATE TRIGGER trg_foo_insert AFTER INSERT ON foo FOR EACH ROW EXECUTE PROCEDURE tg_foo_insert();

-- Trigger function for log update action
CREATE FUNCTION tg_foo_update() RETURNS trigger AS $$
BEGIN
	PERFORM add_revision(NEW.id, array[ROW('description', OLD.description, NEW.description)::changeset]);
	RETURN NEW;
END; $$ LANGUAGE plpgsql;
CREATE TRIGGER trg_foo_update AFTER UPDATE ON foo FOR EACH ROW EXECUTE PROCEDURE tg_foo_update();

-- Trigger function for log delete action
CREATE FUNCTION tg_foo_delete() RETURNS trigger AS $$
BEGIN
	PERFORM add_revision(OLD.id, array[ROW('description', OLD.description, NULL)::changeset]);
	RETURN OLD;
END; $$ LANGUAGE plpgsql;
CREATE TRIGGER trg_foo_delete BEFORE DELETE ON foo FOR EACH ROW EXECUTE PROCEDURE tg_foo_delete();

-- Insert test record
INSERT INTO foo(id, description) VALUES(1, 'install');

COMMIT;
