/*
 * pgstatindex
 *
 * Copyright (c) 2006 Satoshi Nagayasu <nagayasus@nttdata.co.jp>
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose, without fee, and without a
 * written agreement is hereby granted, provided that the above
 * copyright notice and this paragraph and the following two
 * paragraphs appear in all copies.
 *
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING
 * LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS
 * DOCUMENTATION, EVEN IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS
 * IS" BASIS, AND THE AUTHOR HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE,
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include "postgres.h"

#include "fmgr.h"
#include "funcapi.h"
#include "access/heapam.h"
#include "access/itup.h"
#include "access/nbtree.h"
#include "access/transam.h"
#include "catalog/namespace.h"
#include "catalog/pg_type.h"
#include "utils/builtins.h"
#include "utils/inval.h"

PG_FUNCTION_INFO_V1(bt_page);
PG_FUNCTION_INFO_V1(bt_item);
PG_FUNCTION_INFO_V1(bt_itemd);
PG_FUNCTION_INFO_V1(bt_metap);
PG_FUNCTION_INFO_V1(pg_relpages);
PG_FUNCTION_INFO_V1(pgstatindex);
PG_FUNCTION_INFO_V1(bt_page_items);

extern Datum bt_page(PG_FUNCTION_ARGS);
extern Datum bt_item(PG_FUNCTION_ARGS);
extern Datum bt_itemd(PG_FUNCTION_ARGS);
extern Datum bt_metap(PG_FUNCTION_ARGS);
extern Datum pg_relpages(PG_FUNCTION_ARGS);
extern Datum pgstatindex(PG_FUNCTION_ARGS);
extern Datum bt_page_items(PG_FUNCTION_ARGS);

#define BTPAGE_TYPE "public.__bt_page_type"
#define BTPAGE_NCOLUMNS 12

#define BTITEM_TYPE "public.__bt_item_type"
#define BTITEM_NCOLUMNS 8

#define BTMETAP_TYPE "public.__bt_metap_type"
#define BTMETAP_NCOLUMNS 6

#define BTITEMD_NCOLUMNS 8

#define PGSTATINDEX_NCOLUMNS 10

#define IS_INDEX(r) ((r)->rd_rel->relkind == 'i')
#define IS_BTREE(r) ((r)->rd_rel->relam == BTREE_AM_OID)

typedef struct BTPageStat {
		uint32 blkno;
		uint32 live_items;
		uint32 dead_items;
		uint32 page_size;
		uint32 max_avail;
		uint32 free_size;
		uint32 avg_item_size;
		uint32 fragments;
		bool is_root;
		bool is_internal;
		bool is_leaf;
		bool is_deleted;
		bool is_empty;
} BTPageStat;

typedef struct BTIndexStat {
		uint32 version;
		BlockNumber root_blkno;
		uint32 level;

		uint32 live_items;
		uint32 dead_items;

		uint32 root_pages;
		uint32 internal_pages;
		uint32 leaf_pages;
		uint32 empty_pages;
		uint32 deleted_pages;

		uint32 page_size;
		uint32 avg_item_size;

		uint32 max_avail;
		uint32 free_space;

		uint32 fragments;
} BTIndexStat;

/* -------------------------------------------------
 * GetBTPageStatistics()
 *
 * Collect statistics of a b-tree index leaf page
 * -------------------------------------------------
 */
static bool
GetBTPageStatistics(BlockNumber blkno, Buffer buffer, BTPageStat *stat)
{
	Page page           = BufferGetPage(buffer);
	PageHeader phdr     = (PageHeader) page;
	OffsetNumber maxoff = PageGetMaxOffsetNumber(page);
	BTPageOpaque opaque = (BTPageOpaque) PageGetSpecialPointer(page);
	int item_size = 0;
	int off;

	stat->max_avail = BLCKSZ - ( BLCKSZ - phdr->pd_special + SizeOfPageHeaderData );

	stat->dead_items = stat->live_items = 0;

	stat->page_size = PageGetPageSize(page);

	/* kind of page */
	stat->is_root    = false;
	stat->is_leaf    = false;
	stat->is_deleted = false;
	stat->is_empty   = false;

	if ( P_ISDELETED(opaque) )
	{
		stat->is_deleted = true;
		return true;
	}
	else if ( P_IGNORE(opaque) )
		stat->is_empty = true;
	else if ( P_ISLEAF(opaque) )
		stat->is_leaf = true;
	else if ( P_ISROOT(opaque) )
		stat->is_root = true;
	else
		stat->is_internal = true;

	/*----------------------------------------------
	 * If a next leaf is on the previous block,
	 * it means a fragmentation.
	 *----------------------------------------------
	 */
	stat->fragments = 0;
	if ( stat->is_leaf )
	{
		if ( opaque->btpo_next != P_NONE && opaque->btpo_next < blkno )
			stat->fragments++;
	}

	/* count live and dead tuples, and free space */
	for (off=FirstOffsetNumber ; off<=maxoff ; off++)
	{
		BTItem btitem;
		IndexTuple itup;

		ItemId id = PageGetItemId(page, off);

		btitem = (BTItem)PageGetItem(page, id);

		itup = &(btitem->bti_itup);

		item_size += IndexTupleSize(itup);

		if ( !ItemIdDeleted(id) )
			stat->live_items++;
		else
			stat->dead_items++;
	}
	stat->free_size = PageGetFreeSpace(page);

#ifdef NOT_USED
	elog(NOTICE, "%d/%d (%.2f%%)", stat->free_size, stat->max_avail,
		 (float)stat->free_size/(float)stat->max_avail*100.0);
#endif
		
	if ( (stat->live_items + stat->dead_items) > 0 )
		stat->avg_item_size = item_size / ( stat->live_items + stat->dead_items );
	else
		stat->avg_item_size = 0;

	return true;
}


/* ------------------------------------------------------
 * pgstatindex()
 *
 * Usage: SELECT * FROM pgstatindex('t1_pkey');
 * ------------------------------------------------------ 
 */
Datum
pgstatindex(PG_FUNCTION_ARGS)
{
	text	   *relname = PG_GETARG_TEXT_P(0);
	Relation	rel;
	RangeVar    *relrv;
	Datum		result;
	uint32		nblocks;
	uint32		blkno;
	BTIndexStat indexStat;

	relrv = makeRangeVarFromNameList(textToQualifiedNameList(relname));
	rel = relation_openrv(relrv, AccessShareLock);

	if ( !IS_INDEX(rel) || !IS_BTREE(rel) )
		elog(ERROR, "pgstatindex() can be used only on b-tree index.");

	/*
	 * Read a metapage
	 */
	{
		Buffer buffer = ReadBuffer(rel, 0);
		Page page = BufferGetPage(buffer);
		BTMetaPageData *metad = BTPageGetMeta(page);

//		snprintf(values[j++], 32, "%d", metad->btm_magic);
		indexStat.version    = metad->btm_version;
		indexStat.root_blkno = metad->btm_root;
		indexStat.level      = metad->btm_level;
//		snprintf(values[j++], 32, "%d", metad->btm_fastroot);
//		snprintf(values[j++], 32, "%d", metad->btm_fastlevel);

		ReleaseBuffer(buffer);
	}

	nblocks = RelationGetNumberOfBlocks(rel);

	/* -- init stat -- */
	indexStat.fragments      = 0;

	indexStat.root_pages     = 0;
	indexStat.leaf_pages     = 0;
	indexStat.internal_pages = 0;
	indexStat.empty_pages    = 0;
	indexStat.deleted_pages  = 0;

	indexStat.max_avail  = 0;
	indexStat.free_space = 0;

	/*
	 * Scan all blocks
	 */
	for (blkno=1 ; blkno<nblocks ; blkno++)
	{
		Buffer buffer = ReadBuffer(rel, blkno);
		BTPageStat stat;
		
		/* scan one page */
		stat.blkno = blkno;
		GetBTPageStatistics(blkno, buffer, &stat);

		/*
		 * page status (type)
		 */
		if ( stat.is_deleted )
			indexStat.deleted_pages++;
		else if ( stat.is_leaf )
			indexStat.leaf_pages++;
		else if ( stat.is_internal )
			indexStat.internal_pages++;
		else if ( stat.is_empty )
			indexStat.empty_pages++;
		else if ( stat.is_root )
			indexStat.root_pages++;
		else
			elog(ERROR, "unknown page status.");

		/* -- leaf fragmentation -- */
		indexStat.fragments += stat.fragments;

		if ( stat.is_leaf )
		{
			indexStat.max_avail  += stat.max_avail;
			indexStat.free_space += stat.free_size;
//			elog(NOTICE, "leaf density: %d/%d", indexStat.free_space, indexStat.max_avail);
		}

		ReleaseBuffer(buffer);
	}

	relation_close(rel, AccessShareLock);

#ifdef NOT_USED
	elog(NOTICE, "[index information]");
	elog(NOTICE, "version.................: %d", indexStat.version);
	elog(NOTICE, "tree level..............: %d", indexStat.level);
	elog(NOTICE, "index size..............: %d", (indexStat.root_pages +
									indexStat.leaf_pages +
									indexStat.internal_pages +
									indexStat.deleted_pages +
									indexStat.empty_pages) * BLCKSZ);

	elog(NOTICE, "");
	elog(NOTICE, "[page statistics]");
	elog(NOTICE, "root block number.......: %d", indexStat.root_blkno);
	elog(NOTICE, "internal pages..........: %d", indexStat.internal_pages);
	elog(NOTICE, "leaf pages..............: %d", indexStat.leaf_pages);
	elog(NOTICE, "empty pages.............: %d", indexStat.empty_pages);
	elog(NOTICE, "deleted pages...........: %d", indexStat.deleted_pages);

	elog(NOTICE, "");
	elog(NOTICE, "[leaf statistics]");
	elog(NOTICE, "avg. leaf density.......: %.2f%%", 100.0-(float)indexStat.free_space / (float)indexStat.max_avail * 100.0 );
	elog(NOTICE, "leaf fragmentation......: %.2f%%", (float)indexStat.fragments / 2.0 / (float)indexStat.leaf_pages * 100.0 );
#endif

	/*----------------------------
	 * Build a result tuple
	 *----------------------------
	 */
	{
		TupleDesc tupleDesc;
		int j;
		char *values[PGSTATINDEX_NCOLUMNS];

		HeapTupleData tupleData;
		HeapTuple tuple = &tupleData;

		tupleDesc = RelationNameGetTupleDesc("public.__pgstatindex_type");

		j = 0;
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", indexStat.version);
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", indexStat.level);
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", (indexStat.root_pages +
										 indexStat.leaf_pages +
										 indexStat.internal_pages +
										 indexStat.deleted_pages +
										 indexStat.empty_pages) * BLCKSZ );
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", indexStat.root_blkno);
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", indexStat.internal_pages);
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", indexStat.leaf_pages);
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", indexStat.empty_pages);
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", indexStat.deleted_pages);
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%.2f", 100.0-(float)indexStat.free_space / (float)indexStat.max_avail * 100.0 );
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%.2f", (float)indexStat.fragments / (float)indexStat.leaf_pages * 100.0 );

		tuple = BuildTupleFromCStrings(TupleDescGetAttInMetadata(tupleDesc),
									   values);
		
		result = TupleGetDatum(TupleDescGetSlot(tupleDesc), tuple);
	}

	PG_RETURN_DATUM( result );
}

/* -----------------------------------------------
 * bt_page()
 *
 * Usage: SELECT * FROM bt_page('t1_pkey', 0);
 * -----------------------------------------------
 */
Datum
bt_page(PG_FUNCTION_ARGS)
{
	text	   *relname = PG_GETARG_TEXT_P(0);
	uint32	   blkno   = PG_GETARG_UINT32(1);
	Buffer	   buffer;

	Relation	rel;
	RangeVar    *relrv;
	Datum		result;

	relrv = makeRangeVarFromNameList(textToQualifiedNameList(relname));
	rel = relation_openrv(relrv, AccessShareLock);

	if ( blkno >= RelationGetNumberOfBlocks(rel) )
		elog(ERROR, "A block number must be smaller than [relpages].");

	buffer = ReadBuffer(rel, blkno);

	if ( !IS_INDEX(rel) || !IS_BTREE(rel) )
		elog(ERROR, "bt_page() can be used only on b-tree index.");

	{
		HeapTuple tuple;
		TupleDesc tupleDesc;
		int j;
		char *values[12];

		BTPageStat stat;

		GetBTPageStatistics(blkno, buffer, &stat);

		tupleDesc = RelationNameGetTupleDesc(BTPAGE_TYPE);

		j = 0;
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", stat.blkno);
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", stat.live_items);
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", stat.dead_items);
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", stat.live_items + stat.dead_items);
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", stat.avg_item_size);
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", stat.page_size);
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", stat.free_size);
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", stat.is_deleted);
		values[j] = palloc(32);
//		snprintf(values[j++], 32, "%d", opaque->btpo_prev);
		snprintf(values[j++], 32, "%d", 0);
		values[j] = palloc(32);
//		snprintf(values[j++], 32, "%d", opaque->btpo_next);
		snprintf(values[j++], 32, "%d", 0);
		values[j] = palloc(32);
//		snprintf(values[j++], 32, "%d", opaque->btpo.level);
		snprintf(values[j++], 32, "%d", 0);
		values[j] = palloc(32);
//		snprintf(values[j++], 32, "%d", opaque->btpo_flags);
		snprintf(values[j++], 32, "%d", 0);

		tuple = BuildTupleFromCStrings(TupleDescGetAttInMetadata(tupleDesc),
									   values);

		result = TupleGetDatum(TupleDescGetSlot(tupleDesc), tuple);
	}

	ReleaseBuffer(buffer);

	relation_close(rel, AccessShareLock);

	PG_RETURN_DATUM(result);
}

/* -----------------------------------------------
 * bt_item: 
 *
 * Usage: SELECT * FROM bt_item('t1_pkey', 0, 1);
 * -----------------------------------------------
 */
Datum
bt_item(PG_FUNCTION_ARGS)
{
	text	   *relname = PG_GETARG_TEXT_P(0);
	uint32	   blkno   = PG_GETARG_UINT32(1);
	uint16	   offset  = PG_GETARG_UINT16(2);
	Buffer	   buffer;

	Relation	rel;
	RangeVar    *relrv;
	Datum		result;
	bool		live;
	TupleDesc tupleDesc;
	IndexTuple itup;
	BTItem btitem;
	char *values[BTITEM_NCOLUMNS];
	BTPageOpaque opaque;
	Page page;
	HeapTuple tuple;
	ItemId id;

	relrv = makeRangeVarFromNameList(textToQualifiedNameList(relname));
	rel = relation_openrv(relrv, AccessShareLock);

	if ( blkno >= RelationGetNumberOfBlocks(rel) )
		elog(ERROR, "A block number must be smaller than [relpages].");

	buffer = ReadBuffer(rel, blkno);

	if ( !IS_INDEX(rel) || !IS_BTREE(rel) )
		elog(ERROR, "bt_item() can be used only on b-tree index.");

	page = BufferGetPage(buffer);
	
	if ( !(FirstOffsetNumber<=offset && offset<=PageGetMaxOffsetNumber(page)) )
		elog(ERROR, "An offset number out of range.(%d,%d)",
			 offset, PageGetMaxOffsetNumber(page));
	
	opaque = (BTPageOpaque) PageGetSpecialPointer(page);
	
	if ( P_ISDELETED(opaque) )
			elog(NOTICE, "pgstatpageii(): this page is deleted.");
	
	id = PageGetItemId(page, offset);

	btitem = (BTItem)PageGetItem(page, id);
	itup = &(btitem->bti_itup);
	
	if ( ItemIdDeleted(id) )
		live = false;
	else
		live = true;

//	elog(NOTICE, "blkno=%d,offset=%d,live=%d,bi_hi=%d,bi_lo=%d",
//		 blkno, offset, live,
//		 itup->t_tid.ip_blkid.bi_hi, itup->t_tid.ip_blkid.bi_lo);

	tupleDesc = RelationNameGetTupleDesc(BTITEM_TYPE);
		
	{
		int j = 0;
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", blkno);
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", offset);
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", live);
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", itup->t_tid.ip_blkid.bi_hi);
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", itup->t_tid.ip_blkid.bi_lo);
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", IndexTupleSize(itup));
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", IndexTupleHasNulls(itup));
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", IndexTupleHasVarwidths(itup));
	}
	
	tuple = BuildTupleFromCStrings(TupleDescGetAttInMetadata(tupleDesc),
								   values);
	
	result = TupleGetDatum(TupleDescGetSlot(tupleDesc), tuple);

	ReleaseBuffer(buffer);

	relation_close(rel, AccessShareLock);

	PG_RETURN_DATUM(result);
}

/* ------------------------------------------------------------------
 * bt_itemd: Get a single IndexTupleData specified by a page number
 *           and an item offset
 *
 * Usage: SELECT * FROM bt_itemd('t1_pkey', 1, 4);
 * ------------------------------------------------------------------
 */
Datum
bt_itemd(PG_FUNCTION_ARGS)
{
	text	   *relname = PG_GETARG_TEXT_P(0);
	uint32	   blkno   = PG_GETARG_UINT32(1);
	uint16	   offset  = PG_GETARG_UINT16(2);
	Buffer	   buffer;

	Relation	rel;
	RangeVar    *relrv;
	TupleDesc tupleDesc;
	IndexTuple itup;
	BTItem btitem;
	char *result;
	BTPageOpaque opaque;
	Page page;
	ItemId id;

	relrv = makeRangeVarFromNameList(textToQualifiedNameList(relname));
	rel = relation_openrv(relrv, AccessShareLock);

	if ( blkno >= RelationGetNumberOfBlocks(rel) )
		elog(ERROR, "A block number must be smaller than [relpages].");

	buffer = ReadBuffer(rel, blkno);

	if ( !IS_INDEX(rel) || !IS_BTREE(rel) )
		elog(ERROR, "bt_itemd() can be used only on b-tree index.");

	page = BufferGetPage(buffer);
	
	if ( !(FirstOffsetNumber<=offset && offset<=PageGetMaxOffsetNumber(page)) )
		elog(ERROR, "An offset number out of range.(%d,%d)",
			 offset, PageGetMaxOffsetNumber(page));
	
	opaque = (BTPageOpaque) PageGetSpecialPointer(page);
	
	if ( P_ISDELETED(opaque) )
		elog(NOTICE, "bt_itemd(): this page is deleted.");
	
	id = PageGetItemId(page, offset);

	btitem = (BTItem)PageGetItem(page, id);
	itup = &(btitem->bti_itup);
	
	tupleDesc = CreateTemplateTupleDesc(BTITEMD_NCOLUMNS, false);

	{
		int off;
///		char *ptr = (itup+IndexInfoFindDataOffset(itup));
		char *ptr = (char *)itup + IndexInfoFindDataOffset(itup->t_info);

		result = palloc(IndexTupleSize(itup)*3);
		memset(result, 0, IndexTupleSize(itup)*3);

//		elog(NOTICE,"IndexTupleSize=%d", IndexTupleSize(itup));

		for (off=0 ;
			 off < IndexTupleSize(itup)-IndexInfoFindDataOffset(itup->t_info) ;
			 off++)
		{
			if ( result[0]=='\0' )
				sprintf(result, "%02x", *(ptr+off) & 0xff);
			else
			{
				char buf[4];
				sprintf(buf, " %02x", *(ptr+off) & 0xff);
				strcat(result, buf);
			}
//			elog(NOTICE, "d=%02x",  *(ptr+off));
		}
	}
	
	ReleaseBuffer(buffer);

	relation_close(rel, AccessShareLock);

	PG_RETURN_TEXT_P(DirectFunctionCall1(textin, CStringGetDatum(result)));
}

/*-------------------------------------------------------
 * bt_page_items: Get IndexTupleData set in a leaf page
 *
 * Usage: SELECT * FROM bt_page_items('t1_pkey', 0);
 *-------------------------------------------------------
 */
struct user_args
{
	TupleDesc tupd;
	Relation rel;
	Buffer buffer;
	Page page;
	uint16 offset;
};

#define BTPAGEITEMS_NCOLUMNS 5

Datum
bt_page_items(PG_FUNCTION_ARGS)
{
	text	   *relname = PG_GETARG_TEXT_P(0);
	uint32	   blkno   = PG_GETARG_UINT32(1);

	RangeVar    *relrv;
	Datum result;
	char *values[BTPAGEITEMS_NCOLUMNS];
	BTPageOpaque opaque;
	HeapTuple tuple;
	ItemId id;

	FuncCallContext *fctx;
	MemoryContext mctx;
	struct user_args *uargs = NULL;

	if ( SRF_IS_FIRSTCALL() )
	{
		fctx = SRF_FIRSTCALL_INIT();
		mctx = MemoryContextSwitchTo(fctx->multi_call_memory_ctx);

		uargs = palloc(sizeof(struct user_args));

        uargs->tupd = RelationNameGetTupleDesc("public.__bt_page_items_type");
		uargs->offset = FirstOffsetNumber;

		relrv = makeRangeVarFromNameList(textToQualifiedNameList(relname));
		uargs->rel = relation_openrv(relrv, AccessShareLock);

		if ( blkno >= RelationGetNumberOfBlocks(uargs->rel) )
			elog(ERROR, "A block number must be smaller than [relpages].");

		uargs->buffer = ReadBuffer(uargs->rel, blkno);

		if ( !IS_INDEX(uargs->rel) || !IS_BTREE(uargs->rel) )
			elog(ERROR, "bt_page_items() can be used only on b-tree index.");

		uargs->page = BufferGetPage(uargs->buffer);
	
		opaque = (BTPageOpaque) PageGetSpecialPointer(uargs->page);
		
		if ( P_ISDELETED(opaque) )
			elog(NOTICE, "bt_page_items(): this page is deleted.");

		fctx->max_calls = PageGetMaxOffsetNumber(uargs->page);
		fctx->user_fctx = uargs;

		MemoryContextSwitchTo(mctx);
	}
	
	fctx = SRF_PERCALL_SETUP();
	uargs = fctx->user_fctx;

	if ( fctx->call_cntr < fctx->max_calls )
	{
		BTItem btitem;
		IndexTuple itup;

		id = PageGetItemId(uargs->page, uargs->offset);

		if ( !ItemIdIsValid(id) )
			elog(ERROR, "Invalid ItemId.");

		btitem = (BTItem)PageGetItem(uargs->page, id);
		itup = &(btitem->bti_itup);
		
		{
			int j = 0;

			BlockNumber blockno = BlockIdGetBlockNumber(&(itup->t_tid.ip_blkid));

			values[j] = palloc(32);
			snprintf(values[j++], 32, "%d", uargs->offset);
			values[j] = palloc(32);
			snprintf(values[j++], 32, "(%u,%u)", blockno, itup->t_tid.ip_posid);
			values[j] = palloc(32);
			snprintf(values[j++], 32, "%d", IndexTupleSize(itup));
			values[j] = palloc(32);
			snprintf(values[j++], 32, "%c", IndexTupleHasNulls(itup) ? 't' : 'f');
			values[j] = palloc(32);
			snprintf(values[j++], 32, "%c", IndexTupleHasVarwidths(itup) ? 't' : 'f');

			tuple = BuildTupleFromCStrings(TupleDescGetAttInMetadata(uargs->tupd), values);
			result = TupleGetDatum(TupleDescGetSlot(uargs->tupd), tuple);
		}

		uargs->offset = uargs->offset + 1;

		SRF_RETURN_NEXT(fctx, result);
	}
	else
	{
		ReleaseBuffer(uargs->buffer);
		relation_close(uargs->rel, AccessShareLock);

		SRF_RETURN_DONE(fctx);
	}
}


/* ------------------------------------------------
 * bt_metap: Get a b-tree meta-page information
 *
 * Usage: SELECT * FROM bt_metap('t1_pkey')
 * ------------------------------------------------
 */
Datum
bt_metap(PG_FUNCTION_ARGS)
{
	text	   *relname = PG_GETARG_TEXT_P(0);
	Buffer	   buffer;

	Relation	rel;
	RangeVar    *relrv;
	Datum		result;

	relrv = makeRangeVarFromNameList(textToQualifiedNameList(relname));
	rel = relation_openrv(relrv, AccessShareLock);

	if ( !IS_INDEX(rel) || !IS_BTREE(rel) )
		elog(ERROR, "bt_metap() can be used only on b-tree index.");

	buffer = ReadBuffer(rel, 0);

	{
		BTMetaPageData *metad;

		TupleDesc tupleDesc;
		int j;
		char *values[BTMETAP_NCOLUMNS];
		HeapTuple tuple;

		Page page = BufferGetPage(buffer);

		metad = BTPageGetMeta(page);

		tupleDesc = RelationNameGetTupleDesc(BTMETAP_TYPE);

		j = 0;
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", metad->btm_magic);
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", metad->btm_version);
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", metad->btm_root);
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", metad->btm_level);
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", metad->btm_fastroot);
		values[j] = palloc(32);
		snprintf(values[j++], 32, "%d", metad->btm_fastlevel);

		tuple = BuildTupleFromCStrings(TupleDescGetAttInMetadata(tupleDesc),
									   values);

		result = TupleGetDatum(TupleDescGetSlot(tupleDesc), tuple);
	}

	ReleaseBuffer(buffer);

	relation_close(rel, AccessShareLock);

	PG_RETURN_DATUM(result);
}

/* --------------------------------------------------------
 * pg_relpages: Get a number of pages of the table/index.
 *
 * Usage: SELECT pg_relpages('t1');
 *        SELECT pg_relpages('t1_pkey');
 * --------------------------------------------------------
 */
Datum
pg_relpages(PG_FUNCTION_ARGS)
{
	text	   *relname = PG_GETARG_TEXT_P(0);

	Relation	rel;
	RangeVar    *relrv;
	int4  relpages;

	relrv = makeRangeVarFromNameList(textToQualifiedNameList(relname));
	rel = relation_openrv(relrv, AccessShareLock);

	relpages = RelationGetNumberOfBlocks(rel);

	relation_close(rel, AccessShareLock);

	PG_RETURN_INT32(relpages);
}

