/*
 * $Id:$
 *
 * Routines to create a file suitable for reading by the COPY BINARY
 * command. Supports the creation of 2 file formats: if POSTGRESQL_71
 * is defined then files compatible with PostgreSQL 7.1 through to 7.3
 * will be created; if POSTGRESQL_74 is defined then files compatible
 * with PostgreSQL 7.4 and likely later versions will be created.
 *
 * Note, these sources assume you have a 64 bit integer type int64_t
 * and that it "works"...
 *
 * Lee Kindness <lkindness@csl.co.uk> 2003/08/05
 */

#include "postgres.h"

#include <sys/types.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <time.h>
#include <netinet/in.h>

#include "binarycopy.h"

/*
 * User should define either POSTGRESQL_71 (for PostgreSQL 7.1-7.3 format
 * dumps) or POSTGRESQL_74 (for PostgreSQL 7.1-> format dumps) at compile
 * time. If not specified then default to the newest.
 */
#if !defined POSTGRESQL_71 && !defined POSTGRESQL_74
# define POSTGRESQL_74
#endif /* !defined POSTGRESQL_71 && !defined POSTGRESQL_74 */

/*
 * Within CSL this code should use specific debug routines, but since this
 * is in the PostgreSQL contrib area better make sure everyone can compile!
 */
#ifdef USE_CSL_DEBUG
# include "csl_debug.h"
# define DEBUGOUT(M)        csl_debug(CSL_DEBUG_MSG_ERROR,(M))
# define DEBUGOUT1(M,A1)    csl_debug(CSL_DEBUG_MSG_ERROR,(M),(A1))
#else /* !USE_CSL_DEBUG */
# define DEBUGOUT(M)        fprintf(stderr,(M))
# define DEBUGOUT1(M,A1)    fprintf(stderr,(M),(A1))
#endif /* !USE_CSL_DEBUG */

static void bcBulkWrite(FILE *f, void *data,
			      size_t sz, size_t count, short ind);

/*
 * bcBulkOpen - Open a temporary file, returning the file handle and its
 * filename in *filename. NULL is returned in error.
 *
 * In the future an additional open call will probably be supplied to
 * allow setting of flags and the extended header, however the usefulness
 * of this just now is limited (to including OIDs) so best leave it until
 * an actual extended header exists!
 */
FILE *bcBulkOpen(char **filename)
{
#if defined POSTGRESQL_74
  char header[] = "PGCOPY\n\377\r\n\0";
  int32_t flags_field = htonl(0);
  int32_t ext_area_length = htonl(0);
#elif defined POSTGRESQL_71
  char header[] = "PGBCOPY\n\377\r\n\0";
  int32_t layout_field = 0x01020304;
  int32_t flags_field = 0;
  int32_t ext_area_length = 0;
#endif
  char fn[] = "/tmp/lofsXXXXXX";
  int fd;
  FILE *f;

  /* get a temporary file */
  fd = mkstemp(fn);
  if( fd == -1 )
    {
      DEBUGOUT1("Error in bcBulkOpen() building temp filename: %d\n", errno);
      *filename = NULL;
      return( NULL );
    }

  /* get a FILE* handle to it */
  *filename = strdup(fn);
  f = fdopen(fd, "w");
  if( f == NULL )
    {
      DEBUGOUT1("Error in bcBulkOpen() opening temp file: %d\n", errno);
      return( NULL );
    }

  /* and write out the file header */
  fwrite(&header,          sizeof(char), strlen(header) + 1, f);
#if defined POSTGRESQL_71
  fwrite(&layout_field,    sizeof(int32_t), 1,              f);
#endif
  fwrite(&flags_field,     sizeof(int32_t), 1,              f);
  fwrite(&ext_area_length, sizeof(int32_t), 1,              f);

  return( f );
}

/*
 * bcBulkClose - Write out EOF trailer, close file, and ensure it's
 * world readable.
 */
void bcBulkClose(FILE *f, char *filename)
{
  /* write out "EOF" */
#if defined POSTGRESQL_74
  int16_t trailer = htons(-1);
#elif defined POSTGRESQL_71
  int16_t trailer = -1;
#endif
  fwrite(&trailer, sizeof(int16_t), 1, f);

  fclose(f);

  chmod(filename, S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH|S_IWOTH);
}

/*
 * bcBulkFree - Delete file and free the filename.
 */
void bcBulkFree(char **filename, char *table)
{
  if( filename != NULL )
    {
      if( *filename != NULL )
	{
	  unlink(*filename);
	  free(*filename);
	}
    }
  *filename = NULL;
}

/*
 * bcBulkWriteNCols - Write out number of columns which will be written
 * in this row.
 */
void bcBulkWriteNCols(FILE *f, int ncols)
{
#if defined POSTGRESQL_74
  int16_t nc = htons(ncols);
#elif defined POSTGRESQL_71
  int16_t nc = ncols;
#endif
  fwrite(&nc, sizeof(int16_t), 1, f);
}

/*
 * bcBulkWrite - Write data for a column, generic and used by the other
 * calls (especially for 7.1 format files). If ind == -1 then the data
 * is NULL.
 */
static void bcBulkWrite(FILE *f, void *data, size_t sz, size_t count, short ind)
{
  /* write out size of coming data */
#if defined POSTGRESQL_74
  int32_t totalsz = htonl((ind == -1) ? -1 : (int32_t)(sz * count));
  fwrite(&totalsz, sizeof(int32_t), 1, f);
#elif defined POSTGRESQL_71
  int16_t totalsz = (ind == -1) ? 0 : (sz * count);
  fwrite(&totalsz, sizeof(int16_t), 1, f);
#endif

  /* if we're not writing a NULL write the data... */
  if( ind != -1 )
    {
      fwrite(data, sz, count, f);
    }
}

/*
 * bcBulkWriteText - Write data for variable text fields - TEXT, VARCHAR
 * etc. If ind == -1 then the data is NULL.
 */
void bcBulkWriteText(FILE *f, char *data, short ind)
{
#if defined POSTGRESQL_74
  if( ind == -1 )
    {
      const int32_t sz = htonl(-1);
      fwrite(&sz, sizeof(int32_t), 1, f);
    }
  else
    {
      const int32_t sz = htonl(sizeof(char) * strlen(data));

      /* write the length... */
      fwrite(&sz, sizeof(int32_t), 1, f);

      /* and the text itself... */
      fwrite(data, sizeof(char), strlen(data), f);
    }
#elif defined POSTGRESQL_71
  if( ind == -1 )
    {
      const int16_t sz = 0;
      fwrite(&sz, sizeof(int16_t), 1, f);
    }
  else
    {
      const int16_t sz = -1;
      const int32_t varlena = sizeof(int32_t) + (sizeof(char) * strlen(data));

      /* write -1 to signify variable length text follows */
      fwrite(&sz, sizeof(int16_t), 1, f);

      /* and the length... */
      fwrite(&varlena, sizeof(int32_t), 1, f);

      /* and the text itself... */
      fwrite(data, sizeof(char), strlen(data), f);
    }
#endif
}

/*
 * bcBulkWriteBytea - Write data for BYTEA fields. If ind == -1 then
 * the data is NULL.
 */
void bcBulkWriteBytea(FILE *f, char *data, size_t len, short ind)
{
#if defined POSTGRESQL_74
  if( ind == -1 )
    {
      const int32_t sz = htonl(-1);
      fwrite(&sz, sizeof(int32_t), 1, f);
    }
  else
    {
      const int32_t sz = htonl(sizeof(char) * len);

      /* write the length... */
      fwrite(&sz, sizeof(int32_t), 1, f);

      /* and the text itself... */
      fwrite(data, sizeof(char), len, f);
    }
#elif defined POSTGRESQL_71
  if( ind == -1 )
    {
      const int16_t sz = 0;
      fwrite(&sz, sizeof(int16_t), 1, f);
    }
  else
    {
      const int16_t sz = -1;
      const int32_t varlena = sizeof(int32_t) + (sizeof(char) * len);

      /* write -1 to signify variable length text follows */
      fwrite(&sz, sizeof(int16_t), 1, f);

      /* and the length... */
      fwrite(&varlena, sizeof(int32_t), 1, f);

      /* and the bytes themselves... */
      fwrite(data, sizeof(char), len, f);
    }
#endif
}

/*
 * bcBulkWriteInt16 - Write data for SMALLINT 16-bit/2-byte integer (an
 * integer) fields. If ind == -1 then the data is NULL.
 */
void bcBulkWriteInt16(FILE *f, int16_t i, short ind)
{
#if defined POSTGRESQL_74
  i = htons(i);
  bcBulkWrite(f, &i, sizeof(i), 1, ind);
#elif defined POSTGRESQL_71
  bcBulkWrite(f, &i, sizeof(i), 1, ind);
#endif
}

/*
 * bcBulkWriteInt32 - Write data for INTEGER 32-bit/4-byte integer (a
 * long) fields. If ind == -1 then the data is NULL.
 */
void bcBulkWriteInt32(FILE *f, int32_t i, short ind)
{
#if defined POSTGRESQL_74
  i = htonl(i);
  bcBulkWrite(f, &i, sizeof(i), 1, ind);
#elif defined POSTGRESQL_71
  bcBulkWrite(f, &i, sizeof(i), 1, ind);
#endif
}

/*
 * bcBulkWriteInt64 - Write data for BIGINT 64-bit/8-byte integer (a
 * long long) fields. If ind == -1 then the data is NULL.
 */
void bcBulkWriteInt64(FILE *f, int64_t i, short ind)
{
#if defined POSTGRESQL_74
  uint32_t n32[2];
  n32[0] = htonl((uint32_t)(i >> 32));
  n32[1] = htonl((uint32_t)i);
  bcBulkWrite(f, &n32[0], sizeof(uint32_t), 2, ind);
#elif defined POSTGRESQL_71
  bcBulkWrite(f, &i, sizeof(i), 1, ind);
#endif
}

/*
 * bcBulkWriteFloat4 - Write data for REAL 4-byte floating point (a
 * float) fields. If ind == -1 then the data is NULL.
 */
void bcBulkWriteFloat4(FILE *f, float d, short ind)
{
#if defined POSTGRESQL_74
  union {
    float   f;
    int32_t i;
  } swap;
  swap.f = d;
  swap.i = htonl(swap.i);
  bcBulkWrite(f, &swap.i, sizeof(swap.i), 1, ind);
#elif defined POSTGRESQL_71
  bcBulkWrite(f, &d, sizeof(d), 1, ind);
#endif
}

/*
 * bcBulkWriteFloat8 - Write data for DOUBLE PRECISION 8-byte floating
 * point (a double) fields. If ind == -1 then the data is NULL.
 */
void bcBulkWriteFloat8(FILE *f, double d, short ind)
{
#if defined POSTGRESQL_74
  union {
    double  f;
    int64_t i;
  } swap;
  swap.f = d;
  bcBulkWriteInt64(f, swap.i, ind);
#elif defined POSTGRESQL_71
  bcBulkWrite(f, &d, sizeof(d), 1, ind);
#endif
}

/*
 * bcBulkWriteTime - Write data for TIMESTAMP fields. If ind == -1 then
 * the data is NULL. If the time is already in double format and based
 * on the PostgreSQL epoch then bcBulkWriteFloat8 should be used instead
 * (but only if you didn;t specify integer timestamps during the build!).
 */
void bcBulkWriteTime(FILE *f, time_t t, short ind)
{
  /* convert 't' to the PostgreSQL epoch (01/01/2000 based)
     and output as a float/long long */
#ifdef HAVE_INT64_TIMESTAMP
  bcBulkWriteInt64 (f, (int64_t)t - 946684800,   ind);
#else
  bcBulkWriteFloat8(f, (double) t - 946684800.0, ind);
#endif
}
