#ifndef __BINARYCOPY_H__
#define __BINARYCOPY_H__

/*
 * $Id:$
 *
 * Routines to create a file suitable for reading by the COPY BINARY
 * command. Supports the creation of 2 file formats: if POSTGRESQL_71
 * is defined then files compatible with PostgreSQL 7.1 through to 7.3
 * will be created; if POSTGRESQL_74 is defined then files compatible
 * with PostgreSQL 7.4 and likely later versions will be created.
 *
 * Lee Kindness <lkindness@csl.co.uk> 2003/08/05
 *
 */

#include <sys/types.h>
#include <stdio.h>
#include <time.h>

/*
 * bcBulkOpen - Open a temporary file, returning the file handle and its
 * filename in *filename. NULL is returned in error.
 *
 * In the future an additional open call will probably be supplied to
 * allow setting of flags and the extended header, however the usefulness
 * of this just now is limited (to including OIDs) so best leave it until
 * an actual extended header exists!
 */
extern FILE *bcBulkOpen(char **filename);

/*
 * bcBulkClose - Write out EOF trailer, close file, and ensure it's
 * world readable.
 */
extern void bcBulkClose(FILE *f, char *filename);

/*
 * bcBulkFree - Delete file and free the filename.
 */
extern void bcBulkFree(char **filename, char *table);

/*
 * bcBulkWriteNCols - Write out number of columns which will be written
 * in this row.
 */
extern void bcBulkWriteNCols(FILE *f, int ncols);

/*
 * bcBulkWriteText - Write data for variable text fields - TEXT, VARCHAR
 * etc. If ind == -1 then the data is NULL.
 */
extern void bcBulkWriteText(FILE *f, char *data, short ind);

/*
 * bcBulkWriteBytea - Write data for BYTEA fields. If ind == -1 then
 * the data is NULL.
 */
extern void bcBulkWriteBytea(FILE *f, char *data, size_t len, short ind);

/*
 * bcBulkWriteInt16 - Write data for SMALLINT 16-bit/2-byte integer (an
 * integer) fields. If ind == -1 then the data is NULL.
 */
extern void bcBulkWriteInt16(FILE *f, int16_t i, short ind);

/*
 * bcBulkWriteInt32 - Write data for INTEGER 32-bit/4-byte integer (a
 * long) fields. If ind == -1 then the data is NULL.
 */
extern void bcBulkWriteInt32(FILE *f, int32_t i, short ind);

/*
 * bcBulkWriteInt64 - Write data for BIGINT 64-bit/8-byte integer (a
 * long long) fields. If ind == -1 then the data is NULL.
 */
extern void bcBulkWriteInt64(FILE *f, int64_t i, short ind);

/*
 * bcBulkWriteFloat4 - Write data for REAL 4-byte floating point (a
 * float) fields. If ind == -1 then the data is NULL.
 */
extern void bcBulkWriteFloat4(FILE *f, float d, short ind);

/*
 * bcBulkWriteFloat8 - Write data for DOUBLE PRECISION 8-byte floating
 * point (a double) fields. If ind == -1 then the data is NULL.
 */
extern void bcBulkWriteFloat8(FILE *f, double d, short ind);

/*
 * bcBulkWriteTime - Write data for TIMESTAMP fields. If ind == -1 then
 * the data is NULL. If the time is already in double format and based
 * on the PostgreSQL epoch then bcBulkWriteFloat8 should be used instead
 * (but only if you didn;t specify integer timestamps during the build!).
 *
 * bcBulkWriteTimeNow is a convenience macro to output the current time.
 */
extern void bcBulkWriteTime(FILE *f, time_t t, short ind);
#define bcBulkWriteTimeNow(f) bcBulkWriteTime(f, time(NULL), 0)

#endif /* ! __BINARYCOPY_H__ */
