# Copyright (c) 2014 Rock Neurotiko
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
import netifaces

from libqtile.log_utils import logger
from libqtile.widget import base

from typing import Tuple
from math import log


class NetIP(base.ThreadedPollText):
    """Displays interface status and IPv4 address"""
    orientations = base.ORIENTATION_HORIZONTAL
    defaults = [
        ('interface', None, 'List of interfaces or single NIC as string to monitor, \
            None to displays all active NICs combined'),
        ('update_interval', 5, 'The update interval.'),
    ]

    def __init__(self, **config):
        base.ThreadedPollText.__init__(self, **config)
        self.add_defaults(NetIP.defaults)
        self.set_interfaces()

    def set_interfaces(self):

        if not isinstance(self.interface, list):
            if self.interface is None:
                ifaces = netifaces.interfaces()
                #no one cares about loopback
                ifaces.remove('lo')
                self.interfaces = ifaces
            elif isinstance(self.interface, str):
                self.interfaces = [self.interface]
            else:
                raise AttributeError("Invalid Argument passed: %s\nAllowed Types: List, String, None" % self.interface)

    def poll(self):
        self.set_interfaces()
        ret_stat = []
        for intf in self.interfaces:
        
            try:
                addr = netifaces.ifaddresses(intf)

                #Unable to get IP address
                if not addr:
                    ret_stat.append("{0} : X.X.X.X".format(intf))
                    continue

                ip_addrs = [ip['addr'] for ip in addr[netifaces.AF_INET]]
                ret_stat.append("{0} : {1}".format(intf, ",".join(ip_addrs)))


            except Exception as excp:
                logger.error('%s: Caught Exception:\n%s',
                             self.__class__.__name__, excp)

        return " | ".join(ret_stat)
