package test.org.apache.regexp;

/*
 * ====================================================================
 * 
 * The Apache Software License, Version 1.1
 * 
 * Copyright (c) 1999 The Apache Software Foundation.  All rights
 * reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in
 * the documentation and/or other materials provided with the
 * distribution.
 * 
 * 3. The end-user documentation included with the redistribution, if
 * any, must include the following acknowlegement:
 * "This product includes software developed by the
 * Apache Software Foundation (http://www.apache.org/)."
 * Alternately, this acknowlegement may appear in the software itself,
 * if and wherever such third-party acknowlegements normally appear.
 * 
 * 4. The names "The Jakarta Project", "Jakarta-Regexp", and "Apache Software
 * Foundation" must not be used to endorse or promote products derived
 * from this software without prior written permission. For written
 * permission, please contact apache@apache.org.
 * 
 * 5. Products derived from this software may not be called "Apache"
 * nor may "Apache" appear in their names without prior written
 * permission of the Apache Group.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 * 
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 * 
 */

import java.io.*;
import java.util.*;
import junit.framework.*;
import org.apache.regexp.*;

import java.net.URL;

/**
 * JUnit tests for the basic <code>RE.match()</code> functionality.  It is
 * intended that these be created from a test file, but this is not required.
 * The file is to consist of records each separated by a single blank
 * line.  Here is the format of the records:
 * <p>
 * <pre>
 * name
 * regular expression pattern
 * ERR (if regexp is intended to fail compilation) or target text to
 * match (if the regexp is intended to compile successfully)
 * YES if the target text is to match the regexp, NO otherwise
 * if YES, then a list of the expected values of the paren registers
 * after the match, one per line
 * </pre>
 * If a regexp pattern line or paren register line begins with the
 * string <code>"---empty"</code>, it is interpreted as the empty string.
 * <p>
 * @author <a href="mailto:pholser@mindspring.com">Paul Holser</a>
 */

public class BasicREMatchTest extends TestCase
{
  private RE re = new RE();
  private REDebugCompiler compiler = new REDebugCompiler();
  private String rePattern;
  private boolean patternShouldCompile;
  private String targetText;
  private boolean targetShouldMatchPattern;
  private String[] expectedParenRegisters;

/**
 * @param  name  name of the test
 * @param  rePattern  regular expression pattern for the test fixture
 * @param  patternShouldCompile  whether the rePattern is intended to
 * compile
 * @param  targetText  if patternShouldCompile, the target text to
 * match against the pattern
 * @param  targetShouldMatchPattern  whether targetText should match
 * rePattern
 * @param  expectedParenRegisters  if targetShouldMatchPattern, the
 * anticipated values of the paren registers after the match
 * @exception  IllegalArgumentException  if rePattern is null
 */

public BasicREMatchTest(
	String name, 
	String rePattern, 
	boolean patternShouldCompile, 
	String targetText, 
	boolean targetShouldMatchPattern, 
	String[] expectedParenRegisters)
{
	super(name);

	if (rePattern == null)
	{
		throw new IllegalArgumentException("null rePattern");
	}

	this.rePattern = rePattern;
	this.patternShouldCompile = patternShouldCompile;
	this.targetText = targetText;
	this.targetShouldMatchPattern = targetShouldMatchPattern;

	if (expectedParenRegisters == null)
	{
		this.expectedParenRegisters = new String[0];
	}
	else
	{
		this.expectedParenRegisters = expectedParenRegisters;
	}
}


/**
 * Run the test.
 */

public void runTest()
{
	try
	{
		re.setProgram(compiler.compile(rePattern));
	}
	catch (RESyntaxException e)
	{
		if (patternShouldCompile)
		{
			fail("Couldn't construct an RE: " + e.getMessage());
		}

		return;
	}

	assert("match result", targetShouldMatchPattern == re.match(targetText));
	assertEquals(
		"need expected number of paren registers", 
		expectedParenRegisters.length, 
		re.getParenCount()); 

	for (int i = 0; i < re.getParenCount(); i++)
	{
		assertEquals(
			"paren register " + i + " should match", 
			expectedParenRegisters[i], 
			re.getParen(i)); 
	}
}


/**
 * Create a suite of BasicREMatchTests from a file with the given name.
 *
 * @param  fileName  name of the data file
 * @return  a suite of tests
 * @exception  IOException  if there are problems with the data file
 */

public static Test suite(String fileName) throws IOException
{
	return suite(new File(fileName));
}


/**
 * Create a suite of BasicREMatchTests from a File.
 *
 * @param  fileName  the File to read
 * @return  a suite of tests
 * @exception  IOException  if there are problems with the data file
 */

public static Test suite(File file) throws IOException
{
	return suite(new FileReader(file));
}


private static BasicREMatchTest getNextTest(LineNumberReader reader)
throws IOException
{
	String nameLine = "";
	String errorOrTargetTextLine = null;
	String targetTextLine = null;
	String yesNoLine = null;
	String patternLine = null;
	boolean patternShouldCompileTemp = true;
	boolean targetShouldMatchTemp = true;
	String[] expectedParenRegistersTemp = null;

	nameLine = reader.readLine();

	if (nameLine == null)
	{
		throw new IOException(
			"Expecting a test case name at line " + reader.getLineNumber()); 
	}

	nameLine = nameLine.trim();

	patternLine = reader.readLine();
	errorOrTargetTextLine = reader.readLine().trim();

	if (errorOrTargetTextLine.equals("ERR"))
	{
		patternShouldCompileTemp = false;

		String skipBlankLine = reader.readLine();
	}
	else
	{
		targetTextLine = errorOrTargetTextLine;

		if (targetTextLine.startsWith("---empty"))
		{
			targetTextLine = "";
		}

		yesNoLine = reader.readLine().trim();

		if (yesNoLine.equals("NO"))
		{
			targetShouldMatchTemp = false;

			String skipBlankLine = reader.readLine();
		}
		else if (yesNoLine.equals("YES"))
		{
			targetShouldMatchTemp = true;

			List registerLineList = new ArrayList();

			for (String registerLine = reader.readLine(); 
						registerLine != null && ! "".equals(registerLine); 
						registerLine = reader.readLine())
			{
				registerLine = registerLine.trim();

				if (registerLine.startsWith("---empty"))
				{
					registerLineList.add("");
				}
				else
				{
					registerLineList.add(registerLine);
				}
			}

			expectedParenRegistersTemp = 
				(String[]) registerLineList.toArray(new String[0]); 
		}
		else
		{
			throw new IOException("Need YES or NO at line " + reader.getLineNumber());
		}
	}

	return new BasicREMatchTest(
		nameLine, 
		patternLine, 
		patternShouldCompileTemp, 
		targetTextLine, 
		targetShouldMatchTemp, 
		expectedParenRegistersTemp); 
}


/**
 * @return  a String representation of myself, suitable for use in
 * debugging
 */

public String toString()
{
	StringBuffer buffer = new StringBuffer();

	buffer.append("BasicREMatchTest:[name=")
		.append(name())
		.append(",rePattern=")
		.append(rePattern)
		.append(",patternShouldCompile=")
		.append(patternShouldCompile)
		.append(",targetText=")
		.append(targetText)
		.append(",targetShouldMatchPattern=")
		.append(targetShouldMatchPattern)
		.append(",expectedParenRegisters=("); 

	for (int i = 0; i < expectedParenRegisters.length; ++i)
	{
		buffer.append(expectedParenRegisters[i]);
		if (i < expectedParenRegisters.length - 1)
		{
			buffer.append(',');
		}
	}

	buffer.append(")]");

	return buffer.toString();
}

public static void main(String[] args)
{
	new junit.swingui.TestRunner().run(BasicREMatchTest.class);
}

/**
 * @return  a suite of tests
 * @exception  IOException  if the test cannot be read
 */

public static Test suite() throws IOException
{
	URL in = BasicREMatchTest.class.getResource("/RETest.txt");
	return suite(in.openStream());
}

/**
 * Create a suite of BasicREMatchTests from an InputStream.
 *
 * @param  in  the InputStream to read from
 * @return  a suite of tests
 * @exception  IOException  if there are problems with the stream
 */

public static Test suite(InputStream in) throws IOException
{
	return suite(new InputStreamReader(in));
}

/**
 * Create a suite of BasicREMatchTests from a Reader.
 *
 * @param  in  Reader to read test dat from
 * @return  a suite of tests
 * @exception  IOException  if there are problems with the reader
 */

public static Test suite(Reader in) throws IOException
{
	LineNumberReader reader = new LineNumberReader(in);
	TestSuite tests = new TestSuite();

	while (reader.ready())
	{
		tests.addTest(getNextTest(reader));
	}

	return tests;
}

/**
 * Create a suite of BasicREMatchTests from a resource at the given URL.
 *
 * @param  url  the URL to read from
 * @return  a suite of tests
 * @exception  IOException  if there are problems reading from the URL
 */

public static Test suite(URL url) throws IOException
{
	return suite(url.openStream());
}
}