/*
 *  arch-tag: Implementation of several functions to manage the covers
 *
 *  Copyright (C) 2004,2005 Marc Pavot <m.pavot@laposte.net>
 *  Copyright (C) 2005 Christophe Fergeau <teuf@gnome.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include <glib.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtkmessagedialog.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnome/gnome-i18n.h>
#include <libxml/xmlmemory.h>
#include <libxml/parser.h>
#include <libxml/tree.h>
#include <string.h>

#include "rb-cover.h"
#include "rb-file-helpers.h"
#include "rb-debug.h"
#include "rb-preferences.h"
#include "../widgets/eel-gconf-extensions.h"

#define READ_CHUNK_SIZE 8192
#define THREE_MONTHS 7776000
#define ROOT_NODE_NAME (const unsigned char *)"covers"
#define XML_VERSION (const unsigned char *)"1.0"

#define AMAZON_URI  "http://xml.amazon.%s/onca/xml3?t=webservices-20&dev-t=%s&KeywordSearch=%s&mode=%s&locale=%s&type=lite&page=1&f=xml"

#define COVER_SMALL "ImageUrlSmall"
#define COVER_MEDIUM "ImageUrlMedium"
#define COVER_LARGE "ImageUrlLarge"

static GnomeVFSResult rb_cover_create_cover_dir (void);
static void read_file_read_chunk (RBCoverReadFileHandle *handle);

static const char *valid_cover_names[] = {
	"folder",
	".folder",
	"cover",
	NULL
};

static const char *valid_cover_extensions[] = {
	"png",
	"jpg",
	"jpeg",
	"gif",
	NULL
};

gchar *
rb_cover_make_cover_path (const gchar *artist,
			  const gchar *album,
			  RBCoverHomeCoversSize cover_size)
{
	char *cover_path, *tmp;
	char *filename;
	const char *to_strip = "#/";

	/* There is 2 different files : a small one for the labums list
	 * and a normal one for the rb-album-cover widget */
	if (cover_size == SMALL_COVER)
		filename = g_strdup_printf ("SMALL%s-%s.jpg", artist, album);
	else
		filename = g_strdup_printf ("%s-%s.jpg", artist, album);

	/* We replace some special characters with spaces. */
	for (tmp = filename; *tmp != '\0'; tmp++) {
		if (strchr (to_strip, *tmp))
			*tmp = ' ';
	}

	/* The returned path is ~/.gnome2/rhythmbox/covers/filename */
	cover_path = g_build_filename (rb_dot_dir (), "covers", filename, NULL);
	g_free (filename);

	return cover_path;
}

gboolean
rb_cover_cover_exists (const gchar *artist,
		       const gchar *album)
{
	gchar *cover_path, *small_cover_path;
	gboolean result;

	/* The path for the normal cover */
	cover_path = rb_cover_make_cover_path (artist,
					       album,
					       NORMAL_COVER);

	/* The path for the small cover */
	small_cover_path = rb_cover_make_cover_path (artist,
						     album,
						     SMALL_COVER);

	/* We consider that the cover exists only if the normal and small covers exist */
	result = (rb_uri_exists (cover_path) && rb_uri_exists (small_cover_path));

	g_free (cover_path);
	g_free (small_cover_path);

	return result;
}

static char*
rb_cover_make_xml_filename (void)
{
	char *xml_filename;

	/* The xml_filename is ~/.gnome2/rhythmbox/covers/covers.xml */
	xml_filename = g_build_filename (rb_dot_dir (), "covers.xml", NULL);

	return xml_filename;
}

static GnomeVFSResult
rb_cover_create_cover_dir (void)
{
	gchar *cover_dir;
	GnomeVFSResult result;

	/* The cover_dir is ~/.gnome2/rhythmbox/covers/ */
	cover_dir = g_build_filename (rb_dot_dir (), "covers", NULL);

	/* If the cover directory doesn't exist, we create it with permission 744 */
	if (!rb_uri_exists (cover_dir))
		 result = gnome_vfs_make_directory (cover_dir, GNOME_VFS_PERM_USER_ALL
							       | GNOME_VFS_PERM_GROUP_READ
							       | GNOME_VFS_PERM_OTHER_READ);
	else
		result = GNOME_VFS_OK;

	g_free (cover_dir);
	return result;
}

gboolean
rb_cover_size_is_valid (int size)
{
	/* return true if the cover isn't too big or too small (blank amazon image) */
	return (size < 1024 * 200 && size > 900);
}

static gboolean
rb_album_cover_is_cover_file_name (const char *filename)
{
	int i, j;
	char *valid_filename;

	for (i = 0; valid_cover_names[i] != NULL; i++) {
		for (j = 0; valid_cover_extensions[j] != NULL; j++) {
			valid_filename = g_strconcat (valid_cover_names[i], ".", valid_cover_extensions[j], NULL);
			if (g_ascii_strcasecmp (filename, valid_filename) == 0) {
				return TRUE;
			}
			g_free (valid_filename);
		}
	}
	return FALSE;
}

gchar *
rb_cover_find_local_uri (const char *location)
{
	char *path;
	char *cover_uri = NULL;
	GnomeVFSURI *uri;
	GList *fileinfos;
	GList *it;
	GnomeVFSResult res;

	if (rb_uri_is_iradio (location))
		return NULL;

	uri = gnome_vfs_uri_new (location);
	if (uri == NULL) {
		return NULL;
	}

	path = gnome_vfs_uri_extract_dirname (uri);
	gnome_vfs_uri_unref (uri);
	if (path == NULL) {
		return NULL;
	}

	res = gnome_vfs_directory_list_load (&fileinfos, path,
					     GNOME_VFS_FILE_INFO_DEFAULT);
	if (res != GNOME_VFS_OK) {
		return NULL;
	}

	cover_uri = NULL;

	/* For each file in the directory, we determine if he has one of
	 * the possible cover names (folder.png, cover.jpg, ...) */
	for (it = fileinfos; it != NULL; it = it->next) {
		GnomeVFSFileInfo *info;
		info = (GnomeVFSFileInfo*)it->data;
		if (rb_album_cover_is_cover_file_name (info->name) && rb_cover_size_is_valid (info->size)) {
			/* We have found a cover, we use it */
			cover_uri = g_strconcat (path, "/", info->name, NULL);
			break;
		}
	}

	g_list_foreach (fileinfos, (GFunc) gnome_vfs_file_info_unref, NULL);
	g_list_free (fileinfos);
	g_free (path);
	return cover_uri;
}

GnomeVFSResult
rb_cover_load_local_cover (const char *location,
			   int *file_size,
			   char **file_contents)
{
	char *cover_uri;

	cover_uri = rb_cover_find_local_uri (location);

	/* If no cover is found, we return an error */
	if (!cover_uri)
		return GNOME_VFS_ERROR_BAD_PARAMETERS;

	/* If a cover is found, we load it into file_contents and we return the result of this operaton */
	return gnome_vfs_read_entire_file (cover_uri,
					   file_size,
					   file_contents);
}

GList *
rb_cover_parse_amazon_xml_file (char *xmldata,
				int size,
				RBCoverAmazonOperation operation,
				RBCoverAmazonCoversSize cover_size)
{
	xmlDocPtr doc;
	xmlNodePtr cur;
	xmlChar *key;
	GList *cover_uris = NULL;
	char *cover_size_text;

	/* Amazon provides 3 different covers sizes. By default, we use the medium one */
	switch (cover_size)
	{
	case AMAZON_SMALL_COVER:
		cover_size_text = COVER_SMALL;
		break;
	case AMAZON_MEDIUM_COVER:
		cover_size_text = COVER_MEDIUM;
		break;
	case AMAZON_LARGE_COVER:
		cover_size_text = COVER_LARGE;
		break;
	default:
		cover_size_text = COVER_MEDIUM;
		break;
	}

	rb_debug ("Parsing XML file from Amazon");
	doc = xmlParseMemory (xmldata, size);

	if (doc == NULL ) {
		return NULL;
	}

	cur = xmlDocGetRootElement(doc);

	if (cur == NULL) {
		xmlFreeDoc (doc);
		return NULL;
	}

	/* We check that the root node name is "ProductInfo" */
	if (xmlStrcmp (cur->name, (const xmlChar *) "ProductInfo")) {
		xmlFreeDoc (doc);
		return NULL;
	}
	cur = cur->xmlChildrenNode;
	while (cur != NULL) {
		if (!xmlStrcmp (cur->name, (const xmlChar *)"Details")){
			xmlNodePtr cur2;
			cur2 = cur->xmlChildrenNode;
			while (cur2 != NULL) {
				if ((!xmlStrcmp (cur2->name, (const xmlChar *) cover_size_text))) {
					/* A possible cover uri has been found, we add it to the list*/
					key = xmlNodeListGetString (doc, cur2->xmlChildrenNode, 1);
					cover_uris = g_list_append (cover_uris, key);
					if (operation == GET_FIRST_COVER) {
						/* If we only want one cover, we now stop to parse the file */
						xmlFreeDoc (doc);
						return cover_uris;
					}
				}
				cur2 = cur2->next;
			}
		}
		cur = cur->next;
	}

	xmlFreeDoc (doc);

	return cover_uris;
}

static char *strnrepl (char *s, int size, const char *old, const char *new)
{
	char *ptr = s;
	int left = strlen (s);
	int avail = size - (left+1);
	int oldlen = strlen (old);
	int newlen = strlen (new);
	int diff = newlen - oldlen;

	while (left >= oldlen) {
		if (strncmp (ptr, old, oldlen) != 0) {
			left--;
			ptr++;
			continue;
		}
		if (diff > avail)
			break;
		if (diff != 0)
			memmove (ptr+newlen, ptr+oldlen, left);
		strncpy (ptr, new, newlen);
		ptr += newlen;
		left -= oldlen;
	}
	return s;
}

char *
rb_cover_make_amazon_xml_uri (const char *artist,
			      const char *album)
{
	char *xml_uri;
	char *keywords;
	char *locale;
	const char *music_mode;
	const char *ext;
	char *tmp;
	int i;

	/* This the key used to make request on the amazon WebServices */
	const char *mykey = "1BDCAEREYT743R9SXE02";
	const gchar *to_replace[] = {"é", "è", "ê", "à", "ù", "ç", "#", "/", "?", "'", "-", "\"", "&", ":", "*", "(", ")", NULL};
	const gchar *replace_to[] = {"e", "e", "e", "a", "u", "c", "" , "" , "" , " ", " ", " ",  " ", " ", " ", " ", " ", NULL};
	const char *unknown = _("Unknown");

	if (!album || !artist)
		return NULL;

	/* If the artist in unknown, we don't search for a cover */
	if (!strcmp (artist, unknown))
		return NULL;

	/* If the album is unknown, we only use the artist to search for the cover */
	if (!strcmp (album, unknown))
		keywords = g_strdup (artist);
	else
		keywords = g_strdup_printf ("%s %s", artist, album);

	/* We replace some special characters to make more accurate requests
	 * FIXME : find a better way to do it */
	for (i = 0; to_replace[i]; i++) {
		if (replace_to[i])
			keywords = strnrepl (keywords, strlen (keywords) + 1, to_replace[i], replace_to[i]);
	}

	/* We escape the other specials characters */
	tmp = gnome_vfs_escape_string (keywords);
	g_free (keywords);
	keywords = tmp;

	/* What is the amazon country choosen in the preferences? */
	locale = eel_gconf_get_string (CONF_COVER_AMAZON_COUNTRY);

	/* By default, we use amazon US (amazon.com) */
	if (!locale)
		locale = "com";

	/* The default mode is "music" and the default extension is ".com" */
	music_mode = "music";
	ext = "com";

	/* The french amazon need a different mode */
	if (!strcmp (locale, "fr"))
		music_mode = "music-fr";

	/* The japanese amazon need a different mode and a different extension */
	if (!strcmp (locale, "jp")) {
		music_mode = "music-jp";
		ext = "co.jp";
	}

	/* For amazon.com, we need to use "us" for the "locale" argument */
	if (!strcmp (locale, "com")) {
		g_free (locale);
		locale = "us";
	}

	/* We make the xml uri with all the parameters */
	xml_uri = g_strdup_printf (AMAZON_URI, ext, mykey, keywords, music_mode, locale);

	g_free (keywords);
	g_free (locale);

	return xml_uri;
}

GnomeVFSResult
rb_cover_load_amazon_covers (const char *artist,
			     const char *album,
			     GList **cover_uris,
			     GArray **file_size,
			     GList **file_contents,
			     RBCoverAmazonOperation operation,
			     RBCoverAmazonCoversSize cover_size)
{
	char *xml_uri;
	int xml_size;
	char *xml_data;
	GList *temp;
	GnomeVFSResult result, result_temp;
	int temp_size;
	char *temp_contents;

	/* We construct the uri to make a request on the amazon WebServices */
	xml_uri = rb_cover_make_amazon_xml_uri (artist,
						album);

	if (!xml_uri)
		return GNOME_VFS_ERROR_BAD_PARAMETERS;

	/* We laod the xml file in xml_data */
	result = gnome_vfs_read_entire_file (xml_uri,
					     &xml_size,
					     &xml_data);
	g_free (xml_uri);

	if (result != GNOME_VFS_OK) {
		g_free (xml_data);
		return result;
	}

	/* We parse the xml file to extract the cover uris */
	*cover_uris = rb_cover_parse_amazon_xml_file (xml_data,
						      xml_size,
						      operation,
						      cover_size);

	g_free (xml_data);

	/* By default, we return an error */
	result = GNOME_VFS_ERROR_NOT_FOUND;

	for (temp = *cover_uris; temp; temp = temp->next) {
		if (temp->data) {
			/* For each cover uri, we load the image data in temp_contents */
			result_temp = gnome_vfs_read_entire_file (temp->data,
								  &temp_size,
								  &temp_contents);
			if (result_temp == GNOME_VFS_OK && rb_cover_size_is_valid (temp_size)) {
				/* If the cover is not too big and not too small (blank amazon image), we append it to file_contents */
				g_array_append_val (*file_size, temp_size);
				*file_contents = g_list_append (*file_contents, temp_contents);
				/* If at least one cover is found, we return GNOME_VFS_OK */
				result = result_temp;
			}
		}
	}

	return result;
}

static void
rb_cover_create_xml_file (char *xml_filename)
{
	xmlDocPtr doc;
	xmlNodePtr root_node;

	/* if the file exists, we do nothing */
	if (rb_uri_exists (xml_filename))
		return;

	doc = xmlNewDoc (XML_VERSION);

	/* We add the root node */
	root_node = xmlNewDocNode (doc, NULL, ROOT_NODE_NAME, NULL);
	xmlDocSetRootElement (doc, root_node);

	/* We save the xml file */
	xmlSaveFormatFile (xml_filename, doc, TRUE);

	xmlFreeDoc (doc);
}

static void
rb_cover_remove_cover_in_xml (const gchar *artist,
			      const gchar *album)
{
	xmlDocPtr doc;
	xmlNodePtr cur;
	char *current_artist, *current_album;
	char *xml_filename;

	if (!artist || !album)
		return;

	xml_filename = rb_cover_make_xml_filename ();

	/* This option is necessary to save a well formated xml file */
	xmlKeepBlanksDefault (0);
	/* If the covers.xml file doesn't exist, we create it */
	rb_cover_create_xml_file (xml_filename);

	doc = xmlParseFile (xml_filename);

	if (doc == NULL )
		return;

	cur = xmlDocGetRootElement(doc);

	if (cur == NULL) {
		xmlFreeDoc (doc);
		return;
	}

	/* We check that the root node has the right name */
	if (xmlStrcmp(cur->name, (const xmlChar *) ROOT_NODE_NAME)) {
		xmlFreeDoc (doc);
		return;
	}

	cur = cur->children;
	while (cur != NULL) {
		/* For each "cover" entry */
		if (!xmlStrcmp (cur->name, (const xmlChar *)"cover")){
			current_artist = (char *)xmlGetProp (cur, (const unsigned char *)"artist");
			current_album = (char *)xmlGetProp (cur, (const unsigned char *)"album");
			if (current_artist && current_album) {
				if (!strcmp (current_artist, artist) && !strcmp (current_album, album)) {
					/* It is the right artist/album, we remove the entry */
					xmlUnlinkNode (cur);
					xmlFreeNode (cur);
					break;
				}
				xmlFree (current_artist);
				xmlFree (current_album);
			}
		}
		cur = cur->next;
	}

	/* We save the xml file */
	xmlSaveFormatFile (xml_filename, doc, TRUE);
	xmlFreeDoc (doc);
	g_free (xml_filename);
}

void
rb_cover_remove_cover (const gchar *artist,
		       const gchar *album)
{
	gchar *small_cover_path;
	gchar *cover_path;

	if (!rb_cover_cover_exists (artist, album))
		return;

	/* Delete the small cover*/
	small_cover_path = rb_cover_make_cover_path (artist, album, SMALL_COVER);
	if (rb_uri_exists (small_cover_path))
		gnome_vfs_unlink (small_cover_path);
	g_free (small_cover_path);

	/* Delete the normal cover*/
	cover_path = rb_cover_make_cover_path (artist, album, NORMAL_COVER);
	if (rb_uri_exists (cover_path))
		gnome_vfs_unlink (cover_path);
	g_free (cover_path);

	/* Remove the entry in covers.xml*/
	rb_cover_remove_cover_in_xml (artist, album);
}

static void
rb_cover_save_cover_in_xml (const char *artist,
			    const char *album,
			    RBCoverOrigin origin,
			    const char *url)
{
	xmlDocPtr doc;
	xmlNodePtr cur, cur2;
	char *xml_filename;
	xmlChar *converted_artist, *converted_album, *converted_url;

	if (!artist || !album)
		return;

	if (!strcmp (artist, "") || !strcmp (album, ""))
		return;

	/* We escape characters used in an XML file (<, >, &, ", ...) */
	converted_artist = xmlEncodeEntitiesReentrant (NULL, (const unsigned char *) artist);
	converted_album = xmlEncodeEntitiesReentrant (NULL, (const unsigned char *) album);

	if (!converted_artist || !converted_album)
		return;

	if ((origin == AMAZON_COVER) && !url)
		return;

	/* We escape characters used in an XML file (<, >, &, ", ...) */
	converted_url = xmlEncodeEntitiesReentrant (NULL, (const unsigned char *) url);

	if (!converted_url) {
		xmlFree (converted_artist);
		xmlFree (converted_album);
		return;
	}

	/* If the entry already exists in the xml file for the album, we remove it*/
	rb_cover_remove_cover_in_xml (artist, album);

	/* Then we add a new entry*/
	xml_filename = rb_cover_make_xml_filename ();

	/* This option is necessary to save a well formated xml file */
	xmlKeepBlanksDefault (0);
	/* If the covers.xml file doesn't exist, we create it */
	rb_cover_create_xml_file (xml_filename);

	doc = xmlParseFile (xml_filename);

	if (doc == NULL )
		return;

	cur = xmlDocGetRootElement (doc);

	if (cur == NULL) {
		xmlFreeDoc (doc);
		return;
	}

	/* We check that the root node has the right name */
	if (xmlStrcmp (cur->name, (const xmlChar *) ROOT_NODE_NAME)) {
		xmlFreeDoc (doc);
		return;
	}

	cur = xmlDocGetRootElement (doc);

	/* We add a new "cover" entry */
	cur2 = xmlNewChild (cur, NULL, (const unsigned char *)"cover", NULL);

	/* We add the "artist" attribute to the new "cover" entry */
	xmlSetProp (cur2, (const unsigned char *)"artist", converted_artist);
	xmlFree (converted_artist);

	/* We add the "album" attribute to the new "cover" entry */
	xmlSetProp (cur2, (const unsigned char *)"album", converted_album);
	xmlFree (converted_album);

	/*If the cover comes from amazon, we add nodes with the date and the url of the cover to re-download it after 3 months*/
	if (origin == AMAZON_COVER) {
		GTimeVal time;
		gchar *date;
		/* The cover is from amazon */
		xmlNewTextChild (cur2, NULL, (const unsigned char *)"origin",
				 (const unsigned char *)"amazon");

		/* The cover can be re-downloaded at converted_url */
		xmlNewTextChild (cur2, NULL, (const unsigned char *)"url", converted_url);
		xmlFree (converted_url);

		/* And the cover has been downloaded now */
		g_get_current_time (&time);
		date = g_strdup_printf ("%d", (int) time.tv_sec);
		xmlNewTextChild (cur2, NULL, (const unsigned char *)"date",
				 (const unsigned char *)date);
		g_free (date);
	} else {
		/* else, we simply add the an origin node */
		xmlNewTextChild (cur2, NULL, (const unsigned char *)"origin", (const unsigned char *)"local");
	}

	/* We save the xml file */
	xmlSaveFormatFile (xml_filename, doc, TRUE);
	xmlFreeDoc (doc);
	g_free (xml_filename);
}

static gboolean
rb_cover_can_overwrite_cover (const gchar *artist,
			      const gchar *album,
			      RBCoverOverwriteMode overwrite_mode)
{
	GtkWidget *dialog;
	gint retval;

	/* If the cover already exists, we can ask, replace or skip depending of the overwrite_mode argument */
	if (rb_cover_cover_exists (artist, album)) {
		switch (overwrite_mode) {
		case OVERWRITE_MODE_ASK:
			dialog = gtk_message_dialog_new (NULL,
							 GTK_DIALOG_MODAL,
							 GTK_MESSAGE_QUESTION,
							 GTK_BUTTONS_YES_NO,
							 "The cover already exists. Do you want to replace it?");

			retval = gtk_dialog_run (GTK_DIALOG (dialog));
			gtk_widget_destroy (dialog);
			/* If the user doesn't say "yes", we don't overwrite the cover */
			if (retval != GTK_RESPONSE_YES)
				return FALSE;
			break;
		case OVERWRITE_MODE_REPLACE:
			/* We overwrite the cover */
			return TRUE;
			break;
		case OVERWRITE_MODE_SKIP:
			/* We don't overwrite the cover */
			return FALSE;
		default:
			/* By default, we don't overwrite the cover */
			return FALSE;
		}
	}

	/*If the cover doesn't exists, we return TRUE */
	return TRUE;
}

GnomeVFSResult
rb_cover_save_cover (const gchar *artist,
		     const gchar *album,
		     char *data,
		     int size,
		     RBCoverOrigin origin,
		     char *url,
		     RBCoverOverwriteMode overwrite_mode)
{
	GnomeVFSResult result;
	gchar *cover_path, *small_cover_path;
	GdkPixbufLoader *loader;
	GdkPixbuf *pixbuf, *small_pixbuf;
	int width, height;

	if (!artist || !album || !data)
		return GNOME_VFS_ERROR_BAD_PARAMETERS;

	/* If the cover directory doesn't exist, we create it */
	result = rb_cover_create_cover_dir ();
	if (result != GNOME_VFS_OK)
		return result;

	/* If the cover already exists, can we overwrite it? */
	if (!rb_cover_can_overwrite_cover (artist, album, overwrite_mode))
		return GNOME_VFS_OK;

	/* The path for the normal cover */
	cover_path = rb_cover_make_cover_path (artist,
					       album,
					       NORMAL_COVER);

	/* The path for the small cover */
	small_cover_path = rb_cover_make_cover_path (artist,
						     album,
						     SMALL_COVER);

	loader = gdk_pixbuf_loader_new ();

	/*By default, we return an error */
	result = GNOME_VFS_ERROR_BAD_PARAMETERS;

	if (gdk_pixbuf_loader_write (loader,
				     (const guchar *)data,
				     size,
				     NULL)) {
		gdk_pixbuf_loader_close (loader, NULL);

		pixbuf = gdk_pixbuf_loader_get_pixbuf (loader);
		width = gdk_pixbuf_get_width (pixbuf);
		height = gdk_pixbuf_get_height (pixbuf);

		/* We resize the pixbuf to save the small cover.
		 * To do it, we keep the original aspect ratio and
		 * we limit max (height, width) by TREE_COVER_SIZE */
		if (width > height) {
			small_pixbuf = gdk_pixbuf_scale_simple (pixbuf,
								TREE_COVER_SIZE,
								height * TREE_COVER_SIZE / width,
								GDK_INTERP_BILINEAR);
		} else {
			small_pixbuf = gdk_pixbuf_scale_simple (pixbuf,
								width * TREE_COVER_SIZE / height,
								TREE_COVER_SIZE,
								GDK_INTERP_BILINEAR);
		}

		/* We save the normal and the small covers */
		if (gdk_pixbuf_save (pixbuf, cover_path, "jpeg", NULL, NULL) &&
		    gdk_pixbuf_save (small_pixbuf, small_cover_path, "jpeg", NULL, NULL)) {
			/* If we succeed in the 2 operations, we return GNOME_VFS_OK */
			result = GNOME_VFS_OK;
		}
		g_object_unref (G_OBJECT (pixbuf));
		g_object_unref (G_OBJECT (small_pixbuf));
	}

	/* We save an new entry in the covers.xml file */
	rb_cover_save_cover_in_xml (artist, album, origin, url);

	g_free (cover_path);
	g_free (small_cover_path);

	return result;
}

void
rb_cover_update_amazon_covers (void)
{
	xmlDocPtr doc;
	xmlNodePtr cur;
	char *artist, *album;
	char *origin = NULL, *url = NULL, *date = NULL;
	char *xml_filename;
	int date_int;
	GTimeVal time;
	int current_date_int;
	char *file_contents;
	int file_size;
	GnomeVFSResult result;

	/* This function is called every time rhythmbox is launched.
	 * It redownload the covers older than 3 months to respect the
	 * Amazon Web Services Licensing Agreement */
	rb_debug ("Start the update of the old amazon covers");

	xml_filename = rb_cover_make_xml_filename ();

	/* This option is necessary to save a well formated xml file */
	xmlKeepBlanksDefault (0);
	/* If the covers.xml file doesn't exist, we create it */
	rb_cover_create_xml_file (xml_filename);

	doc = xmlParseFile (xml_filename);
	g_free (xml_filename);

	if (doc == NULL )
		return;

	cur = xmlDocGetRootElement(doc);

	if (cur == NULL) {
		xmlFreeDoc (doc);
		return;
	}

	/* We check that the root node has the right name */
	if (xmlStrcmp(cur->name, (const xmlChar *) ROOT_NODE_NAME)) {
		xmlFreeDoc (doc);
		return;
	}

	/* Get the current time */
	g_get_current_time (&time);
	current_date_int = (int) time.tv_sec;

	cur = cur->children;
	while (cur != NULL) {
		/* For each "cover" entry */
		if (!xmlStrcmp (cur->name, (const xmlChar *)"cover")){
			/* FIXME : I think that there is a problem with encoded special characters
			 * we should decode them but I don't know how to do it 
			 * Nevertheless, it works quite fine but some covers may not be redownloaded */
			artist = (char *) xmlGetProp (cur, (const unsigned char *)"artist");
			album = (char *) xmlGetProp (cur, (const unsigned char *)"album");
			if (artist && album) {
				xmlNodePtr cur2;
				cur2 = cur->children;
				/* We get the content of the origin, url and date nodes */
				while (cur2 != NULL) {
					if ((!xmlStrcmp (cur2->name, (const xmlChar *) "origin"))) {
						origin = (char *)xmlNodeListGetString (doc, cur2->children, 1);
					}
					if ((!xmlStrcmp (cur2->name, (const xmlChar *) "url"))) {
						url = (char *)xmlNodeListGetString (doc, cur2->children, 1);
					}
					if ((!xmlStrcmp (cur2->name, (const xmlChar *) "date"))) {
						date = (char *)xmlNodeListGetString (doc, cur2->children, 1);
					}
					cur2 = cur2->next;
				}
				if (origin && url && date) {
					if (!strcmp (origin, "amazon")) {
						/* This cover comes from amazon */
						date_int = atoi (date);
						/* We check if this cover is older than 3 months */
						if ((current_date_int-date_int) > THREE_MONTHS) {
							rb_debug ("Updating the cover : %s-%s", artist, album);
							/* It is older than 3 months, remove it... */
							rb_cover_remove_cover (artist, album);
							/* ...we redownload it... */
							result = gnome_vfs_read_entire_file (url,
											     &file_size,
											     &file_contents);
							if (result == GNOME_VFS_OK)
								/* ...and we save it. */
								rb_cover_save_cover (artist,
										     album,
										     file_contents,
										     file_size,
										     AMAZON_COVER,
										     url,
										     OVERWRITE_MODE_REPLACE);
							g_free (file_contents);
						}
					}
					g_free (origin);
					origin = NULL;
					g_free (url);
					url = NULL;
					g_free (date);
					date = NULL;
				}
				g_free (artist);
				g_free (album);
			}
		}
		cur = cur->next;
	}

	xmlFreeDoc (doc);
	rb_debug ("End of the update of the old amazon covers");
}

gboolean
rb_cover_get_do_not_search (const gchar *artist,
			    const gchar *album)
{
	xmlDocPtr doc;
	xmlNodePtr cur;
	char *xml_filename;
	xmlChar *current_artist, *current_album;
	xmlChar *converted_artist, *converted_album;
	char *origin;
	gboolean result, found;
	const char *various_artists = _("Various Artists");

	if (!artist || !album)
		return FALSE;

	/* We don't want covers for "Various Artists" albums */
	if (!strcmp (artist, various_artists))
		return TRUE;

	/* We escape characters used in an XML file (<, >, &, ", ...) */
	converted_artist = xmlEncodeEntitiesReentrant (NULL, (const unsigned char *)artist);
	converted_album = xmlEncodeEntitiesReentrant (NULL, (const unsigned char *)album);

	if (!converted_artist || !converted_album)
		return FALSE;

	xml_filename = rb_cover_make_xml_filename ();

	/* This option is necessary to save a well formated xml file */
	xmlKeepBlanksDefault (0);
	/* If the covers.xml file doesn't exist, we create it */
	rb_cover_create_xml_file (xml_filename);

	doc = xmlParseFile (xml_filename);
	g_free (xml_filename);

	if (doc == NULL )
		return FALSE;

	cur = xmlDocGetRootElement (doc);

	if (cur == NULL) {
		xmlFreeDoc (doc);
		return FALSE;
	}

	/* We check that the root node has the right name */
	if (xmlStrcmp(cur->name, (const xmlChar *) ROOT_NODE_NAME)) {
		xmlFreeDoc (doc);
		return FALSE;
	}

	/* At the beginning, we haven't found the right entry and 
	 * we consider that rb have to search for the cover*/
	found = FALSE;
	result = FALSE;

	cur = cur->children;
	while (cur != NULL && !found) {
		/* For each "cover" entry */
		if (!xmlStrcmp (cur->name, (const xmlChar *)"cover")){
			current_artist = xmlGetProp (cur, (const unsigned char *)"artist");
			current_album = xmlGetProp (cur, (const unsigned char *)"album");
			if (current_artist && current_album) {
				if (!xmlStrcmp (current_artist, converted_artist) && !xmlStrcmp (current_album, converted_album)) {
					/* It is the right artist/album */
					xmlNodePtr cur2;
					cur2 = cur->children;
					while (cur2 != NULL && !found) {
						if ((!xmlStrcmp (cur2->name, (const xmlChar *) "origin"))) {
							origin = (char *) xmlNodeListGetString (doc, cur2->children, 1);
							/* We have found the right node */
							found = TRUE;
							if (origin) {
								/* If the origin is "none", the user has that rb doesn't have to search
								 * for this cover and we return TRUE else, we return FALSE */
								result = !strcmp (origin, "none");
								g_free (origin);
							}
						}
						cur2 = cur2->next;
					}
				}
				xmlFree (current_artist);
				xmlFree (current_album);
			}
		}
		cur = cur->next;
	}

	xmlFree (converted_artist);
	xmlFree (converted_album);
	xmlFreeDoc (doc);

	return result;
}

void
rb_cover_set_do_not_search (const gchar *artist,
			    const gchar *album,
			    gboolean do_not_search)
{
	xmlDocPtr doc;
	xmlNodePtr cur;
	char *xml_filename;
	xmlChar *current_artist, *current_album;
	xmlChar *converted_artist, *converted_album;
	char *origin;
	gboolean previous_none;
	gboolean found = FALSE;


	if (!artist || !album)
		return;

	/* We escape characters used in an XML file (<, >, &, ", ...) */
	converted_artist = xmlEncodeEntitiesReentrant (NULL, (const unsigned char *)artist);
	converted_album = xmlEncodeEntitiesReentrant (NULL, (const unsigned char *)album);

	if (!converted_artist || !converted_album)
		return;

	/* Then we add a new entry*/
	xml_filename = rb_cover_make_xml_filename ();

	/* This option is necessary to save a well formated xml file */
	xmlKeepBlanksDefault (0);
	/* If the covers.xml file doesn't exist, we create it */
	rb_cover_create_xml_file (xml_filename);

	doc = xmlParseFile (xml_filename);

	if (doc == NULL )
		return;

	cur = xmlDocGetRootElement (doc);

	if (cur == NULL) {
		xmlFreeDoc (doc);
		return;
	}

	/* We check that the root node has the right name */
	if (xmlStrcmp(cur->name, (const xmlChar *) ROOT_NODE_NAME)) {
		xmlFreeDoc (doc);
		return;
	}

	cur = cur->children;
	while (cur != NULL && !found) {
		/* For each "cover" entry */
		if (!xmlStrcmp (cur->name, (const xmlChar *)"cover")){
			current_artist = xmlGetProp (cur, (const unsigned char *)"artist");
			current_album = xmlGetProp (cur, (const unsigned char *)"album");
			if (current_artist && current_album) {
				if (!xmlStrcmp (current_artist, converted_artist) && !xmlStrcmp (current_album, converted_album)) {
					xmlNodePtr cur2;
					/* We have found the right artist/album */
					found = TRUE;
					cur2 = cur->children;
					while (cur2 != NULL) {
						if ((!xmlStrcmp (cur2->name, (const xmlChar *) "origin"))) {
							origin = (char *)xmlNodeListGetString (doc, cur2->children, 1);
							if (origin) {
								previous_none = !strcmp (origin, "none");
								/* If the previous origin was "none" and do_not_search==FALSE,
								 * we remove the entry corresponding to this album */
								if (previous_none) {
									if (!do_not_search) {
										xmlUnlinkNode(cur);
										xmlFreeNode (cur);
										break;
									}
								} else {
									/* If do_not_search == TRUE, we set the origin node to "none" */
									if (do_not_search) {
										xmlNodeSetContent (cur2, (const xmlChar *) "none");
									}
								}
								g_free (origin);
							}
						}
						/* If do_not_search == TRUE, we remove the url node */
						if ((!xmlStrcmp (cur2->name, (const xmlChar *) "url"))) {
							if (do_not_search) {
								xmlUnlinkNode(cur2);
								xmlFreeNode (cur2);
								break;
							}
						}
						/* If do_not_search == TRUE, we remove the date node */
						if ((!xmlStrcmp (cur2->name, (const xmlChar *) "date"))) {
							if (do_not_search) {
								xmlUnlinkNode(cur2);
								xmlFreeNode (cur2);
								break;
							}
						}
						cur2 = cur2->next;
					}
				}
				xmlFree (current_artist);
				xmlFree (current_album);
			}
		}
		cur = cur->next;
	}

	/* If there was no entry in covers.xml for this album, we add a new one with origin=none */
	if (!found && do_not_search) {
		cur = xmlDocGetRootElement(doc);
		xmlNewChild (cur, NULL, (const unsigned char *)"cover", NULL);
		cur = cur->last;
		xmlSetProp (cur, (const unsigned char *)"artist", converted_artist);
		xmlSetProp (cur, (const unsigned char *)"album", converted_album);
		xmlNewChild (cur, NULL, (const unsigned char *)"origin",
			     (const unsigned char *)"none");
	}

	xmlFree (converted_artist);
	xmlFree (converted_album);

	/* We save the xml file */
	xmlSaveFormatFile (xml_filename, doc, TRUE);
	xmlFreeDoc (doc);
	g_free (xml_filename);

	return;
}

/* All the following code is the eel_read_entire_file_async function
 * from the libeel. Maybe would it be better to use libeel and to add it in
 * rhythmbox dependencies. Nevertheless I think that the best solution would
 * be to include this function in gnome_vfs */
struct RBCoverReadFileHandle {
	GnomeVFSAsyncHandle *handle;
	RBCoverReadFileCallback callback;
	gpointer callback_data;
	gboolean is_open;
	char *buffer;
	GnomeVFSFileSize bytes_read;
};

/* When close is complete, there's no more work to do. */
static void
read_file_close_callback (GnomeVFSAsyncHandle *handle,
			  GnomeVFSResult result,
			  gpointer callback_data)
{
}

/* Do a close if it's needed.
 * Be sure to get this right, or we have extra threads hanging around.
 */
static void
read_file_close (RBCoverReadFileHandle *read_handle)
{
	if (read_handle->is_open) {
		gnome_vfs_async_close (read_handle->handle,
				       read_file_close_callback,
				       NULL);
		read_handle->is_open = FALSE;
	}
}

/* Close the file and then tell the caller we succeeded, handing off
 * the buffer to the caller.
 */
static void
read_file_succeeded (RBCoverReadFileHandle *read_handle)
{
	read_file_close (read_handle);

	/* Reallocate the buffer to the exact size since it might be
	 * around for a while.
	 */
	(* read_handle->callback) (GNOME_VFS_OK,
				   read_handle->bytes_read,
				   g_realloc (read_handle->buffer,
					      read_handle->bytes_read),
				   read_handle->callback_data);

	g_free (read_handle);
}

/* Tell the caller we failed. */
static void
read_file_failed (RBCoverReadFileHandle *read_handle, GnomeVFSResult result)
{
	read_file_close (read_handle);
	g_free (read_handle->buffer);

	(* read_handle->callback) (result, 0, NULL, read_handle->callback_data);
	g_free (read_handle);
}

/* A read is complete, so we might or might not be done. */
static void
read_file_read_callback (GnomeVFSAsyncHandle *handle,
			 GnomeVFSResult result,
			 gpointer buffer,
			 GnomeVFSFileSize bytes_requested,
			 GnomeVFSFileSize bytes_read,
			 gpointer callback_data)
{
	RBCoverReadFileHandle *read_handle;
	gboolean read_more;

	/* Do a few reality checks. */
	g_assert (bytes_requested == READ_CHUNK_SIZE);
	read_handle = callback_data;
	g_assert (read_handle->handle == handle);
	g_assert (read_handle->buffer + read_handle->bytes_read == buffer);
	g_assert (bytes_read <= bytes_requested);

	/* Check for a failure. */
	if (result != GNOME_VFS_OK && result != GNOME_VFS_ERROR_EOF) {
		read_file_failed (read_handle, result);
		return;
	}

	/* Check for the extremely unlikely case where the file size overflows. */
	if (read_handle->bytes_read + bytes_read < read_handle->bytes_read) {
		read_file_failed (read_handle, GNOME_VFS_ERROR_TOO_BIG);
		return;
	}

	/* Bump the size. */
	read_handle->bytes_read += bytes_read;

	/* Read more unless we are at the end of the file. */
	if (bytes_read == 0 || result != GNOME_VFS_OK)
		read_more = FALSE;
	else
		read_more = TRUE;

	if (read_more) {
		read_file_read_chunk (read_handle);
		return;
	}

	/* If at the end of the file, we win! */
	read_file_succeeded (read_handle);
}

/* Start reading a chunk. */
static void
read_file_read_chunk (RBCoverReadFileHandle *handle)
{
	handle->buffer = g_realloc (handle->buffer, handle->bytes_read + READ_CHUNK_SIZE);
	gnome_vfs_async_read (handle->handle,
			      handle->buffer + handle->bytes_read,
			      READ_CHUNK_SIZE,
			      read_file_read_callback,
			      handle);
}

/* Once the open is finished, read a first chunk. */
static void
read_file_open_callback (GnomeVFSAsyncHandle *handle,
			 GnomeVFSResult result,
			 gpointer callback_data)
{
	RBCoverReadFileHandle *read_handle;

	read_handle = callback_data;
	g_assert (read_handle->handle == handle);

	/* Handle the failure case. */
	if (result != GNOME_VFS_OK) {
		read_file_failed (read_handle, result);
		return;
	}

	/* Handle success by reading the first chunk. */
	read_handle->is_open = TRUE;

	read_file_read_chunk (read_handle);
}
/* Set up the read handle and start reading. */
static RBCoverReadFileHandle *
eel_read_file_async (const char *uri,
		     int priority,
		     RBCoverReadFileCallback callback,
		     gpointer callback_data)
{
	RBCoverReadFileHandle *handle;

	handle = g_new0 (RBCoverReadFileHandle, 1);

	handle->callback = callback;
	handle->callback_data = callback_data;

	gnome_vfs_async_open (&handle->handle,
			      uri,
			      GNOME_VFS_OPEN_READ,
			      priority,
			      read_file_open_callback,
			      handle);
	return handle;
}

/* Set up the read handle and start reading. */
RBCoverReadFileHandle *
rb_cover_read_entire_file_async (const char *uri,
				 int priority,
				 RBCoverReadFileCallback callback,
				 gpointer callback_data)
{
	return eel_read_file_async (uri, priority, callback, callback_data);
}
