/*
 *  arch-tag: Implementation of the cover downloader
 *
 *  Copyright (C) 2004,2005 Marc Pavot <m.pavot@laposte.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include <gtk/gtk.h>
#include <libgnome/gnome-i18n.h>
#include <string.h>

#include "rb-cover.h"
#include "rhythmdb.h"
#include "rb-shell-coverdownloader.h"
#include "rb-shell-coverselect.h"
#include "rb-file-helpers.h"
#include "rb-glade-helpers.h"
#include "rhythmdb-query-model.h"

static void rb_shell_coverdownloader_class_init (RBShellCoverdownloaderClass *klass);
static void rb_shell_coverdownloader_init (RBShellCoverdownloader *shell_coverdownloader);
static void rb_shell_coverdownloader_finalize (GObject *object);
static void rb_shell_coverdownloader_set_property (GObject *object,
			   			   guint prop_id,
			   	    		   const GValue *value,
			   	    		   GParamSpec *pspec);
static void rb_shell_coverdownloader_get_property (GObject *object,
				    		   guint prop_id,
				    		   GValue *value,
				    		   GParamSpec *pspec);
static GObject * rb_shell_coverdownloader_constructor (GType type, guint n_construct_properties,
						       GObjectConstructParam *construct_properties);
static gboolean rb_shell_coverdownloader_window_delete_cb (GtkWidget *window,
							   GdkEventAny *event,
							   RBShellCoverdownloader *shell_coverdownloader);
static void rb_shell_coverdownloader_add_covers_to_list_from_model (RBShellCoverdownloader *shell_coverdownloader,
								    GtkTreeModel *main_model);
static void rb_shell_coverdownloader_get_covers_from_entry (RBShellCoverdownloader *shell_coverdownloader,
							    RhythmDBEntry *entry,
							    RBShellCoverdownloaderOperation operation);
static void rb_shell_coverdownloader_find_local_image (RBShellCoverdownloader *shell_coverdownloader,
						       const char *artist,
			  			       const char *album,
						       const char *location);
static void rb_shell_coverdownloader_find_amazon_image (RBShellCoverdownloader *shell_coverdownloader,
							const char *artist,
							const char *album);
static void rb_shell_coverdownloader_get_covers_from_list (RBShellCoverdownloader *shell_coverdownloader,
							   RBShellCoverdownloaderOperation operation);
static void rb_shell_coverdownloader_close_cb (GtkButton *button,
					       RBShellCoverdownloader *shell_coverdownloader);
static void rb_shell_coverdownloader_cancel_cb (GtkButton *button,
						RBShellCoverdownloader *shell_coverdownloader);

enum
{
	PROP_0,
	PROP_DB
};

struct RBShellCoverdownloaderPrivate
{
	int nb_covers;
	int nb_covers_already_exist;
	int nb_covers_found;
	int nb_covers_not_found;

	gboolean cancelled;
	gboolean closed;

	GSList *entries;

	GtkWidget *progress_artist_label;
	GtkWidget *progress_album_label;
	GtkWidget *progress_operation_label;
	GtkWidget *progress_hbox;
	GtkWidget *progress_const_artist_label;
	GtkWidget *progressbar;
	GtkWidget *cancel_button;
	GtkWidget *close_button;

	RhythmDB *db;

	RBLibrarySource *source;
};

static GObjectClass *parent_class = NULL;

GType
rb_shell_coverdownloader_get_type (void)
{
	static GType rb_shell_coverdownloader_type = 0;
	if (rb_shell_coverdownloader_type == 0)
	{
		static const GTypeInfo our_info =
		{
			sizeof (RBShellCoverdownloaderClass),
			NULL,
			NULL,
			(GClassInitFunc) rb_shell_coverdownloader_class_init,
			NULL,
			NULL,
			sizeof (RBShellCoverdownloader),
			0,
			(GInstanceInitFunc) rb_shell_coverdownloader_init
		};

		rb_shell_coverdownloader_type = g_type_register_static (GTK_TYPE_WINDOW,
									"RBShellCoverdownloader",
									&our_info, 0);
	}
	return rb_shell_coverdownloader_type;
}

static void
rb_shell_coverdownloader_class_init (RBShellCoverdownloaderClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = rb_shell_coverdownloader_finalize;
	object_class->constructor = rb_shell_coverdownloader_constructor;

	object_class->set_property = rb_shell_coverdownloader_set_property;
	object_class->get_property = rb_shell_coverdownloader_get_property;

	g_object_class_install_property (object_class,
					 PROP_DB,
					 g_param_spec_object ("db",
							      "RhythmDB",
							      "RhythmDB database",
							      RHYTHMDB_TYPE,
							      G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));
}

static void
rb_shell_coverdownloader_init (RBShellCoverdownloader *shell_coverdownloader)
{
	shell_coverdownloader->priv = g_new0 (RBShellCoverdownloaderPrivate, 1);
	shell_coverdownloader->priv->cancelled = FALSE;
	shell_coverdownloader->priv->closed = FALSE;
}

static void
rb_shell_coverdownloader_finalize (GObject *object)
{
	RBShellCoverdownloader *shell_coverdownloader;

	g_return_if_fail (object != NULL);
	g_return_if_fail (RB_IS_SHELL_COVERDOWNLOADER (object));

	shell_coverdownloader = RB_SHELL_COVERDOWNLOADER (object);

	g_return_if_fail (shell_coverdownloader->priv != NULL);
	g_free (shell_coverdownloader->priv);
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
rb_shell_coverdownloader_set_property (GObject *object,
			   	    guint prop_id,
			   	    const GValue *value,
			   	    GParamSpec *pspec)
{
	RBShellCoverdownloader *shell_coverdownloader = RB_SHELL_COVERDOWNLOADER (object);

	switch (prop_id)
	{
	case PROP_DB:
		shell_coverdownloader->priv->db = g_value_get_object (value);
		break;	
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}

static void 
rb_shell_coverdownloader_get_property (GObject *object,
				    guint prop_id,
				    GValue *value,
				    GParamSpec *pspec)
{
	RBShellCoverdownloader *shell_coverdownloader = RB_SHELL_COVERDOWNLOADER (object);

	switch (prop_id)
	{
	case PROP_DB:
		g_value_set_object (value, shell_coverdownloader->priv->db);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}

GtkWidget *
rb_shell_coverdownloader_new (RhythmDB *db)
{
	RBShellCoverdownloader *shell_coverdownloader;

	shell_coverdownloader = g_object_new (RB_TYPE_SHELL_COVERDOWNLOADER,
					      "db", db,
					      NULL);

	g_return_val_if_fail (shell_coverdownloader->priv != NULL, NULL);

	return GTK_WIDGET (shell_coverdownloader);
}

static GObject *
rb_shell_coverdownloader_constructor (GType type, guint n_construct_properties,
			   	   GObjectConstructParam *construct_properties)
{
	RBShellCoverdownloader *shell_coverdownloader;
	RBShellCoverdownloaderClass *klass;
	GObjectClass *parent_class;
	GladeXML *xml = NULL;
	GtkWidget *vbox;

	klass = RB_SHELL_COVERDOWNLOADER_CLASS (g_type_class_peek (RB_TYPE_SHELL_COVERDOWNLOADER));

	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	shell_coverdownloader = RB_SHELL_COVERDOWNLOADER (parent_class->constructor (type, n_construct_properties,
							 		       construct_properties));

	xml = rb_glade_xml_new ("cover-progress.glade", "vbox", NULL);
	vbox =
		glade_xml_get_widget (xml, "vbox");
	shell_coverdownloader->priv->progress_artist_label =
		glade_xml_get_widget (xml, "artist_label");
	shell_coverdownloader->priv->progress_album_label =
		glade_xml_get_widget (xml, "album_label");
	shell_coverdownloader->priv->progress_operation_label =
		glade_xml_get_widget (xml, "operation_label");
	shell_coverdownloader->priv->progressbar =
		glade_xml_get_widget (xml, "progressbar");
	shell_coverdownloader->priv->progress_hbox =
		glade_xml_get_widget (xml, "hbox2");
	shell_coverdownloader->priv->progress_const_artist_label =
		glade_xml_get_widget (xml, "const_artist_label");
	shell_coverdownloader->priv->cancel_button =
		glade_xml_get_widget (xml, "cancel_button");
	shell_coverdownloader->priv->close_button =
		glade_xml_get_widget (xml, "close_button");

	g_object_unref (G_OBJECT (xml));

	gtk_window_set_title (GTK_WINDOW (shell_coverdownloader), _("Music Player Cover Download"));
	gtk_container_add (GTK_CONTAINER (shell_coverdownloader), 
			   vbox);
	gtk_window_set_position (GTK_WINDOW (shell_coverdownloader),
 				 GTK_WIN_POS_CENTER);

	g_signal_connect_object (G_OBJECT (shell_coverdownloader->priv->cancel_button),
				 "clicked",
				 G_CALLBACK (rb_shell_coverdownloader_cancel_cb),
				 shell_coverdownloader, 0);

	g_signal_connect_object (G_OBJECT (shell_coverdownloader->priv->close_button),
				 "clicked",
				 G_CALLBACK (rb_shell_coverdownloader_close_cb),
				 shell_coverdownloader, 0);

	g_signal_connect_object (G_OBJECT (shell_coverdownloader),
				 "delete_event",
				 G_CALLBACK (rb_shell_coverdownloader_window_delete_cb),
				 shell_coverdownloader, 0);

	return G_OBJECT (shell_coverdownloader);
}

static void
rb_shell_coverdownloader_close_cb (GtkButton *button,
				   RBShellCoverdownloader *shell_coverdownloader)
{
	/* Close button pressed : we close and destroy the window */
	shell_coverdownloader->priv->closed = TRUE;
}

static void
rb_shell_coverdownloader_cancel_cb (GtkButton *button,
				    RBShellCoverdownloader *shell_coverdownloader)
{
	/* Cancel button pressed : we wait until the end of the current download and we stop the search */
	shell_coverdownloader->priv->cancelled = TRUE;
}

static gboolean
rb_shell_coverdownloader_window_delete_cb (GtkWidget *window,
					   GdkEventAny *event,
					   RBShellCoverdownloader *shell_coverdownloader)
{
	if (!shell_coverdownloader->priv->cancelled)
		/*window destroyed for the first time : we wait until the end of the current download and we stop the search */
		shell_coverdownloader->priv->cancelled = TRUE;
	else
		/* window destroyed for the second time : we close and destroy the window */
		shell_coverdownloader->priv->closed = TRUE;

	return TRUE;
}

static void 
rb_shell_coverdownloader_progress_start (RBShellCoverdownloader *shell_coverdownloader)
{
	gtk_window_resize (GTK_WINDOW (shell_coverdownloader), 350, 150);

	/*gtk_window_set_policy (GTK_WINDOW (shell_coverdownloader), FALSE, TRUE, FALSE);*/
	gtk_window_set_resizable (GTK_WINDOW (shell_coverdownloader),
				  FALSE);

	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (shell_coverdownloader->priv->progressbar), 0);
	gtk_widget_show_all (GTK_WIDGET (shell_coverdownloader));

	/* We only want the cancel button at beginning, not the close button */
	gtk_widget_hide (shell_coverdownloader->priv->close_button);

	/* We refresh the window */
	while (gtk_events_pending ())
		gtk_main_iteration ();
}

static void
rb_shell_coverdownloader_progress_update (RBShellCoverdownloader *shell_coverdownloader,
			  const gchar *artist,
			  const gchar *album)
{
	/* We have already searched for nb_covers_done covers */
	gdouble nb_covers_done = (shell_coverdownloader->priv->nb_covers_found 
				+ shell_coverdownloader->priv->nb_covers_not_found 
				+ shell_coverdownloader->priv->nb_covers_already_exist);

	/* We update the progress bar */
	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (shell_coverdownloader->priv->progressbar), 
				       nb_covers_done / shell_coverdownloader->priv->nb_covers);

	/* We update the artist and the album label */
	gtk_label_set_text (GTK_LABEL (shell_coverdownloader->priv->progress_artist_label), artist);
	gtk_label_set_text (GTK_LABEL (shell_coverdownloader->priv->progress_album_label), album);

	/* We refresh the window */
	while (gtk_events_pending ())
		gtk_main_iteration ();
}

static void
rb_shell_coverdownloader_progress_end (RBShellCoverdownloader *shell_coverdownloader)
{
	char *label_text;

	/* We only want the close button at the end, not the cancel button */
	gtk_widget_hide (shell_coverdownloader->priv->cancel_button);
	gtk_widget_show (shell_coverdownloader->priv->close_button);

	gtk_label_set_text (GTK_LABEL (shell_coverdownloader->priv->progress_artist_label), "");
	gtk_label_set_text (GTK_LABEL (shell_coverdownloader->priv->progress_album_label), "");
	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (shell_coverdownloader->priv->progressbar), 1);

	gtk_label_set_text (GTK_LABEL (shell_coverdownloader->priv->progress_const_artist_label),
			    "Download Finished!");

	/* We show the numbers of covers found and not found */
	label_text = g_strdup_printf ("%i covers found\n%i covers not found\n%i covers already exist", 
				      shell_coverdownloader->priv->nb_covers_found, 
				      shell_coverdownloader->priv->nb_covers_not_found,
				      shell_coverdownloader->priv->nb_covers_already_exist);

	gtk_label_set_text (GTK_LABEL (shell_coverdownloader->priv->progress_const_artist_label),
			    label_text);
	g_free (label_text);

	gtk_widget_destroy (shell_coverdownloader->priv->progress_hbox);
	gtk_widget_destroy (shell_coverdownloader->priv->progress_artist_label);

	/* We wait until the user click on the close button */
	while (!shell_coverdownloader->priv->closed)
		gtk_main_iteration ();
}

static void
rb_shell_coverdownloader_nocover (RBShellCoverdownloader *shell_coverdownloader)
{
	/* This information will be displayed at the end of the search */
	shell_coverdownloader->priv->nb_covers_not_found ++;
}

void
rb_shell_coverdownloader_get_covers_from_genres (RBShellCoverdownloader *shell_coverdownloader,
						 RBLibrarySource *source,
						 GList *selected_genres,
						 RBShellCoverdownloaderOperation operation)
{
	GtkTreeModel *main_model;
	GList *temp;

	shell_coverdownloader->priv->source = source;
	if (!selected_genres) {
		/* No genre was selected, we do a query for the entire database */
		main_model = GTK_TREE_MODEL (rhythmdb_query_model_new_empty (shell_coverdownloader->priv->db));
		rhythmdb_do_full_query (shell_coverdownloader->priv->db, main_model,
     	  				RHYTHMDB_QUERY_PROP_EQUALS,
      	 				RHYTHMDB_PROP_TYPE, RHYTHMDB_ENTRY_TYPE_SONG,
					RHYTHMDB_QUERY_END);
		/* We add the result of the query to a list */
		rb_shell_coverdownloader_add_covers_to_list_from_model (shell_coverdownloader, main_model);
	} else {
		for (temp = selected_genres; temp; temp = temp->next) {
			/* We make a query for each selected genre */
			main_model = GTK_TREE_MODEL (rhythmdb_query_model_new_empty (shell_coverdownloader->priv->db));
			rhythmdb_do_full_query (shell_coverdownloader->priv->db, main_model,
   		  				RHYTHMDB_QUERY_PROP_EQUALS,
      	 					RHYTHMDB_PROP_TYPE, RHYTHMDB_ENTRY_TYPE_SONG,
						RHYTHMDB_QUERY_PROP_EQUALS,
						RHYTHMDB_PROP_GENRE, (gchar *) temp->data,
						RHYTHMDB_QUERY_END);

			/* We add the result of the query to a list */
			rb_shell_coverdownloader_add_covers_to_list_from_model (shell_coverdownloader, main_model);
		}
	}

	/* We search for all the covers of the entry in the list */
	rb_shell_coverdownloader_get_covers_from_list (shell_coverdownloader, operation);
}

void
rb_shell_coverdownloader_get_covers_from_artists (RBShellCoverdownloader *shell_coverdownloader,
						  RBLibrarySource *source,
						  GList *selected_artists,
						  RBShellCoverdownloaderOperation operation)
{
	GtkTreeModel *main_model;
	GList *temp;

	shell_coverdownloader->priv->source = source;
	if (!selected_artists)
		return;

	for (temp = selected_artists; temp; temp = temp->next) {
		/* We make a query for each selected artist */
		main_model = GTK_TREE_MODEL (rhythmdb_query_model_new_empty (shell_coverdownloader->priv->db));
		rhythmdb_do_full_query (shell_coverdownloader->priv->db, main_model,
     	  				RHYTHMDB_QUERY_PROP_EQUALS,
      	 				RHYTHMDB_PROP_TYPE, RHYTHMDB_ENTRY_TYPE_SONG,
					RHYTHMDB_QUERY_PROP_EQUALS,
					RHYTHMDB_PROP_ARTIST, (gchar *) temp->data,
					RHYTHMDB_QUERY_END);
		/* We add the result of the query to a list */
		rb_shell_coverdownloader_add_covers_to_list_from_model (shell_coverdownloader, main_model);
	}

	/* We search for all the covers of the entry in the list */
	rb_shell_coverdownloader_get_covers_from_list (shell_coverdownloader, operation);
}

void
rb_shell_coverdownloader_get_covers_from_albums (RBShellCoverdownloader *shell_coverdownloader,
						 RBLibrarySource *source,
						 GList *selected_artists_albums,
						 RBShellCoverdownloaderOperation operation)
{
	GtkTreeModel *main_model;
	GList *temp;
	gchar *artist;
	gchar *album;

	shell_coverdownloader->priv->source = source;
	if (!selected_artists_albums)
		return;

	for (temp = selected_artists_albums; temp; temp = temp->next) {
		/* We make a query for each selected album */
		main_model = GTK_TREE_MODEL (rhythmdb_query_model_new_empty (shell_coverdownloader->priv->db));
		album = (gchar *) temp->data;
		if (!album)
			break;
		temp = temp->next;
		if (!temp)
			break;
		artist = (gchar *) temp->data;
		if (!artist)
			break;
		rhythmdb_do_full_query (shell_coverdownloader->priv->db, main_model,
     	  				RHYTHMDB_QUERY_PROP_EQUALS,
      	 				RHYTHMDB_PROP_TYPE, RHYTHMDB_ENTRY_TYPE_SONG,
					RHYTHMDB_QUERY_PROP_EQUALS,
					RHYTHMDB_PROP_ARTIST, artist,
					RHYTHMDB_QUERY_PROP_EQUALS,
					RHYTHMDB_PROP_ALBUM, album,
					RHYTHMDB_QUERY_END);
		/* We add the result of the query to a list */
		rb_shell_coverdownloader_add_covers_to_list_from_model (shell_coverdownloader, main_model);
	}

	/* We search for all the covers of the entry in the list */
	rb_shell_coverdownloader_get_covers_from_list (shell_coverdownloader, operation);
}

static void
rb_shell_coverdownloader_add_covers_to_list_from_model (RBShellCoverdownloader *shell_coverdownloader,
							GtkTreeModel *main_model)
{
	RhythmDBEntry *tree_entry;
	GtkTreeIter iter;
	const gchar *previous_artist = g_strdup ("");
	const gchar *previous_album = g_strdup ("");
	const gchar *artist;
	const gchar *album;
	gboolean result;

	result = gtk_tree_model_get_iter_first (main_model, &iter);

	while (result) {
		/* We get the entry from the tree model */
		gtk_tree_model_get (main_model, &iter, 0, &tree_entry, -1);

		artist = rb_refstring_get (tree_entry->artist);
		album = rb_refstring_get (tree_entry->album);

		if (!album || !artist) {
			result = gtk_tree_model_iter_next (main_model, &iter);
			break;
		}

		/* If it isn't the same album as the precedent one, we add it to the list */
		if (strcmp (previous_artist, artist) || strcmp (previous_album, album)) {
				shell_coverdownloader->priv->nb_covers ++;
				shell_coverdownloader->priv->entries = g_slist_append (shell_coverdownloader->priv->entries,
										       tree_entry);
		}

		previous_artist = artist;
		previous_album = album;
		
		result = gtk_tree_model_iter_next (main_model, &iter);
	}
}

static void
rb_shell_coverdownloader_get_covers_from_list (RBShellCoverdownloader *shell_coverdownloader,
					       RBShellCoverdownloaderOperation operation)
{
	GSList *entries_temp = shell_coverdownloader->priv->entries;

	if (!entries_temp)
		return;

	/* We show window with the progress bar */
	if (operation == GET_LOCAL_COVERS || operation == GET_AMAZON_COVERS)
		rb_shell_coverdownloader_progress_start (shell_coverdownloader);

	/* While there are still covers to search */
	while (entries_temp) {
		/* The user has pressed the "cancel button" or has closed the window : we stop the search */
		if (shell_coverdownloader->priv->cancelled)
			break;

		/* We search for a new cover */
		rb_shell_coverdownloader_get_covers_from_entry (shell_coverdownloader,
								entries_temp->data,
								operation);
		entries_temp = g_slist_next (entries_temp);
	}
	g_slist_free (shell_coverdownloader->priv->entries);

	/* We change the window to show a close button and infos about the search */
	if (operation == GET_LOCAL_COVERS || operation == GET_AMAZON_COVERS)
		rb_shell_coverdownloader_progress_end (shell_coverdownloader);
}

static void
rb_shell_coverdownloader_get_covers_from_entry (RBShellCoverdownloader *shell_coverdownloader,
						RhythmDBEntry *entry,
						RBShellCoverdownloaderOperation operation)
{
	const gchar *artist;
	const gchar *album;
	gchar *location;

	artist = rb_refstring_get (entry->artist);
	album = rb_refstring_get (entry->album);

	if (!album || !artist)
		return;

	/* If the user has said that we don't have to search dor this cover, we stop */
	if (rb_cover_get_do_not_search (artist, album))
		return;
	
	switch (operation)
	{
	case GET_LOCAL_COVERS:
	{
		/* We update the progress bar */
		rb_shell_coverdownloader_progress_update (shell_coverdownloader, artist, album);

		if (rb_cover_cover_exists (artist, album)) {
			/* The cover already exists, we do nothing */
			shell_coverdownloader->priv->nb_covers_already_exist++;
		} else {
			location = (char *) rhythmdb_entry_get_string (entry,
								       RHYTHMDB_PROP_LOCATION);

			if (!location)
				return;

			/* We search for the cover in the song directory */
			rb_shell_coverdownloader_find_local_image (shell_coverdownloader, artist, album, location);
		}
	}
		break;

	case GET_AMAZON_COVERS:
	{
		/* We update the progress bar */
		rb_shell_coverdownloader_progress_update (shell_coverdownloader, artist, album);

		if (rb_cover_cover_exists (artist, album))
			/* The cover already exists, we do nothing */
			shell_coverdownloader->priv->nb_covers_already_exist++;
		else
			/* We search for the cover on amazon */
			rb_shell_coverdownloader_find_amazon_image (shell_coverdownloader, artist, album);
	}
		break;

	case REMOVE_COVERS:
	{
		/* We remove the cover from the ~/.gnome2/rhythmbox/covers/ directory */
		rb_cover_remove_cover (artist, album);
	}
		break;

	default:
		break;
	}
}

static void 
rb_shell_coverdownloader_find_local_image (RBShellCoverdownloader *shell_coverdownloader,
			   		   const char *artist,
			   		   const char *album,
			   		   const char *location)
{
	int size;
	char *data;
	GnomeVFSResult result;

	/* If a cover is found, it is loaded in data */
	result = rb_cover_load_local_cover (location,
					    &size,
					    &data);

	if (result != GNOME_VFS_OK) {
		/* No cover was found or an error occured */
		rb_shell_coverdownloader_nocover (shell_coverdownloader);
		return;
	}

	/* If the cover is too big or too small (blank amazon image), we don't use it */
	if (!rb_cover_size_is_valid (size)) {
		g_free (data);
		rb_shell_coverdownloader_nocover (shell_coverdownloader);
		return;
	}

	/* We save the cover in the ~/.gnome2/rhythmbox/covers/ directory */
	result = rb_cover_save_cover (artist,
				      album,
				      data,
				      size,
				      LOCAL_COVER,
				      NULL,
				      OVERWRITE_MODE_SKIP);
	g_free (data);

	if (result == GNOME_VFS_OK)
		shell_coverdownloader->priv->nb_covers_found ++;
	else
		shell_coverdownloader->priv->nb_covers_not_found ++;
}

static void 
rb_shell_coverdownloader_find_amazon_image (RBShellCoverdownloader *shell_coverdownloader,
					    const char *artist,
					    const char *album)
{
	GArray *size;
	GList *data = NULL, *cover_uris = NULL;
	GnomeVFSResult result;

	size = g_array_new (TRUE, TRUE, sizeof (int));

	/* If a cover is found on amazon, it is loaded in data(0) */
	result = rb_cover_load_amazon_covers (artist,
					      album,
					      &cover_uris,
					      &size,
					      &data,
					      GET_FIRST_COVER,
					      AMAZON_MEDIUM_COVER);
	/* If the cover is not too big and not too small (blank amazon image), we save it */
	if (result == GNOME_VFS_OK && rb_cover_size_is_valid (g_array_index (size, int, 0))) {
		result = rb_cover_save_cover (artist,
					      album,
				 	      g_list_nth_data (data, 0),
					      g_array_index (size, int, 0),
					      AMAZON_COVER,
					      g_list_nth_data (cover_uris, 0),
					      OVERWRITE_MODE_SKIP);

		if (result == GNOME_VFS_OK)
			shell_coverdownloader->priv->nb_covers_found ++;
		else
			shell_coverdownloader->priv->nb_covers_not_found ++;
	} else {
		rb_shell_coverdownloader_nocover (shell_coverdownloader);
	}

	g_array_free (size, TRUE);
	g_list_foreach (data, (GFunc) g_free, NULL);
	g_list_free (data);
	g_list_foreach (cover_uris, (GFunc) g_free, NULL);
	g_list_free (cover_uris);
}

static void
rb_shell_coverdownloader_get_single_local_cover (RBShellCoverdownloader *shell_coverdownloader,
						 RBLibrarySource *source,
						 const gchar *artist,
			        		 const gchar *album)
{
	GtkWidget *fileselection;
	GtkWidget *dialog;
	GnomeVFSResult result;
	gchar *data;
	gint size;

	/* If the user has only selected one album, we launch a dialog box to select a cover */
	shell_coverdownloader->priv->source = source;
	
	fileselection = gtk_file_selection_new ("Select a file");
   
	gtk_window_set_modal (GTK_WINDOW (fileselection), TRUE);

	gtk_file_selection_set_filename (GTK_FILE_SELECTION (fileselection),
					 g_get_home_dir ());
					 
	if (gtk_dialog_run (GTK_DIALOG (fileselection)) != GTK_RESPONSE_OK) {
		gtk_widget_destroy (fileselection);
		return;	
	}

	/* We the user has choosen a cover, it is loaded in data */
	result = gnome_vfs_read_entire_file (gtk_file_selection_get_filename (GTK_FILE_SELECTION (fileselection)),
					     &size,
					     &data);

	gtk_widget_destroy (fileselection);

	if (result != GNOME_VFS_OK){
		dialog = gtk_message_dialog_new(NULL,
						GTK_DIALOG_MODAL,
						GTK_MESSAGE_ERROR,
						GTK_BUTTONS_OK,
						"Error reading file");
		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);
	}

	/* We save the cover in the ~/.gnome2/rhythmbox/covers/ directory */
	result = rb_cover_save_cover (artist, album, data, size, LOCAL_COVER, NULL, OVERWRITE_MODE_ASK);
	g_free (data);

	if (result != GNOME_VFS_OK){
		dialog = gtk_message_dialog_new(NULL,
						GTK_DIALOG_MODAL,
						GTK_MESSAGE_ERROR,
						GTK_BUTTONS_OK,
						"Error saving file");
		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);
	}
}

static void
rb_shell_coverdownloader_get_single_amazon_cover (RBShellCoverdownloader *shell_coverdownloader,
						  RBLibrarySource *source,
						  const gchar *artist,
						  const gchar *album)
{
	GtkWidget *coverselect;

	/* If the user has only selected one album, we launch the coverselect widget
	 * to make more accurate search on amazon */
	shell_coverdownloader->priv->source = source;
	coverselect = rb_shell_coverselect_new (artist, album);

	gtk_dialog_run (GTK_DIALOG (coverselect));
	gtk_widget_destroy (coverselect);
}

void
rb_shell_coverdownloader_get_single_cover (RBShellCoverdownloader *shell_coverdownloader,
					   RBLibrarySource *source,
					   const gchar *artist,
					   const gchar *album,
					   RBShellCoverdownloaderOperation operation)
{
	switch (operation)
	{
	case GET_LOCAL_COVERS:
		rb_shell_coverdownloader_get_single_local_cover (shell_coverdownloader,
								 source,
								 artist,
								 album);
		break;
	case GET_AMAZON_COVERS:
		rb_shell_coverdownloader_get_single_amazon_cover (shell_coverdownloader,
								  source,
								  artist,
								  album);
		break;
	default:
		break;
	}
}
