/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
 * @file c_esp.c
 * @brief ROHC compression context for the ESP profile.
 * @author Didier Barvaux <didier.barvaux@toulouse.viveris.com>
 * @author The hackers from ROHC for Linux
 * @author FWX <rohc_team@dialine.fr>
 */

#include "c_esp.h"
#include "rohc_traces.h"
#include "rohc_packets.h"
#include "crc.h"

#include <stdlib.h>


/*
 * Private function prototypes.
 */

int esp_code_dynamic_esp_part(const struct c_context *context,
                              const unsigned char *next_header,
                              unsigned char *const dest,
                              int counter);

int esp_changed_esp_dynamic(const struct c_context *context,
                            const struct esphdr *esp);


/**
 * @brief Create a new ESP context and initialize it thanks to the given IP/ESP
 *        packet.
 *
 * This function is one of the functions that must exist in one profile for the
 * framework to work.
 *
 * @param context The compression context
 * @param ip      The IP/ESP packet given to initialize the new context
 * @return        1 if successful, 0 otherwise
 */
int c_esp_create(struct c_context *const context, const struct ip_packet *ip)
{
	struct c_generic_context *g_context;
	struct sc_esp_context *esp_context;
	struct ip_packet ip2;
	const struct ip_packet *last_ip_header;
	const struct esphdr *esp;
	unsigned int ip_proto;

	/* create and initialize the generic part of the profile context */
	if(!c_generic_create(context, ip))
	{
		rohc_debugf(0, "generic context creation failed\n");
		goto quit;
	}
	g_context = (struct c_generic_context *) context->specific;

	/* check if packet is IP/ESP or IP/IP/ESP */
	ip_proto = ip_get_protocol(ip);
	if(ip_proto == IPPROTO_IPIP || ip_proto == IPPROTO_IPV6)
	{
		/* get the last IP header */
		if(!ip_get_inner_packet(ip, &ip2))
		{
			rohc_debugf(0, "cannot create the inner IP header\n");
			goto clean;
		}

		/* two IP headers, the last IP header is the second one */
		last_ip_header = &ip2;

		/* get the transport protocol */
		ip_proto = ip_get_protocol(last_ip_header);
	}
	else
	{
		/* only one single IP header, the last IP header is the first one */
		last_ip_header = ip;
	}

	if(ip_proto != IPPROTO_ESP)
	{
		rohc_debugf(0, "next header is not ESP (%d), cannot use this profile\n",
		            ip_proto);
		goto clean;
	}

	esp = (struct esphdr *) ip_get_next_layer(last_ip_header);

	/* create the ESP part of the profile context */
	esp_context = malloc(sizeof(struct sc_esp_context));
	if(esp_context == NULL)
	{
	  rohc_debugf(0, "no memory for the ESP part of the profile context\n");
	  goto clean;
	}
	g_context->specific = esp_context;

	/* initialize the ESP part of the profile context */
	esp_context->esp_sequence_number_change_count = 0;
	esp_context->esp_last_sequence_number = -1;
        memcpy(&(esp_context->old_esp), esp, sizeof(struct esphdr));
        
	/* init the ESP-specific temporary variables */
	esp_context->tmp_variables.send_esp_dynamic = -1;

	/* init the ESP-specific variables and functions */
	g_context->next_header_proto = IPPROTO_ESP;
	g_context->next_header_len = sizeof(struct esphdr);
	g_context->decide_state = esp_decide_state;
	g_context->init_at_IR = NULL;
	g_context->code_static_part = esp_code_static_esp_part;
	g_context->code_dynamic_part = esp_code_dynamic_esp_part;
	g_context->code_UO_packet_head = NULL;
	g_context->code_UO_packet_tail = NULL;
	g_context->compute_crc_static = esp_compute_crc_static;
	g_context->compute_crc_dynamic = esp_compute_crc_dynamic;

	return 1;

clean:
	c_generic_destroy(context);
quit:
	return 0;
}


/**
 * @brief Check if the IP/ESP packet belongs to the context
 *
 * Conditions are:
 *  - the number of IP headers must be the same as in context
 *  - IP version of the two IP headers must be the same as in context
 *  - IP packets must not be fragmented
 *  - the source and destination addresses of the two IP headers must match the
 *    ones in the context
 *  - the transport protocol must be ESP
 *  - the security parameters index of the ESP header must match the ones in
 *    the context
 *  - IPv6 only: the Flow Label of the two IP headers must match the ones the
 *    context
 *
 * This function is one of the functions that must exist in one profile for the
 * framework to work.
 *
 * @param context The compression context
 * @param ip      The IP/ESP packet to check
 * @return        1 if the IP/ESP packet belongs to the context,
 *                0 if it does not belong to the context and
 *                -1 if the profile cannot compress it or an error occurs
 */
int c_esp_check_context(const struct c_context *context,
                        const struct ip_packet *ip)
{
	struct c_generic_context *g_context;
	struct sc_esp_context *esp_context;
	struct ip_header_info *ip_flags;
	struct ip_header_info *ip2_flags;
	struct ip_packet ip2;
	const struct ip_packet *last_ip_header;
	const struct esphdr *esp;
	ip_version version;
	unsigned int ip_proto;
	int is_ip_same;
	int is_ip2_same;
	int is_esp_same;

	g_context = (struct c_generic_context *) context->specific;
	esp_context = (struct sc_esp_context *) g_context->specific;
	ip_flags = &g_context->ip_flags;
	ip2_flags = &g_context->ip2_flags;

	/* check the IP version of the first header */
	version = ip_get_version(ip);
	if(version != ip_flags->version)
		goto bad_context;

	/* compare the addresses of the first header */
	if(version == IPV4)
	{
		is_ip_same = ip_flags->info.v4.old_ip.saddr == ipv4_get_saddr(ip) &&
		             ip_flags->info.v4.old_ip.daddr == ipv4_get_daddr(ip);
	}
	else /* IPV6 */
	{
		is_ip_same =
			IPV6_ADDR_CMP(&ip_flags->info.v6.old_ip.ip6_src, ipv6_get_saddr(ip)) &&
			IPV6_ADDR_CMP(&ip_flags->info.v6.old_ip.ip6_dst, ipv6_get_daddr(ip));
	}

	if(!is_ip_same)
		goto bad_context;

	/* compare the Flow Label of the first header if IPv6 */
	if(version == IPV6 && ipv6_get_flow_label(ip) !=
	   IPV6_GET_FLOW_LABEL(ip_flags->info.v6.old_ip))
		goto bad_context;

	/* check the second IP header */
	ip_proto = ip_get_protocol(ip);
	if(ip_proto == IPPROTO_IPIP || ip_proto == IPPROTO_IPV6)
	{
		/* check if the context used to have a second IP header */
		if(!g_context->is_ip2_initialized)
			goto bad_context;

		/* get the second IP header */
		if(!ip_get_inner_packet(ip, &ip2))
		{
			rohc_debugf(0, "cannot create the inner IP header\n");
			goto error;
		}

		/* check the IP version of the second header */
		version = ip_get_version(&ip2);
		if(version != ip2_flags->version)
			goto bad_context;

		/* compare the addresses of the second header */
		if(version == IPV4)
		{
			is_ip2_same = ip2_flags->info.v4.old_ip.saddr == ipv4_get_saddr(&ip2) &&
			              ip2_flags->info.v4.old_ip.daddr == ipv4_get_daddr(&ip2);
		}
		else /* IPV6 */
		{
			is_ip2_same = IPV6_ADDR_CMP(&ip2_flags->info.v6.old_ip.ip6_src,
			                            ipv6_get_saddr(&ip2)) &&
			              IPV6_ADDR_CMP(&ip2_flags->info.v6.old_ip.ip6_dst,
			                            ipv6_get_daddr(&ip2));
		}

		if(!is_ip2_same)
			goto bad_context;

		/* compare the Flow Label of the second header if IPv6 */
		if(version == IPV6 && ipv6_get_flow_label(&ip2) !=
		   IPV6_GET_FLOW_LABEL(ip2_flags->info.v6.old_ip))
			goto bad_context;

		/* get the last IP header */
		last_ip_header = &ip2;

		/* get the transport protocol */
		ip_proto = ip_get_protocol(&ip2);
	}
	else /* no second IP header */
	{
		/* check if the context used not to have a second header */
		if(g_context->is_ip2_initialized)
			goto bad_context;

		/* only one single IP header, the last IP header is the first one */
		last_ip_header = ip;
	}

	/* check the transport protocol */
	if(ip_proto != IPPROTO_ESP)
		goto bad_context;
	
	/* check Security parameters index (SPI) */
	esp = (struct esphdr *) ip_get_next_layer(last_ip_header);
	is_esp_same = esp_context->old_esp.security_parameters_index == esp->security_parameters_index;

	return is_esp_same;

bad_context:
	return 0;
error:
	return -1;
}


/**
 * @brief Encode an IP/ESP packet according to a pattern decided by several
 *        different factors.
 *
 * @param context        The compression context
 * @param ip             The IP packet to encode
 * @param packet_size    The length of the IP packet to encode
 * @param dest           The rohc-packet-under-build buffer
 * @param dest_size      The length of the rohc-packet-under-build buffer
 * @param packet_type    OUT: The type of ROHC packet that is created
 * @param payload_offset The offset for the payload in the IP packet
 * @return               The length of the created ROHC packet
 *                       or -1 in case of failure
 */
int c_esp_encode(struct c_context *const context,
                 const struct ip_packet *ip,
                 const int packet_size,
                 unsigned char *const dest,
                 const int dest_size,
                 rohc_packet_t *const packet_type,
                 int *const payload_offset)
{
	struct c_generic_context *g_context;
	struct sc_esp_context *esp_context;
	struct ip_packet ip2;
	const struct ip_packet *last_ip_header;
	const struct esphdr *esp;
	unsigned int ip_proto;
	int size;

	g_context = (struct c_generic_context *) context->specific;
	if(g_context == NULL)
	{
		rohc_debugf(0, "generic context not valid\n");
		return -1;
	}

	esp_context = (struct sc_esp_context *) g_context->specific;
	if(esp_context == NULL)
	{
		rohc_debugf(0, "ESP context not valid\n");
		return -1;
	}

	ip_proto = ip_get_protocol(ip);
	if(ip_proto == IPPROTO_IPIP || ip_proto == IPPROTO_IPV6)
	{
		/* get the last IP header */
		if(!ip_get_inner_packet(ip, &ip2))
		{
			rohc_debugf(0, "cannot create the inner IP header\n");
			return -1;
		}
		last_ip_header = &ip2;

		/* get the transport protocol */
		ip_proto = ip_get_protocol(last_ip_header);
	}
	else
	{
		/* only one single IP header, the last IP header is the first one */
		last_ip_header = ip;
	}

	if(ip_proto != IPPROTO_ESP)
	{
		rohc_debugf(0, "packet is not an ESP packet\n");
		return -1;
	}
	esp = (struct esphdr *) ip_get_next_layer(last_ip_header);

	/* how many ESP fields changed? */
	esp_context->tmp_variables.send_esp_dynamic = esp_changed_esp_dynamic(context, esp);

	/* encode the IP packet */
	size = c_generic_encode(context, ip, packet_size, dest, dest_size,
	                        packet_type, payload_offset);
	if(size < 0)
		goto quit;

	/* update the context with the new ESP header */
	if(g_context->tmp_variables.packet_type == PACKET_IR ||
	   g_context->tmp_variables.packet_type == PACKET_IR_DYN)
                memcpy(&(esp_context->old_esp), esp, sizeof(struct esphdr));

quit:
	return size;
}


/**
 * @brief Decide the state that should be used for the next packet compressed
 *        with the ROHC ESP profile.
 *
 * The three states are:
 *  - Initialization and Refresh (IR),
 *  - First Order (FO),
 *  - Second Order (SO).
 *
 * @param context The compression context
 */
void esp_decide_state(struct c_context *const context)
{
	struct c_generic_context *g_context;
	struct sc_esp_context *esp_context;

	g_context = (struct c_generic_context *) context->specific;
	esp_context = (struct sc_esp_context *) g_context->specific;

        rohc_debugf(3, "current state %d\n", context->state);

	if(esp_context->tmp_variables.send_esp_dynamic)
		change_state(context, IR);
	else
		/* generic function used by the IP-only, UDP and UDP-Lite profiles */
		decide_state(context);

        rohc_debugf(3, "next state %d\n", context->state);
}



/**
 * @brief Build the static part of the ESP header.
 *
 * \verbatim

 Static part of ESP header (5.7.7.7):
 
    +---+---+---+---+---+---+---+---+
 1  /  Security Parameters Index    /   4 octets
    +---+---+---+---+---+---+---+---+

\endverbatim
 *
 * @param context     The compression context
 * @param next_header The ESP header
 * @param dest        The rohc-packet-under-build buffer
 * @param counter     The current position in the rohc-packet-under-build buffer
 * @return            The new position in the rohc-packet-under-build buffer 
 */
int esp_code_static_esp_part(const struct c_context *context,
                             const unsigned char *next_header,
                             unsigned char *const dest,
                             int counter)
{
	const struct esphdr *esp = (struct esphdr *) next_header;

	/* part 1 */
	rohc_debugf(3, "ESP security parameters index = 0x%08x\n", ntohl(esp->security_parameters_index));
	memcpy(&dest[counter], &esp->security_parameters_index, 4);
	counter += 4;

	return counter;
}


/**
 * @brief Build the dynamic part of the ESP header.
 *
 * \verbatim

 Dynamic part of ESP header (5.7.7.7):

    +---+---+---+---+---+---+---+---+
 1  /         Sequence Number       /   4 octets
    +---+---+---+---+---+---+---+---+

\endverbatim
 *
 * @param context     The compression context
 * @param next_header The ESP header
 * @param dest        The rohc-packet-under-build buffer
 * @param counter     The current position in the rohc-packet-under-build buffer
 * @return            The new position in the rohc-packet-under-build buffer 
 */
int esp_code_dynamic_esp_part(const struct c_context *context,
                              const unsigned char *next_header,
                              unsigned char *const dest,
                              int counter)
{
	struct c_generic_context *g_context;
	struct sc_esp_context *esp_context;
	const struct esphdr *esp;

	g_context = (struct c_generic_context *) context->specific;
	esp_context = (struct sc_esp_context *) g_context->specific;

	esp = (struct esphdr *) next_header;

	/* part 1 */
	rohc_debugf(3, "ESP sequence number = 0x%08x\n", ntohl(esp->sequence_number));
	memcpy(&dest[counter], &esp->sequence_number, 4);
	counter += 4;
	esp_context->esp_last_sequence_number = ntohl(esp->sequence_number);
	esp_context->esp_sequence_number_change_count++;

	return counter;
}


/**
 * @brief Check if the dynamic part of the ESP header changed.
 *
 * @param context The compression context
 * @param esp     The ESP header
 * @return        The number of ESP fields that changed
 */
int esp_changed_esp_dynamic(const struct c_context *context,
                            const struct esphdr *esp)
{
	const struct c_generic_context *g_context;
	struct sc_esp_context *esp_context;
	u_int32_t sequence_number;

	g_context = (struct c_generic_context *) context->specific;
	esp_context = (struct sc_esp_context *) g_context->specific;

	sequence_number = ntohl(esp->sequence_number);

        rohc_debugf(3, "sequence_number 0x%08x last 0x%08x old 0x%08x change_count %d\n", 
                    sequence_number, esp_context->esp_last_sequence_number,
                    ntohl(esp_context->old_esp.sequence_number),
                    esp_context->esp_sequence_number_change_count);

	if( (sequence_number != (esp_context->esp_last_sequence_number+1)) ||
	   (esp_context->esp_sequence_number_change_count < MAX_IR_COUNT))
	{
	        if(sequence_number != (esp_context->esp_last_sequence_number+1))
                {		
                        esp_context->esp_last_sequence_number = sequence_number;
			esp_context->esp_sequence_number_change_count = 0;
                        rohc_debugf(3, "sequence_number_change_count = 0\n");
                }
        	else
        	{
                        rohc_debugf(3, "sequence_number_change_count = %d\n", esp_context->esp_sequence_number_change_count);
        	}
		return 1;
	}
	else
	{
                esp_context->esp_last_sequence_number = sequence_number;
                rohc_debugf(3, "sequence_number_change_count = %d\n", esp_context->esp_sequence_number_change_count);
		return 0;
	}
}


/**
 * @brief Define the compression part of the ESP profile as described
 *        in the RFC 3095.
 */
struct c_profile c_esp_profile =
{
	IPPROTO_ESP,         /* IP protocol */
	NULL,                /* list of UDP ports, not relevant for UDP */
	ROHC_PROFILE_ESP,    /* profile ID (see 8 in RFC 3095) */
	"ESP / Compressor",  /* profile description */
	c_esp_create,        /* profile handlers */
	c_generic_destroy,
	c_esp_check_context,
	c_esp_encode,
	c_generic_feedback,
};
