/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
 * @file d_esp.c
 * @brief ROHC decompression context for the ESP profile.
 * @author Didier Barvaux <didier.barvaux@toulouse.viveris.com>
 * @author The hackers from ROHC for Linux
 * @author FWX <rohc_team@dialine.fr>
 */

#include "d_esp.h"
#include "rohc_bit_ops.h"
#include "rohc_traces.h"
#include "rohc_debug.h"
#include "crc.h"

#include <netinet/ip.h>


/*
 * Private function prototypes.
 */

/*
int esp_decode_uo_tail_esp(struct d_generic_context *context,
                           const unsigned char *packet,
                           unsigned int length,
                           unsigned char *dest);
*/

/**
 * @brief Create the ESP decompression context.
 *
 * This function is one of the functions that must exist in one profile for the
 * framework to work.
 *
 * @return The newly-created ESP decompression context
 */
void * d_esp_create(void)
{
	struct d_generic_context *context;
	struct d_esp_context *esp_context;

	/* create the generic context */
	context = d_generic_create();
	if(context == NULL)
		goto quit;

	/* create the ESP-specific part of the context */
	esp_context = malloc(sizeof(struct d_esp_context));
	if(esp_context == NULL)
	{
		rohc_debugf(0, "cannot allocate memory for the ESP-specific context\n");
		goto destroy_context;
	}
	bzero(esp_context, sizeof(struct d_esp_context));
	context->specific = esp_context;

	/* the ESP sequence number field present flag will be initialized
	 * with the IR packets */
	esp_context->esp_sequence_number = 0;

	/* some ESP-specific values and functions */
	context->next_header_len = sizeof(struct esphdr);
	context->build_next_header = esp_build_uncompressed_esp;
	context->decode_static_next_header = esp_decode_static_esp;
	context->decode_dynamic_next_header = esp_decode_dynamic_esp;
	context->decode_uo_tail = NULL; // esp_decode_uo_tail_esp;
	context->compute_crc_static = esp_compute_crc_static;
	context->compute_crc_dynamic = esp_compute_crc_dynamic;

	/* create the ESP-specific part of the header changes */
	context->last1->next_header_len = sizeof(struct esphdr);
	context->last1->next_header = malloc(sizeof(struct esphdr));
	if(context->last1->next_header == NULL)
	{
		rohc_debugf(0, "cannot allocate memory for the ESP-specific "
		               "part of the header changes last1\n");
		goto free_esp_context;
	}
	bzero(context->last1->next_header, sizeof(struct esphdr));

	context->last2->next_header_len = sizeof(struct esphdr);
	context->last2->next_header = malloc(sizeof(struct esphdr));
	if(context->last2->next_header == NULL)
	{
		rohc_debugf(0, "cannot allocate memory for the ESP-specific "
		               "part of the header changes last2\n");
		goto free_last1_next_header;
	}
	bzero(context->last2->next_header, sizeof(struct esphdr));

	context->active1->next_header_len = sizeof(struct esphdr);
	context->active1->next_header = malloc(sizeof(struct esphdr));
	if(context->active1->next_header == NULL)
	{
		rohc_debugf(0, "cannot allocate memory for the ESP-specific "
		               "part of the header changes active1\n");
		goto free_last2_next_header;
	}
	bzero(context->active1->next_header, sizeof(struct esphdr));

	context->active2->next_header_len = sizeof(struct esphdr);
	context->active2->next_header = malloc(sizeof(struct esphdr));
	if(context->active2->next_header == NULL)
	{
		rohc_debugf(0, "cannot allocate memory for the ESP-specific "
		               "part of the header changes active2\n");
		goto free_active1_next_header;
	}
	bzero(context->active2->next_header, sizeof(struct esphdr));

	/* set next header to ESP */
	context->next_header_proto = IPPROTO_ESP;

	return context;

free_active1_next_header:
	zfree(context->active1->next_header);
free_last2_next_header:
	zfree(context->last2->next_header);
free_last1_next_header:
	zfree(context->last1->next_header);
free_esp_context:
	zfree(esp_context);
destroy_context:
	d_generic_destroy(context);
quit:
	return NULL;
}


/**
 * @brief Destroy the context.
 * 
 * This function is one of the functions that must exist in one profile for the
 * framework to work.
 *
 * @param context The compression context
 */
void d_esp_destroy(void *context)
{
	struct d_generic_context *c = context;

	if(c != NULL)
	{
		/* clean ESP-specific memory */
		if(c->last1 != NULL && c->last1->next_header != NULL)
			zfree(c->last1->next_header);
		if(c->last2 != NULL && c->last2->next_header != NULL)
			zfree(c->last2->next_header);
		if(c->active1 != NULL && c->active1->next_header != NULL)
			zfree(c->active1->next_header);
		if(c->active2 != NULL && c->active2->next_header != NULL)
			zfree(c->active2->next_header);

		/* destroy the generic decompression context (c->specific is
		 * destroyed by d_generic_destroy) */
		d_generic_destroy(c);
	}
}

/**
 * @brief Get the size of the static part of an IR packet
 * @return the size
 */
int esp_get_static_size(void)
{
	return 4;
}

/**
 * @brief Decode one IR packet for the ESP profile.
 *
 * This function is one of the functions that must exist in one profile for the
 * framework to work.
 *
 * @param decomp          The ROHC decompressor
 * @param context         The decompression context
 * @param rohc_packet     The ROHC packet to decode
 * @param rohc_length     The length of the ROHC packet to decode
 * @param large_cid_len   The length of the large CID field
 * @param is_addcid_used  Whether the add-CID field is present or not
 * @param dest            The decoded IP packet
 * @return                The length of the uncompressed IP packet
 *                        or ROHC_OK_NO_DATA if packet is feedback only
 *                        or ROHC_ERROR if an error occurs
 */
int d_esp_decode_ir(struct rohc_decomp *decomp,
                    struct d_context *context,
                    const unsigned char *rohc_packet,
                    const unsigned int rohc_length,
                    int large_cid_len,
                    int is_addcid_used,
                    unsigned char *dest)
{
	struct d_generic_context *g_context = context->specific;
	struct d_esp_context *esp_context = g_context->specific;

	esp_context->esp_sequence_number = 0;

	return d_generic_decode_ir(decomp, context, rohc_packet, rohc_length,
	                           large_cid_len, is_addcid_used, dest);
}


/**
 * @brief Find the length of the IR header.
 *
 * This function is one of the functions that must exist in one profile for the
 * framework to work.
 *
 * \verbatim

 Basic structure of the IR packet (5.7.7.1):

      0   1   2   3   4   5   6   7
     --- --- --- --- --- --- --- ---
 1  |         Add-CID octet         |  if for small CIDs and CID != 0
    +---+---+---+---+---+---+---+---+
 2  | 1   1   1   1   1   1   0 | D |
    +---+---+---+---+---+---+---+---+
    |                               |
 3  /    0-2 octets of CID info     /  1-2 octets if for large CIDs
    |                               |
    +---+---+---+---+---+---+---+---+
 4  |            Profile            |  1 octet
    +---+---+---+---+---+---+---+---+
 5  |              CRC              |  1 octet
    +---+---+---+---+---+---+---+---+
    |                               |
 6  |         Static chain          |  variable length
    |                               |
    +---+---+---+---+---+---+---+---+
    |                               |
 7  |         Dynamic chain         |  present if D = 1, variable length
    |                               |
    +---+---+---+---+---+---+---+---+
 8  |             SN                |  2 octets if not RTP
    +---+---+---+---+---+---+---+---+
    |                               |
 9  |           Payload             |  variable length
    |                               |
     - - - - - - - - - - - - - - - -

\endverbatim
 *
 * The function computes the length of the fields 2 + 4-8, ie. the first byte,
 * the Profile and CRC fields, the static and dynamic chains (outer and inner
 * IP headers + ESP header) and the SN.
 *
 * @param context         The decompression context
 * @param packet          The pointer on the IR packet minus the Add-CID byte
 *                        (ie. the field 2 in the figure)
 * @param plen            The length of the IR packet minus the Add-CID byte
 * @param large_cid_len   The size of the large CID field
 *                        (ie. field 3 in the figure)
 * @return                The length of the IR header,
 *                        0 if an error occurs
 */
unsigned int esp_detect_ir_size(struct d_context *context,
                                unsigned char *packet,
                                unsigned int plen,
                                unsigned int large_cid_len)
{
	unsigned int length, d;

	/* Profile and CRC fields + IP static & dynamic chains + SN */
	length = ip_detect_ir_size(context, packet, plen, large_cid_len);
	if(length == 0)
		goto quit;

	/* ESP static part (see 5.7.7.7 in RFC 3095) */ 
	length += esp_get_static_size();

	/* ESP dynamic part if included (see 5.7.7.7 in RFC 3095) */
	d = GET_BIT_0(packet);
	if(d)
		length += 4;

quit:
	return length;
}


/**
 * @brief Find the length of the IR-DYN header.
 * 
 * This function is one of the functions that must exist in one profile for the
 * framework to work.
 *
 * \verbatim

 Basic structure of the IR-DYN packet (5.7.7.2):

      0   1   2   3   4   5   6   7
     --- --- --- --- --- --- --- ---
 1  :         Add-CID octet         : if for small CIDs and CID != 0
    +---+---+---+---+---+---+---+---+
 2  | 1   1   1   1   1   0   0   0 | IR-DYN packet type
    +---+---+---+---+---+---+---+---+
    :                               :
 3  /     0-2 octets of CID info    / 1-2 octets if for large CIDs
    :                               :
    +---+---+---+---+---+---+---+---+
 4  |            Profile            | 1 octet
    +---+---+---+---+---+---+---+---+
 5  |              CRC              | 1 octet
    +---+---+---+---+---+---+---+---+
    |                               |
 6  /         Dynamic chain         / variable length
    |                               |
    +---+---+---+---+---+---+---+---+
 7  |             SN                | 2 octets if not RTP
    +---+---+---+---+---+---+---+---+
    :                               :
 8  /           Payload             / variable length
    :                               :
     - - - - - - - - - - - - - - - -

\endverbatim
 *
 * The function computes the length of the fields 2 + 4-8, ie. the first byte,
 * the Profile and CRC fields, the dynamic chains (outer and inner IP headers +
 * ESP header) and the SN.
 *
 * @param context         The decompression context
 * @param packet          The IR-DYN packet after the Add-CID byte if present
 *                        (ie. the field 2 in the figure)
 * @param plen            The length of the IR-DYN packet minus the Add-CID byte
 * @param large_cid_len   The size of the large CID field
 *                        (ie. field 3 in the figure)
 * @return                The length of the IR-DYN header,
 *                        0 if an error occurs
 */
unsigned int esp_detect_ir_dyn_size(struct d_context *context,
                                    unsigned char *packet,
                                    unsigned int plen,
                                    unsigned int large_cid_len)
{
	unsigned int length;

	/* Profile and CRC fields + IP dynamic chains */
	length = ip_detect_ir_dyn_size(context, packet, plen, large_cid_len);
	if(length == 0)
		goto quit;

	/* ESP dynamic part (see 5.7.7.7 in RFC 3095) */
	length += 4;

quit:
	return length;
}


/**
 * @brief Decode the ESP static part of the ROHC packet.
 *
 * @param context The generic decompression context
 * @param packet  The ROHC packet to decode
 * @param length  The length of the ROHC packet
 * @param dest    The decoded ESP header
 * @return        The number of bytes read in the ROHC packet,
 *                -1 in case of failure
 */
int esp_decode_static_esp(struct d_generic_context *context,
                          const unsigned char *packet,
                          unsigned int length,
                          unsigned char *dest)
{
	struct esphdr *esp = (struct esphdr *) dest;
	int read = 0; /* number of bytes read from the packet */

	/* check the minimal length to decode the ESP static part */
	if(length < 4)
	{
		rohc_debugf(0, "ROHC packet too small (len = %d)\n", length);
		goto error;
	}

	memcpy(&esp->security_parameters_index, packet, 4);
	packet += 4;
	rohc_debugf(3, "ESP security parameters index = 0x%08x\n", ntohl(esp->security_parameters_index));
	read += 4;

	return read;

error:
	return -1;
}


/**
 * @brief Decode the ESP dynamic part of the ROHC packet.
 *
 * @param context      The generic decompression context
 * @param packet       The ROHC packet to decode
 * @param length       The length of the ROHC packet
 * @param dest         The decoded ESP header
 * @return             The number of bytes read in the ROHC packet,
 *                     -1 in case of failure
 */
int esp_decode_dynamic_esp(struct d_generic_context *context,
                           const unsigned char *packet,
                           unsigned int length,
                           unsigned char *dest)
{
	struct d_esp_context *esp_context;
	struct esphdr *esp;
	int read = 0; /* number of bytes read from the packet */
	int ret;
	
	esp_context = context->specific;
	esp = (struct esphdr *) dest;

	/* check the minimal length to decode the ESP dynamic part */
	if(length < 4)
	{
		rohc_debugf(0, "ROHC packet too small (len = %d)\n", length);
		goto error;
	}

	/* retrieve the ESP sequence number from the ROHC packet */
	memcpy(&esp->sequence_number, packet, 4);
	packet += 4;		
	rohc_debugf(3, "ESP sequence number = 0x%08x\n", ntohl(esp->sequence_number));
	read += 4;

	/* init the ESP context */
	esp_context->esp_sequence_number = ntohl(esp->sequence_number);

	/* SN field */
	ret = ip_decode_dynamic_ip(context, packet, length - read, dest + read);
	if(ret == -1)
		goto error;
	packet += ret;
	read += ret;

	return read;

error:
	return -1;
}



/**
 * @brief Build an uncompressed ESP header.
 *
 * @param context      The generic decompression context
 * @param active       The ESP header changes
 * @param dest         The buffer to store the ESP header (MUST be at least
 *                     of sizeof(struct esphdr) length)
 * @param payload_size The length of the ESP payload
 * @return             The length of the next header (ie. the ESP header),
 *                     -1 in case of error
 */
int esp_build_uncompressed_esp(struct d_generic_context *context,
                               struct d_generic_changes *active,
                               unsigned char *dest,
                               int payload_size)
{
	struct d_esp_context *esp_context = context->specific;
	struct esphdr *esp_active = (struct esphdr *) active->next_header;
	struct esphdr *esp = (struct esphdr *) dest;

	rohc_debugf(3, "ESP active security_parameters_index 0x%08x sequence_number 0x%08x\n",
		    ntohl(esp_active->security_parameters_index), ntohl(esp_active->sequence_number));
	rohc_debugf(3, "ESP context sequence number = 0x%08x\n", esp_context->esp_sequence_number);

	switch( context->packet_type )
	{
		case PACKET_IR:
				rohc_debugf(3, "Packet type IR\n");
				esp->sequence_number = htonl(esp_context->esp_sequence_number);
				break;
		case PACKET_IR_DYN:
				rohc_debugf(3, "Packet type IR-DYN\n");
				esp->sequence_number = htonl(esp_context->esp_sequence_number);
				break;		
		default:
				rohc_debugf(3, "Packet type Ux\n");
				++esp_context->esp_sequence_number;
				esp->sequence_number = htonl(esp_context->esp_sequence_number);
				break;
	}

	esp->security_parameters_index = esp_active->security_parameters_index;

	rohc_debugf(3, "ESP security_parameters_index 0x%08x sequence_number 0x%08x\n",
		    ntohl(esp->security_parameters_index), ntohl(esp->sequence_number));

	return sizeof(struct esphdr);

//error:
	return -1;
}


/**
 * @brief Define the decompression part of the ESP profile as described
 *        in the RFC 3095.
 */
struct d_profile d_esp_profile =
{
	ROHC_PROFILE_ESP,       /* profile ID (see 8 in RFC 3095) */
	"ESP / Decompressor",   /* profile description */
	d_generic_decode,       /* profile handlers */
	d_esp_decode_ir,
	d_esp_create,
	d_esp_destroy,
	esp_detect_ir_size,
	esp_detect_ir_dyn_size,
	esp_get_static_size,
	d_generic_get_sn,
};

