//#include <inttypes.h>
#include <endian.h>
//#include <byteswap.h>

#include "flashlib.h"
#include "strataflash.h"

//#define WITH_RTEMS

#ifdef WITH_RTEMS
#include <rtems.h>
#endif /*WITH_RTEMS*/

/*******************************************************************/
#if FLASH_BYTE_ORDER == __LITTLE_ENDIAN

static inline
int flashlib_prepval_x16(uint16_t *pval, flash_addr_t addr, const void *data, long count)
{
  if(!((long)addr&1) && (count>=2)){
   #ifdef WITH_FLASH_NATIVE_ORDER
    *pval=*(uint16_t*)data;
   #else /*WITH_FLASH_NATIVE_ORDER*/
    *pval=(((uint8_t*)data)[0])|(((uint8_t*)data)[1]<<8);
   #endif /*WITH_FLASH_NATIVE_ORDER*/
    return 2;
  }
  if(!count) return count;
  if(!((long)addr&1)){
    *pval=((uint8_t*)data)[0] | (FLASH_RD8((uint8_t*)addr+1)<<8);
  }else{
    *pval=FLASH_RD8((uint8_t*)addr-1) | ((((uint8_t*)data)[0])<<8);
  }
  return 1;

  /*mem_op_error:
    return -4;*/
}

#ifdef WITH_TARGET_BUS32
static inline
int flashlib_prepval_x32(uint32_t *pval, flash_addr_t addr, const void *data, long count)
{
  int offs=(long)addr&3;
  int rest=4-offs;
  uint32_t val=0;
  if(!offs && (count>=4)){
   #ifdef WITH_FLASH_NATIVE_ORDER
    *pval=*(uint32_t*)data;
   #else /*WITH_FLASH_NATIVE_ORDER*/
    *pval=(((uint8_t*)data)[0])|(((uint8_t*)data)[1]<<8)|
          ((uint32_t)((uint8_t*)data)[2]<<16)|((uint32_t)((uint8_t*)data)[3]<<24);
   #endif /*WITH_FLASH_NATIVE_ORDER*/
    return 4;
  }
  if(!count) return count;
  if(count>rest) count=rest;

  val=FLASH_RD32((uint32_t*)addr-offs);
  
  offs*=8;
  rest=count;
  while(rest--){
    val<<=8;
    val&=~((uint32_t)0xff<<offs);
    val|=(*(uint8_t*)data++)<<offs;
    offs+=8;
  }
  *pval=val;
  return count;

  mem_op_error:
    return -4;
}

#endif /*WITH_TARGET_BUS32*/

/*******************************************************************/
#else /*FLASH_BYTE_ORDER == __BIG_ENDIAN*/

static inline
int flashlib_prepval_x16(uint16_t *pval, flash_addr_t addr, const void *data, long count)
{
  if(!((long)addr&1) && (count>=2)){
   #ifdef WITH_FLASH_NATIVE_ORDER
    *pval=*(uint16_t*)data;
   #else /*WITH_FLASH_NATIVE_ORDER*/
    *pval=(((uint8_t*)data)[0]<<8)|(((uint8_t*)data)[1]);
   #endif /*WITH_FLASH_NATIVE_ORDER*/
    return 2;
  }
  if(!count) return count;
  if(!((long)addr&1)){
    *pval=(((uint8_t*)data)[0]<<8) | FLASH_RD8((uint8_t*)addr+1);
  }else{
    *pval=(FLASH_RD8((uint8_t*)addr-1)<<8) | (((uint8_t*)data)[0]);
  }
  return 1;

  mem_op_error:
    return -4;
}

#ifdef WITH_TARGET_BUS32
static inline
int flashlib_prepval_x32(uint32_t *pval, flash_addr_t addr, const void *data, long count)
{
  int offs=(long)addr&3;
  int rest=4-offs;
  uint32_t val=0;
  if(!offs && (count>=4)){
   #ifdef WITH_FLASH_NATIVE_ORDER
    *pval=*(uint32_t*)data;
   #else /*WITH_FLASH_NATIVE_ORDER*/
    *pval=((uint32_t)((uint8_t*)data)[0]<<24)|((uint32_t)((uint8_t*)data)[1]<<16)|
          (((uint8_t*)data)[2]<<8)|(((uint8_t*)data)[3]);
   #endif /*WITH_FLASH_NATIVE_ORDER*/
    return 4;
  }
  if(!count) return count;
  if(count>rest) count=rest;

  val=FLASH_RD32((uint32_t*)addr-offs);
  
  offs=(3-offs)*8;
  rest=count;
  while(rest--){
    val<<=8;
    val&=~((uint32_t)0xff<<offs);
    val|=(*(uint8_t*)data++)<<offs;
    offs-=8;
  }
  *pval=val;
  return count;

  mem_op_error:
    return -4;
}

#endif /*WITH_TARGET_BUS32*/

#endif /*FLASH_BYTE_ORDER == __BIG_ENDIAN*/

/*******************************************************************/
/* routines for 16 bit wide StrataFlash memories */

int strataflash_check_id_x16(const flash_alg_info_t *alg, flash_addr_t addr, int retid[2])
{
  unsigned long devid, manid;
  int ret;

  ret = alg->getinfo(alg, addr, FLASH_INFO_MANUFACTURE_CODE, &manid);
  if(ret>=0)
    ret = alg->getinfo(alg, addr, FLASH_INFO_DEVICE_CODE, &devid);

  retid[0] = manid;
  retid[1] = devid;

  return ret;

}

int strataflash_prog_x16(const flash_alg_info_t *alg, flash_addr_t addr, const void *data, long count)
{
  int ret, n, tout;
  uint16_t sr,val,last;
  flash_addr_t reset_addr;
  /*flash_addr_t a=addr&~FLASH_ADDR_MASK;*/
  ret=flashlib_prepval_x16(&val,addr,data,count);
  if(ret<=0)
    return ret;
  if(((uint32_t)addr+count)&1){
    ret=flashlib_prepval_x16(&last,addr+count-1,(uint8_t*)data+count-1,1);
  }

  ret=count;

  /* allign address and count with FLASH words */
  if((uint32_t)addr&1) {
    count++;
    addr=addr-1;
    data=(uint8_t*)data-1;
  }
  reset_addr=addr;

  /* clear previous errors */
  FLASH_WR16(addr,FLASH_Clear_Status);

  while(count) {
    /* do not cross FLASH row boudary for block writes */
    n=0x20-((uint32_t)addr&0x1f);
    if(n>count) n=count;
    n=(n+1)/2;

    tout=200000;
    do {
      if(!tout--) { ret=-2; goto flash_error; }
      FLASH_WR16(addr,FLASH_Write_Buffer);
    } while((FLASH_RD16(addr)&FLASH_Status_Ready)!=FLASH_Status_Ready);

    /* write number of words to be written */
    FLASH_WR16(addr,n-1);
    for(;n;n--) {
      FLASH_WR16(addr,val);
      data=((uint8_t*)data+2);
      addr+=2;
      count-=2;

      if(count>=2){
       #if defined(WITH_FLASH_NATIVE_ORDER)
	val=*((uint16_t*)data);
       #elif FLASH_BYTE_ORDER == __LITTLE_ENDIAN
	val=(((uint8_t*)data)[0])|(((uint8_t*)data)[1]<<8);
       #else /* FLASH_BYTE_ORDER == __BIG_ENDIAN */
	val=(((uint8_t*)data)[0]<<8)|(((uint8_t*)data)[1]);
       #endif
      } else if(count) {
        count++;
	val=last;
      }
    }

    addr-=2;

    FLASH_WR16(addr,FLASH_Confirm);
    // FLASH_WR16(addr,FLASH_Read_Status);
   #ifdef WITH_RTEMS
    tout=50;
   #else /*WITH_RTEMS*/
    tout=200000;
   #endif /*WITH_RTEMS*/
    do {
      if(!tout--) { ret=-3; goto flash_error; }
      sr=FLASH_RD16(addr);
      if((sr&FLASH_Status_Ready)==FLASH_Status_Ready)
        break;
     #ifdef WITH_RTEMS
      rtems_task_wake_after(2);
     #endif /*WITH_RTEMS*/
    } while(1);

    if(sr&FLASH_ErrorMask) { ret=-4; goto flash_error; }

    addr+=2;

  }

  FLASH_WR16(reset_addr,FLASH_Reset);
  return ret;

  /*mem_op_error:
    ret=-5;*/
  flash_error:
    FLASH_WR16(reset_addr,FLASH_Reset);
    return ret;
}

int strataflash_erase_x16(const flash_alg_info_t *alg, flash_addr_t addr, long size)
{
  uint16_t sr;
  volatile int tout;
  int ret=0;
  flash_addr_t cur_sec_size=alg->sec_size;
  flash_addr_t change_sec_size_addr=0;

  if((alg->bb_sec_cnt > 0) && (addr >= alg->start) &&
     (addr-alg->start < alg->bb_sec_cnt*alg->bb_sec_size)) {
    cur_sec_size=alg->bb_sec_size;
    change_sec_size_addr=alg->start+alg->bb_sec_cnt*alg->bb_sec_size;
  } else if(alg->bb_sec_cnt < 0) {
    if((addr > alg->start+alg->size+alg->bb_sec_cnt*alg->bb_sec_size) &&
       (addr-alg->start < alg->size))
      cur_sec_size=alg->bb_sec_size;
    change_sec_size_addr=addr-alg->start+alg->size+alg->bb_sec_cnt*alg->bb_sec_size;
  }

  /*flash_addr_t a=(flash_addr_t)((uint32_t)addr&~FLASH_ADDR_MASK);*/

  if(size==-1) {
    size=1;
  }

  if(size==0) {
    size=1;
  }

  size+=addr&(cur_sec_size-1);
  addr&=~(cur_sec_size-1);

  FLASH_WR16(addr,FLASH_Clear_Status);

  while(size>0) {

    FLASH_WR16(addr,FLASH_Block_Erase);
    FLASH_WR16(addr,FLASH_Confirm);

    // FLASH_WR16(addr,FLASH_Read_Status);

    tout=2000000;
    do {
      if(!tout--) { ret=-3; goto flash_error; }
      sr=FLASH_RD16(addr);
      if((sr&FLASH_Status_Ready)==FLASH_Status_Ready)
        break;
     #ifdef WITH_RTEMS
      rtems_task_wake_after(2);
     #endif /*WITH_RTEMS*/
    } while(1);

    if(sr&FLASH_ErrorMask) { ret=-4; goto flash_error; }

    if(cur_sec_size<=0)
      break;
    if(size<=cur_sec_size)
      break;
    size-=cur_sec_size;
    addr+=cur_sec_size;

    if(addr==change_sec_size_addr) {
      if(alg->bb_sec_cnt > 0)
        cur_sec_size=alg->sec_size;
      else if(alg->bb_sec_cnt < 0)
        cur_sec_size=alg->bb_sec_size;
    }
  }

  FLASH_WR16(addr,FLASH_Reset);
  FLASH_WR16(addr,FLASH_Reset);
  return 0;

  /*mem_op_error:
    ret=-5;*/
  flash_error:
    FLASH_WR16(addr,FLASH_Reset);
    FLASH_WR16(addr,FLASH_Reset);
    return ret;
}

int strataflash_lockctrl_x16(const struct flash_alg_info *alg, flash_addr_t addr, long size, int lock)
{
  int ret=0;
  volatile int tout;
  uint16_t sr;
  flash_addr_t cur_sec_size=alg->sec_size;
  flash_addr_t change_sec_size_addr=0;

  if((alg->bb_sec_cnt > 0) && (addr >= alg->start) &&
     (addr-alg->start < alg->bb_sec_cnt*alg->bb_sec_size)) {
    cur_sec_size=alg->bb_sec_size;
    change_sec_size_addr=alg->start+alg->bb_sec_cnt*alg->bb_sec_size;
  } else if(alg->bb_sec_cnt < 0) {
    if((addr > alg->start+alg->size+alg->bb_sec_cnt*alg->bb_sec_size) &&
       (addr-alg->start < alg->size))
      cur_sec_size=alg->bb_sec_size;
    change_sec_size_addr=addr-alg->start+alg->size+alg->bb_sec_cnt*alg->bb_sec_size;
  }

  if(size==-1) {
    size=1;
  }

  if(size==0) {
    size=1;
  }

  size+=addr&(cur_sec_size-1);
  addr&=~(cur_sec_size-1);

  FLASH_WR16(addr,FLASH_Clear_Status);

  while(size>0) {

    if(lock) {
      FLASH_WR16(addr,FLASH_Set_Lock);
      FLASH_WR16(addr,FLASH_Set_Lock_Confirm);
    }else{
      FLASH_WR16(addr,FLASH_Clear_Lock);
      FLASH_WR16(addr,FLASH_Clear_Lock_Confirm);
    }

    // FLASH_WR16(addr,FLASH_Read_Status);

    tout=2000000;
    do {
      if(!tout--) { ret=-3; goto flash_error; }
      sr=FLASH_RD16(addr);
      if((sr&FLASH_Status_Ready)==FLASH_Status_Ready)
        break;
     #ifdef WITH_RTEMS
      rtems_task_wake_after(2);
     #endif /*WITH_RTEMS*/
    } while(1);

    if(sr&FLASH_ErrorMask) { ret=-4; goto flash_error; }

    if(cur_sec_size<=0)
      break;
    if(size<=cur_sec_size)
      break;
    size-=cur_sec_size;
    addr+=cur_sec_size;

    if(addr==change_sec_size_addr) {
      if(alg->bb_sec_cnt > 0)
        cur_sec_size=alg->sec_size;
      else if(alg->bb_sec_cnt < 0)
        cur_sec_size=alg->bb_sec_size;
    }

  }

  FLASH_WR16(addr,FLASH_Reset);
  FLASH_WR16(addr,FLASH_Reset);
  return 0;

  /*mem_op_error:
    ret=-5;*/
  flash_error:
    FLASH_WR16(addr,FLASH_Reset);
    FLASH_WR16(addr,FLASH_Reset);
    return ret;
}

static flash_addr_t strataflash_getinfo_table[] = {
  [FLASH_INFO_MANUFACTURE_CODE] = 0,
  [FLASH_INFO_DEVICE_CODE]      = 1,
  [FLASH_INFO_BLOCK_LOCK_STATE] = 2,
  [FLASH_INFO_CFI_SIZE_LOG2]    = 0x27,
  [FLASH_INFO_CFI_ERBLK_REGIONS]= 0x2C

};

static flash_addr_t strataflash_getinfo_mask[] = {
  [FLASH_INFO_MANUFACTURE_CODE] = 0xff,
  [FLASH_INFO_DEVICE_CODE]      = 0xff,
  [FLASH_INFO_BLOCK_LOCK_STATE] = 0x01,
  [FLASH_INFO_CFI_SIZE_LOG2]    = 0xff,
  [FLASH_INFO_CFI_ERBLK_REGIONS]= 0xff
};

int strataflash_getinfo_x16(const struct flash_alg_info *alg, flash_addr_t addr, int what, unsigned long *infobuf)
{
  unsigned long mask;

  if(addr == 0)
    addr = alg->start;

  if(alg->bb_sec_cnt&&alg->bb_sec_size)
    addr&=~(alg->bb_sec_size-1);
  else
    addr&=~(alg->sec_size-1);

  if(what>=sizeof(strataflash_getinfo_table)/sizeof(strataflash_getinfo_table[0]))
    return -1;

  addr+=2*strataflash_getinfo_table[what];
  mask=strataflash_getinfo_mask[what];

  if(!mask)
    return -1;

  FLASH_WR16(addr,FLASH_Read_ID);
  *infobuf=FLASH_RD16(addr) & mask;

  FLASH_WR16(addr,FLASH_Reset);
  FLASH_WR16(addr,FLASH_Reset);
  return 0;
}
