// matrix class that represents matrix in mathematics
// $Id: mathmatrix.cc,v 1.4 1999/02/23 03:48:18 hideto Exp $
//
// Copyright (C) 1999  Hideto Ishibashi
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.


#include "mathmatrix.hh"


//////////////////////////////////////////////////////////////////
// private functions


void
MathMatrix::initialize(const double& a_var)
{
	m_element[0][0] = double(0);
	
	for (unsigned int i = 1; i <= m_rsize; i++)
	{
		for (unsigned int j = 1; j <= m_csize; j++)
		{
			m_element[i][j] = a_var;
		}
	}
}


void
MathMatrix::initialize(const MathMatrix& a_var)
{
	m_element[0][0] = double(0);

	// copying
	for (unsigned int i = 1; i <= a_var.m_rsize; i++)
	{
		for (unsigned int j = 1; j <= a_var.m_csize; j++)
		{
			m_element[i][j] = a_var.m_element[i][j];
		}
	}

	// extra rows(lower left side) init
	if (m_rsize > a_var.m_rsize)
	{
		for (unsigned int j = 1; j <= a_var.m_csize; j++)
		{
			for (unsigned int i = a_var.m_rsize + 1; i <= m_rsize; i++)
			{
				m_element[i][j] = double(0);
			}
		}
	}
	
	// extra columns(upper right side) init
	if (m_csize > a_var.m_csize)
	{
		for (unsigned int i = 1; i <= a_var.m_rsize; i++)
		{
			for (unsigned int j = a_var.m_csize + 1; j <= m_csize; j++)
			{
				m_element[i][j] = double(0);
			}
		}
	}
	
	// an extra square part(lower right side) init
	for (unsigned int i = a_var.m_rsize + 1; i <= m_rsize; i++)
	{
		for (unsigned int j = a_var.m_csize + 1; j <= m_csize; j++)
		{
			m_element[i][j] = double(0);
		}
	}
}


void
MathMatrix::allocate(unsigned int a_rsize, unsigned int a_csize)
{
	m_rsize = a_rsize > 0 ? a_rsize : 0;
	m_csize = a_csize > 0 ? a_csize : 0;

	m_element = new (double*)[m_rsize+1];
	for (unsigned int i = 0; i <= m_rsize; i++)
	{
		m_element[i] = new double[m_csize+1];
	}
}


void
MathMatrix::destroy(void)
{
	m_rsize = m_csize = 0;

	for (unsigned int i = 0; i <= m_rsize; i++)
	{
		delete[] m_element[i];
	}
	delete m_element;
}


//////////////////////////////////////////////////////////////////
// constructor, destructor and assign operator


// default constructor

MathMatrix::MathMatrix()
{
	allocate(0, 0);
	initialize(double(0));
}


MathMatrix::MathMatrix(unsigned int a_rsize, unsigned int a_csize)
{
	allocate(a_rsize, a_csize);
	initialize(double(0));
}


// copy constructor

MathMatrix::MathMatrix(const MathMatrix& a_var)
{
	allocate(a_var.m_rsize, a_var.m_csize);
	initialize(a_var);
}


// constructor for type conversion

MathMatrix::MathMatrix(const MathVector& a_var, int direction)
{
	if (direction == 0)  // row vector
	{
		allocate(a_var.size(), 1);
		
		for (unsigned int i = 1; i <= m_rsize; i++) // elements init
		{
			m_element[i][1] = a_var(i);
		}
		
	}
	else  // column vector
	{
		allocate(1, a_var.size());
		
		for (unsigned int i = 1; i <= m_csize; i++) // elements init
		{
			m_element[1][i] = a_var(i);
		}
	}
}


// assign operator

MathMatrix&
MathMatrix::operator =(const MathMatrix& a_var)
{
	if (this == &a_var)	return *this;  // check assignment to itself

	destroy();
	allocate(a_var.m_rsize, a_var.m_csize);
	initialize(a_var);

	return *this;
}


// destructor

MathMatrix::~MathMatrix()
{
	destroy();
}


//////////////////////////////////////////////////////////////////
// member operations


double&
MathMatrix::operator ()(unsigned int a_i, unsigned int a_j)
{
	if (a_i < 1 || a_i > m_rsize ||
		a_j < 1 || a_j > m_csize)
	{
		return m_element[0][0];
	}
	
	return m_element[a_i][a_j];
}


const double&
MathMatrix::operator ()(unsigned int a_i, unsigned int a_j) const
{
	if (a_i < 1 || a_i > m_rsize ||
		a_j < 1 || a_j > m_csize)
	{
		return m_element[0][0];
	}
	
	return m_element[a_i][a_j];
}


MathMatrix
MathMatrix::cut(unsigned int a_rfrom, unsigned int a_cfrom,
				unsigned int a_rto, unsigned int a_cto) const
{
	if (a_rfrom < 1 || a_rto > m_rsize ||
		a_cfrom < 1 || a_cto > m_csize ||
		a_rfrom >= a_rto || a_cfrom >= a_cto)
	{
		return *this;
	}
	
	unsigned int ret_rsize = a_rto - a_rfrom + 1;
	unsigned int ret_csize = a_cto - a_cfrom + 1;
	MathMatrix ret_matrix(ret_rsize, ret_csize);
	
	for (unsigned int i = 1; i <= ret_matrix.m_rsize; i++)
	{
		for (unsigned int j = 1; j <= ret_matrix.m_csize; j++)
		{
			ret_matrix.m_element[i][j] = m_element[a_rfrom+i-1][a_cfrom+j-1];
		}
	}

	return ret_matrix;
}


//////////////////////////////////////////////////////////////////
// binary operations


MathMatrix
matrix_add(const MathMatrix& a_var1, const MathMatrix& a_var2)
{
	MathMatrix ret_matrix(a_var1.m_rsize, a_var1.m_csize);
	
	if (a_var1.m_rsize != a_var2.m_rsize ||
		a_var1.m_csize != a_var2.m_csize)
	{
		return ret_matrix;
	}
	
	for (unsigned int i = 1; i <= ret_matrix.m_rsize; i++)
	{
		for (unsigned int j = 1; j <= ret_matrix.m_csize; j++)
		{
			ret_matrix.m_element[i][j] =
				a_var1.m_element[i][j] + a_var2.m_element[i][j];
		}
	}

	return ret_matrix;
}


MathMatrix
matrix_sub(const MathMatrix& a_var1, const MathMatrix& a_var2)
{
	MathMatrix ret_matrix(a_var1.m_rsize, a_var1.m_csize);
	
	if (a_var1.m_rsize != a_var2.m_rsize ||
		a_var1.m_csize != a_var2.m_csize)
	{
		return ret_matrix;
	}
	
	for (unsigned int i = 1; i <= ret_matrix.m_rsize; i++)
	{
		for (unsigned int j = 1; j <= ret_matrix.m_csize; j++)
		{
			ret_matrix.m_element[i][j] =
				a_var1.m_element[i][j] - a_var2.m_element[i][j];
		}
	}

	return ret_matrix;
}


MathMatrix
matrix_mul(const MathMatrix& a_var1, const double& a_var2)
{
	MathMatrix ret_matrix(a_var1.m_rsize, a_var1.m_csize);
	
	for (unsigned int i = 1; i <= ret_matrix.m_rsize; i++)
	{
		for (unsigned int j = 1; j <= ret_matrix.m_csize; j++)
		{
			ret_matrix.m_element[i][j] *= a_var2;
		}
	}

	return ret_matrix;
}


MathMatrix
matrix_div(const MathMatrix& a_var1, const double& a_var2)
{
	MathMatrix ret_matrix(a_var1.m_rsize, a_var1.m_csize);
	
	for (unsigned int i = 1; i <= ret_matrix.m_rsize; i++)
	{
		for (unsigned int j = 1; j <= ret_matrix.m_csize; j++)
		{
			ret_matrix.m_element[i][j] /= a_var2;
		}
	}

	return ret_matrix;
}


MathMatrix
matrix_inp(const MathMatrix& a_var1, const MathMatrix& a_var2)
{
	MathMatrix ret_matrix(a_var1.m_rsize, a_var2.m_csize);
	
	if (a_var1.m_csize != a_var2.m_rsize)
	{
		return ret_matrix;
	}
	
	for (unsigned int i = 1; i <= ret_matrix.m_rsize; i++)
	{
		for (unsigned int j = 1; j <= ret_matrix.m_csize; j++)
		{
			double sum = 0;
			for (unsigned int k = 1; k <= a_var1.m_csize; k++)
			{
				sum += a_var1.m_element[i][k] * a_var2.m_element[k][j];
			}
			ret_matrix.m_element[i][j] = sum;
		}
	}

	return ret_matrix;
}


//////////////////////////////////////////////////////////////////
// unary operators


MathMatrix
MathMatrix::operator +=(const MathMatrix& a_var)
{
	if (m_rsize != a_var.m_rsize || m_csize != a_var.m_csize)
	{
		return *this;
	}
	
	for (unsigned int i = 1; i <= m_rsize; i++)
	{
		for (unsigned int j = 1; j <= m_csize; j++)
		{
			m_element[i][j] += a_var.m_element[i][j];
		}
	}

	return *this;
}


MathMatrix
MathMatrix::operator -=(const MathMatrix& a_var)
{
	if (m_rsize != a_var.m_rsize || m_csize != a_var.m_csize)
	{
		return *this;
	}
	
	for (unsigned int i = 1; i <= m_rsize; i++)
	{
		for (unsigned int j = 1; j <= m_csize; j++)
		{
			m_element[i][j] -= a_var.m_element[i][j];
		}
	}

	return *this;
}


MathMatrix
MathMatrix::operator *=(const double& a_var)
{
	for (unsigned int i = 1; i <= m_rsize; i++)
	{
		for (unsigned int j = 1; j <= m_csize; j++)
		{
			m_element[i][j] *= a_var;
		}
	}

	return *this;
}


MathMatrix
MathMatrix::operator /=(const double& a_var)
{
	for (unsigned int i = 1; i <= m_rsize; i++)
	{
		for (unsigned int j = 1; j <= m_csize; j++)
		{
			m_element[i][j] /= a_var;
		}
	}

	return *this;
}


MathMatrix
MathMatrix::t() const
{
	MathMatrix ret_matrix(m_csize, m_rsize);

	for (unsigned int i = 1; i <= m_rsize; i++)
	{
		for (unsigned int j = 1; j <= m_csize; j++)
		{
			ret_matrix.m_element[j][i] = m_element[i][j];
		}
	}

	return ret_matrix;
}


//////////////////////////////////////////////////////////////////
// type conversion


MathVector
MathMatrix::asVector() const
{
	if (m_rsize != 1 && m_csize != 1)
	{
		return MathVector(1);
	}

	if (m_rsize == 1)
	{
		MathVector ret_vector(m_csize);
		
		for (unsigned int i = 1; i <= m_csize; i++)
		{
			ret_vector(i) = m_element[1][i];
		}

		return ret_vector;
	}
	else
	{
		MathVector ret_vector(m_rsize);

		for (unsigned int i = 1; i <= m_rsize; i++)
		{
			ret_vector(i) = m_element[i][1];
		}
		
		return ret_vector;
	}
}
