// matrix class that represents matrix in mathematics
// $Id: mathmatrix.hh,v 1.10 1999/02/20 06:18:20 hideto Exp $
//
// Copyright (C) 1999  Hideto Ishibashi
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.


#ifndef _MATHMATRIX_HH_
#define _MATHMATRIX_HH_

#include "mathvector.hh"


class MathMatrix
{
public:
	MathMatrix();
	MathMatrix(unsigned int rows, unsigned int columns);
	MathMatrix(const MathMatrix&);
	MathMatrix(const MathVector&, int direction = 0);
		// the 2nd parameter --> 0:row vector(default), !0:column vector
	MathMatrix& operator =(const MathMatrix&);
	virtual ~MathMatrix();

	// member operations
	virtual double& operator ()(unsigned int row, unsigned int column);
		// Index is from 1 to n(m_v).
	virtual const double& operator ()
		(unsigned int row, unsigned int column) const;
	virtual unsigned int rsize() const { return m_rsize; }
	virtual unsigned int csize() const { return m_csize; }
	virtual MathMatrix cut(unsigned int row_from, unsigned int column_from,
						   unsigned int row_to, unsigned int column_to) const;

	// binary operations
	friend MathMatrix matrix_add(const MathMatrix&, const MathMatrix&);
	friend MathMatrix matrix_sub(const MathMatrix&, const MathMatrix&);
	friend MathMatrix matrix_mul(const MathMatrix&, const double&);
	friend MathMatrix matrix_div(const MathMatrix&, const double&);
	friend MathMatrix matrix_inp(const MathMatrix&, const MathMatrix&);

	// unary operators
	virtual MathMatrix operator -() const;
	virtual MathMatrix operator +=(const MathMatrix&);
	virtual MathMatrix operator -=(const MathMatrix&);
	virtual MathMatrix operator *=(const double&);
	virtual MathMatrix operator /=(const double&);

	virtual MathMatrix t() const;  // transpose

	// conversion to vector
	virtual MathVector asVector() const;

private:
	unsigned int	m_rsize, m_csize;
	double			**m_element;

private:
	virtual void initialize(const double&);
	virtual void initialize(const MathMatrix&);
	virtual void allocate(unsigned int rows, unsigned int columns);
	virtual void destroy();
};


//////////////////////////////////////////////////////////////////
// binary operators


MathMatrix operator +(const MathMatrix&, const MathMatrix&);
MathMatrix operator -(const MathMatrix&, const MathMatrix&);
MathMatrix operator *(const MathMatrix&, const double&);
MathMatrix operator *(const double&, const MathMatrix&);
MathMatrix operator /(const MathMatrix&, const double&);
MathMatrix operator *(const MathMatrix&, const MathMatrix&);


//////////////////////////////////////////////////////////////////
// binary operators


inline
MathMatrix
operator +(const MathMatrix& a_var1, const MathMatrix& a_var2)
{
	return matrix_add(a_var1, a_var2);
}

inline
MathMatrix
operator -(const MathMatrix& a_var1, const MathMatrix& a_var2)
{
	return matrix_sub(a_var1, a_var2);
}

inline
MathMatrix
operator *(const MathMatrix& a_var1, const double& a_var2)
{
	return matrix_mul(a_var1, a_var2);
}

inline
MathMatrix
operator *(const double& a_var1, const MathMatrix& a_var2)
{
	return matrix_mul(a_var2, a_var1);
}

inline
MathMatrix
operator /(const MathMatrix& a_var1, const double& a_var2)
{
	return matrix_div(a_var1, a_var2);
}

inline
MathMatrix
operator *(const MathMatrix& a_var1, const MathMatrix& a_var2)
{
	return matrix_inp(a_var1, a_var2);
}


//////////////////////////////////////////////////////////////////
// unary operators


inline
MathMatrix
MathMatrix::operator -() const
{
	return matrix_mul(*this, double(-1));
}


#endif // _MATHMATRIX_HH_
