// vector class that represents vector in mathematics
// $Id: mathvector.cc,v 1.3 1999/02/22 07:06:01 hideto Exp $
//
// Copyright (C) 1999  Hideto Ishibashi
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.


#include "mathvector.hh"


//////////////////////////////////////////////////////////////////
// private functions


void
MathVector::initialize(const double& a_var)
{
	m_element[0] = double(0);

	for (unsigned int i = 1; i <= m_size; i++)
	{
		m_element[i] = a_var;
	}
}


void
MathVector::initialize(const MathVector& a_var)
{
	m_element[0] = double(0);

	for (unsigned int i = 1; i <= a_var.m_size; i++)
	{
		m_element[i] = a_var.m_element[i];
	}
	
	for (unsigned int i = a_var.m_size + 1; i <= m_size; i++)
	{
		m_element[i] = double(0);
	}
}


void
MathVector::allocate(unsigned int a_size)
{
	m_size = a_size > 0 ? a_size : 0;
	m_element = new double[m_size+1];
}


void
MathVector::destroy(void)
{
	m_size = 0;
	delete [] m_element;
}


//////////////////////////////////////////////////////////////////
// constructor, destructor and assign operator


// default constructor

MathVector::MathVector()
{
	allocate(0);
	initialize(double(0));
}


MathVector::MathVector(unsigned int a_size)
{
	allocate(a_size);
	initialize(double(0));
}


// copy constructor

MathVector::MathVector(const MathVector& a_var)
{
	allocate(a_var.m_size);
	initialize(a_var);
}


// assign operator

MathVector&
MathVector::operator =(const MathVector& a_var)
{
	if (this == &a_var)	return *this;  // check assignment to itself

	destroy();
	allocate(a_var.m_size);
	initialize(a_var);

	return *this;
}


// destructor

MathVector::~MathVector()
{
	destroy();
}


//////////////////////////////////////////////////////////////////
// member operations


double&
MathVector::operator ()(unsigned int a_i)
{
	if (a_i < 1 || a_i > m_size)
	{
		return m_element[0];
	}
	
	return m_element[a_i];
}


const double&
MathVector::operator ()(unsigned int a_i) const
{
	if (a_i < 1 || a_i > m_size)
	{
		return m_element[0];
	}
	
	return m_element[a_i];
}


MathVector
MathVector::cut(unsigned int a_from, unsigned int a_to) const
{
	if (a_from < 1 || a_from > m_size ||
		a_to < 1 || a_to > m_size)
	{
		return *this;
	}
	
	unsigned int ret_size = (a_to >= a_from ? 1: -1) * (a_to - a_from) + 1;
							// always plus
	MathVector ret_vector(ret_size);
	
	if (a_to >= a_from)
	{
		for (unsigned int i = 1; i <= ret_vector.m_size; i++)
		{
			ret_vector.m_element[i] = m_element[a_from+i-1];
		}
	}
	else
	{
		for (unsigned int i = 1; i <= ret_vector.m_size; i++)
		{
			ret_vector.m_element[i] = m_element[a_from-i+1];
		}
	}

	return ret_vector;
}


MathVector
MathVector::cat(const MathVector& a_var) const
{
	MathVector ret_vector(m_size + a_var.m_size);
	
	for (unsigned int i = 1; i <= m_size; i++)
	{
		ret_vector.m_element[i] = m_element[i];
	}

	for (unsigned int i = 1; i <= a_var.m_size; i++)
	{
		ret_vector.m_element[m_size+i] = a_var.m_element[i];
	}

	return ret_vector;
}


//////////////////////////////////////////////////////////////////
// binary operations


MathVector
vector_add(const MathVector& a_var1, const MathVector& a_var2)
{
	MathVector ret_vector(a_var1.m_size);
	
	if (a_var1.m_size != a_var2.m_size)
	{
		return ret_vector;
	}
	
	for (unsigned int i = 1; i <= ret_vector.m_size; i++)
	{
		ret_vector.m_element[i] = a_var1.m_element[i] + a_var2.m_element[i];
	}

	return ret_vector;
}


MathVector
vector_sub(const MathVector& a_var1, const MathVector& a_var2)
{
	MathVector ret_vector(a_var1.m_size);
	
	if (a_var1.m_size != a_var2.m_size)
	{
		return ret_vector;
	}
	
	for (unsigned int i = 1; i <= ret_vector.m_size; i++)
	{
		ret_vector.m_element[i] = a_var1.m_element[i] - a_var2.m_element[i];
	}

	return ret_vector;
}


MathVector
vector_mul(const MathVector& a_var1, const double& a_var2)
{
	MathVector ret_vector(a_var1);
	
	for (unsigned int i = 1; i <= ret_vector.m_size; i++)
	{
		ret_vector.m_element[i] *= a_var2;
	}

	return ret_vector;
}


MathVector
vector_div(const MathVector& a_var1, const double& a_var2)
{
	MathVector ret_vector(a_var1);
	
	for (unsigned int i = 1; i <= ret_vector.m_size; i++)
	{
		ret_vector.m_element[i] /= a_var2;
	}

	return ret_vector;
}


double
vector_inp(const MathVector& a_var1, const MathVector& a_var2)
{
	double ret_scalar = double(0);
	
	if (a_var1.m_size != a_var2.m_size)
	{
		return ret_scalar;
	}
	
	for (unsigned int i = 1; i <= a_var1.m_size; i++)
	{
		ret_scalar += a_var1.m_element[i] * a_var2.m_element[i];
	}

	return ret_scalar;
}


//////////////////////////////////////////////////////////////////
// unary operators


MathVector
MathVector::operator +=(const MathVector& a_var)
{
	if (m_size != a_var.m_size)
	{
		return *this;
	}
	
	for (unsigned int i = 1; i <= m_size; i++)
	{
		m_element[i] += a_var.m_element[i];
	}

	return *this;
}


MathVector
MathVector::operator -=(const MathVector& a_var)
{
	if (m_size != a_var.m_size)
	{
		return *this;
	}
	
	for (unsigned int i = 1; i <= m_size; i++)
	{
		m_element[i] -= a_var.m_element[i];
	}

	return *this;
}


MathVector
MathVector::operator *=(const double& a_var)
{
	for (unsigned int i = 1; i <= m_size; i++)
	{
		m_element[i] *= a_var;
	}

	return *this;
}


MathVector
MathVector::operator /=(const double& a_var)
{
	for (unsigned int i = 1; i <= m_size; i++)
	{
		m_element[i] /= a_var;
	}

	return *this;
}
