////////////////////////////////////////////////////////////////////////////////
//
//  Copyright  2000 Zentropic Computing
//
//
//  Authors:		Stuart Hughes
//  Contact:		info@zentropix.com
//  Original date:	Mar 2000
//  Ident:		@(#)$Id$
//  Description:	This code measures and displays context switch times
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

// 
////////////////////////////////////////////////////////////////////////////////
static char id_context_c[] __attribute__ ((unused)) = "@(#)$Id$";

///////////////////////////////////////////////////////////////////////////////
//
// common user/kernel 
//
///////////////////////////////////////////////////////////////////////////////
#define NSECS_PER_SEC	1000000000
#define FREQ		2			// basic frequency in Hz
#define BASE_PER	(NSECS_PER_SEC/FREQ)	// basic periodic in seconds
#define NTASKS		2			// don't change unless you 
						// kill task 2 wakeup call
typedef void *(* VP_FP)();
typedef void  (* V_FP_V  )(void);
typedef void  (* V_FP_I  )(int);


///////////////////////////////////////////////////////////////////////////////
//
// kernel module
//
///////////////////////////////////////////////////////////////////////////////
#if __KERNEL__

#include <linux/module.h>
#include <linux/kernel.h>
#include <asm/io.h>

#if RTAI
#    include <rtai.h>
#    include <rtai_sched.h>
#    include <rtai_fifos.h>
#    define TASK_STR		  RT_TASK
#    define rtf_create(num, size) rtf_create_using_bh(num, size, 0)
#    define get_time_ns		  rt_get_cpu_time_ns
#    define rtl_schedule()
#    define rtl_task_make_periodic(task_ptr, when_ns, period_ns)	\
	rt_task_make_periodic(task_ptr, 				\
				nano2count(when_ns),			\
				nano2count(period_ns))			
#else
#    include <rtl.h>
#    include <rtl_fifo.h>
#    include <rtl_sched.h>
#    include <pthread.h>
#    define TASK_STR		  pthread_t
#    define get_time_ns		  gethrtime
#    define rt_task_wait_period   pthread_wait_np
#    define rt_task_suspend(x)	  pthread_suspend_np(*x)
#    define rt_task_resume(x)     pthread_wakeup_np(*x)
#    define rt_get_time_ns        get_time_ns
#    define rtl_task_make_periodic(task_ptr, when_ns, period_ns)	\
	pthread_make_periodic_np(*task_ptr, when_ns, period_ns)
#endif

// task control data
#define MASTER 0
#define SLAVE  1
void *master();
void *slave();
struct t_dat_ {
    TASK_STR task;
    VP_FP func;
    int arg;
    int stack_size;
    int priority;
    int period;
    int    uses_fpu;
    V_FP_V sig_han;
    long long period_ns;
    long long when_ns;

} td[NTASKS] = {
	    // func    arg   stack prior period uses_fpu sig_han
       {  {0}, master,  0,  3000,    10,      1,     0,     0  },
       {  {0}, slave,   1,  3000,    5,       0,       0,     0  },
};

struct data_ {
	int seqn;
	long long t_master;
	long long t_slave;
} d = { 0,0,0 };


// function prototypes
int rt_task_create( struct t_dat_ *t);

////////////////////////////////////////////////////////////////
// RT module initialisation
///////////////////////////////////////////////////////////////
int init_module(void)
{                   
    int i;

    for(i = 0; i < NTASKS; i++) {
	rt_task_create( &td[i] );
    }
    return 0;
}
////////////////////////////////////////////////////////////////
// task creation wrapper to keep both parties happy (or not)
///////////////////////////////////////////////////////////////
int rt_task_create( struct t_dat_ *t)
{
#if RTAI
    static int timer_started = 0;

    if( timer_started == 0) {
	// start timer, this defaults to periodic mode
        start_rt_timer(nano2count(BASE_PER));
	timer_started 	= 1;
    }

    rt_task_init(	&t->task,
                   	(V_FP_I)t->func,
                        t->arg,
                        t->stack_size,
                        t->priority,
                        t->uses_fpu,
                        t->sig_han
		);

    if(t->period == 0) {
	return 0;
    }
    t->when_ns 		= rt_get_time_ns() + 1 * NSECS_PER_SEC;
    t->period_ns	= (long long)t->period * BASE_PER;

    rtl_task_make_periodic(&t->task, t->when_ns, t->period_ns);


#else					// RTL 
    struct sched_param p;
    pthread_create(	&t->task,
			NULL,
			t->func,
			(void *)t->arg
		);
    p.sched_priority = 100 - t->priority;
    pthread_setschedparam(t->task, SCHED_FIFO, &p);

    if(t->period == 0) {
	return 0;
    }
    t->when_ns 		= rt_get_time_ns() + 1 * NSECS_PER_SEC;
    t->period_ns	= (long long)t->period * BASE_PER;
    rtl_task_make_periodic( &t->task, t->when_ns, t->period_ns);

#endif

    return 0;
}

////////////////////////////////////////////////////////////////
// RT threads
///////////////////////////////////////////////////////////////
void *master(void *arg)
{
    long long seqn;
    struct t_dat_ *t	= &td[(int)arg];
    rt_task_wait_period();                                        

    while (1) {  
	// let linux breath, we need 2 cycles as we may be immediately ready
	// to run once the high prioriy task resumes
        t->when_ns 	= rt_get_time_ns();
        t->period_ns	= (long long)t->period * BASE_PER;
        rtl_task_make_periodic( &t->task, t->when_ns, t->period_ns);
        rt_task_wait_period();
        rt_task_wait_period();	// for RTL you need 2 otherwise you don't
				// ever get any suspend time

	// wakeup slave rt task, this will block us as we are
	// a lower priority
	seqn = d.seqn;
	d.t_master = get_time_ns();
	rt_task_resume(&td[SLAVE].task);
	if(seqn && seqn == d.seqn) {
	    printk("context switch time = %d ns\n", 
					(int)(d.t_slave - d.t_master) );
	}
	d.seqn++;
    }
}                                        
void *slave(void *arg)
{
    while (1) {  
	rt_task_suspend(&td[SLAVE].task);
	d.t_slave = get_time_ns();
	rtl_schedule();
    }
}                                        

////////////////////////////////////////////////////////////////
// RT module cleanup
///////////////////////////////////////////////////////////////
void cleanup_module(void)
{
    int i;
#if RTAI
    stop_rt_timer();
    rt_busy_sleep(1e7);
#endif
    for(i = 0; i < NTASKS; i++) {
#if RTAI
	rt_task_suspend(&td[i].task);
	rt_task_delete(&td[i].task);
#else
	pthread_delete_np(td[i].task);
#endif
    }
}

///////////////////////////////////////////////////////////////////////////////
//
// user code
//
///////////////////////////////////////////////////////////////////////////////
#else

#endif


