/* -*- C -*- ***************************************************
 * $Id: $
 *
 * DESCRIPTION: test handling with shared memory
 *
 *
 *
 * Copyright (C) 2000 O.Petzold <opetzold@wit.regiocom.net>
 */


/***
 * standard linux modules includes
 */
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/version.h>
#include <linux/errno.h>
#include <linux/string.h>	// memcpy etc.
 
#include <rtl.h>
#include <rtl_fifo.h>
#include <rtl_sched.h>
#include <rtl_sync.h>
#include <rtl_time.h>
#include <rtl_debug.h>
 
#include <stdbool.h>
#include <asm/errno.h>
#include <mbuff.h>

#include "debug.h"

#define NI_PCI_MIO_XE10_RES	65535
#define SHM_NAME_SLUT		"SensorLUT"
#define THREAD_FIFO1		0
#define THREAD_FIFO2		1
typedef unsigned int flag_t;

typedef struct {
  flag_t inuse;
  flag_t valid;
  double um[NI_PCI_MIO_XE10_RES+1];
  
} shm_slut_t;

dbg_level_t rtreg_debug_level = 4;
pthread_t rt_thread;
static volatile shm_slut_t *shm_slut = NULL;
static bool shm_isopen = false;

int shm_lock()
{
  register int ret = 0;

  // paranoia check
  if(shm_isopen == true) {
    RTREG_WARN("Shared Memory Regions allready locked !, ignored\n");
    return -EBUSY;
  }
  
  if((ret = shm_allocate(SHM_NAME_SLUT, sizeof(shm_slut_t), (void **)&shm_slut)) < 0) {
    RTREG_ERR("Could not open Shared Memory for Sensor LUT (error %u)\n", (unsigned int)ret);
    return ret;
  }
  
  /* all allocations where sucessfully */
  shm_isopen = true;

  return 0;
}

void shm_release()
{
  register int ret;

  // paranoia check
  if(shm_isopen == false) {
      RTREG_WARN("Shared Memory was NOT open, can't release it!, ignored\n");
      return;
  }
  
  if((ret = shm_deallocate((void *)shm_slut) != 0)) {
      RTREG_ERR("Could not close Shared Memory for Sensor LUT (error %d)\n", ret);
  }

  shm_isopen = false;
  shm_slut = NULL;
}

inline int shm_get_sensorLUT(shm_slut_t *slut_dest)
{
  unsigned long flags;

  if(!shm_slut->inuse) {
    RTREG_DEBUG(4, "Get Sensor LUT from Shm\n");

    shm_slut->inuse = true;

    #warning memcpy enabled
    my_memcpy((void *)slut_dest, (void *)shm_slut, sizeof(shm_slut_t));

    shm_slut->inuse = false;

    return 0;
  } else {
    RTREG_DEBUG(4, "Sensor LUT is in use, ignored !\n");

    return -EBUSY;
  }
}


/* 
 * gcc -Wall -O6 -DMODULE -D__KERNEL__ -D__RT__ -D__RTL__ -I ../include/ -I /usr/include/rtlinux/ -I /usr/include -c shm_test.c
 */
void *thread_code(void *param)
{
  register int r;
  static char buf1[64];
  static char buf2[64];
  shm_slut_t slut;

  while(1) {
      pthread_wait_np();

      for(r=0; r < NI_PCI_MIO_XE10_RES + 1; r++)  {
	slut.um[r] = shm_slut->um[r];
	  if((r+1) % 4096 == 0 || r == 0) {
	    /* print the shm copy to /dev/rtf2 */
	    sprintf(buf1, "<%5d\t%5d>\n", 
		    r, (int)shm_slut->um[r]);
	    rtf_put(THREAD_FIFO1, buf1, sizeof(buf1));
	  }
      }

      (void)shm_get_sensorLUT(&slut);

      for(r=0; r < NI_PCI_MIO_XE10_RES + 1; r++)  {
	  if((r+1) % 4096 == 0 || r == 0) {
	      /* print out local copy to /dev/rtf2 */
	      sprintf(buf2, "<%5d\t%5d>\n", r, (int)slut.um[r]);
	      rtf_put(THREAD_FIFO2, buf2, sizeof(buf2));
	  }
      }
  }
    
  return 0;
}

int init_module()
{
  pthread_attr_t attr;
  struct sched_param sched_param;
  hrtime_t now; 
  hrtime_t period_ns = 10000000;

  RTREG_INFO("Loading Shm Testdriver ...\n");
  shm_isopen = false;
  shm_slut = NULL;

  if(shm_lock() != 0) {
    RTREG_ERR("Unable to allocate shared memory\n");
    return -EIO;
  }

  rtf_destroy(THREAD_FIFO1);
  if(rtf_create(THREAD_FIFO1, 12000) != 0) {
      RTREG_ERR("unable to create rtf\n");
      return -EIO;
  }
  
  rtf_destroy(THREAD_FIFO2);
  if(rtf_create(THREAD_FIFO2, 12000) != 0) {
      RTREG_ERR("unable to create rtf\n");
      return -EIO;
  }
  
  pthread_attr_init(&attr);
  sched_param.sched_priority = 1;
  pthread_attr_setschedparam(&attr, &sched_param);
  now = gethrtime();
  
  if(pthread_create(&rt_thread, &attr, thread_code, (void *)0) != 0) {
      RTREG_ERR("unable to create thread\n");
      return -EIO;
  }

  if(pthread_make_periodic_np(rt_thread, now + 1000, period_ns) != 0) {
      RTREG_ERR("unable to make thread periodic\n");
      return -EIO;
  }

  RTREG_INFO("Shm Testdriver loaded.\n");

  return 0;
}


void cleanup_module()
{
  RTREG_INFO("Removing Shm Testdriver ...\n");

  pthread_delete_np(rt_thread);
  shm_release();
  rtf_destroy(THREAD_FIFO1);
  rtf_destroy(THREAD_FIFO2);

  RTREG_INFO("Shm Testdriver removed.\n");
}
