////////////////////////////////////////////////////////////////////////////////
//
// Copyright  1999 Zentropic Computing LLC 
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
// ZENTROPIC COMPUTING LLC BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//
// Except as contained in this notice, the name of the Zentropic Computing LLC
// shall not be used in advertising or otherwise to promote the sale, use or
// other dealings in this Software without prior written authorization from the
// Zentropic Computing LLC
//
// Authors:		Stuart Hughes (sehughes@zentropix.com)
// Origin:		Some parts derived from material originally
//			published by: 
//				Paolo Mantegazza (mantegazza@aero.polimi.it)
//				Michael Barabanov (baraban@fsmlabs.com)
//				Linux Device Drivers (Alessandro Rubini)
//
// Original date:	Dec 17 1999
// Id:			@(#)$Id: jitter.c,v 1.1 1999/12/20 16:37:32 seh Exp $
//
// Description:		This module attempts make an assesment of jitter in
//			an rt-module. To run this type ./run_test.
//			At any time the max/min values may be read by
//			typing cat /proc/jitter.
//
// Note:		The values found depend on system, environment and
//			loading.  To achieve high loading I usually use
//			top with a the run wait time set to zero, with a
//			ping -f directed at the machine under test to 
//			load the kernel.
//
//			to get a continuous display at the terminal:
//				while true; do cat /proc/jitter; done
//
//			This test is for the 2.2.x kernels only
//
////////////////////////////////////////////////////////////////////////////////

#define NSECS_PER_SEC   1000000000
#define FREQ            10000                   // Basic frequency in Hz 
#define BASE_PERIOD     (NSECS_PER_SEC/FREQ)    // Basic period in seconds 
#define NTASKS		1

typedef void *(* VP_FP)(void *);
typedef void  (* V_FP_V  )(void);
typedef void  (* V_FP_I  )(int);

#include <linux/module.h>
#include <linux/kernel.h>
#include <asm/io.h>
#include <linux/proc_fs.h>

#ifndef __RTL__
#include <rtai.h>
#include <rtai_sched.h>
#define TASK_STR		RT_TASK
#define get_time_ns		rt_get_cpu_time_ns
#else
#include <rtl.h>
#include <rtl_sched.h>
#include <posix/pthread.h>
#define TASK_STR		pthread_t
#define get_time_ns		gethrtime
#define rt_task_wait_period	pthread_wait_np
#endif

// prototypes
void *rt_task(void *);
int jitter_output( char *buf, char **start, off_t offset, int len, int unused);

// Task control data
struct t_dat_ {
    TASK_STR task;
    VP_FP func;
    int arg;
    int stack_size;
    int priority;
    int period;
    int    uses_fpu;
    V_FP_V sig_han;
    long long period_ns;
    long long when_ns;

} td = {
           // func    arg  stack prior period uses_fpu sig_han 
         {0}, rt_task, 0,  3000,    5,      1,       0,     0
};


// global data
struct data_ { 
	long long av;
	long long min; 
	long long max; 
} data = { 0, 0, 0 };

struct proc_dir_entry proc_entry = {
        0,                      // low_ino: the inode -- dynamic
        6, "jitter",            // len of name and name
        S_IFREG | S_IRUGO,      // mode
        1, 0, 0,                // nlinks, owner, group */
        0, NULL,                // size - unused; operations -- use default
        &jitter_output,          // function used to read data
    };

// realtime task
void *rt_task(void *arg) {

	long long diff;
	long long now;
	long long expected = 0;

	while (1) {
		rt_task_wait_period();
		now = get_time_ns();
		if( expected == 0 ) {
			expected = now + td.period_ns;
			continue;
		}

		diff = now - expected;
		if( diff < data.min ) {
			data.min = diff;
		}
		if( diff > data.max) {
			data.max = diff;
		}
		expected = now + td.period_ns;
	}
}

// proc output routine
int jitter_output( char *buf, char **start, off_t offset, int len, int unused)
{
    len = 0;
    len += sprintf(buf+len, "min: %8d ns, max: %8d ns\n", 
				(int) data.min, (int) data.max);
    return len;
}

// module initialisation
int init_module(void)
{
	int rt_task_create( struct t_dat_ *t);
	proc_register(&proc_root, &proc_entry);
	rt_task_create(&td);
	return 0;
}

// task creation wrapper.
int rt_task_create( struct t_dat_ *t)
{
#ifndef __RTL__
    static int timer_started = 0;

    if( timer_started == 0) {
        // Start timer, this defaults to periodic mode.
        rt_set_oneshot_mode();
        start_rt_timer(nano2count(BASE_PERIOD));
        timer_started   = 1;
    }

    rt_task_init(       &t->task,
                        (V_FP_I)t->func,
                        t->arg,
                        t->stack_size,
                        t->priority,
                        t->uses_fpu,
                        t->sig_han
                );
    t->when_ns          = rt_get_time_ns() + 1 * NSECS_PER_SEC;
    t->period_ns        = (long long)t->period * BASE_PERIOD;

    rt_task_make_periodic(
                        &t->task,
                        nano2count(t->when_ns),
                        nano2count(t->period_ns)
                );


#else                                   // RTLinux 
    struct sched_param p;
    pthread_create(     &t->task,
                        NULL,
                        t->func,
                        (void *)t->arg
                );
    t->when_ns          = get_time_ns() + 1 * NSECS_PER_SEC;
    t->period_ns        = (long long)t->period * BASE_PERIOD;
    pthread_make_periodic_np(
                        t->task,
                        t->when_ns,
                        t->period_ns
                );
    p.sched_priority = 100 - t->priority;
    pthread_setschedparam(t->task, SCHED_FIFO, &p);

#endif

    return 0;
}

void cleanup_module(void)
{

#ifndef __RTL__
	stop_rt_timer();	
	rt_busy_sleep(1E7);
	rt_task_suspend(&td.task);
	rt_task_delete(&td.task);
#else
	pthread_delete_np(td.task);
#endif
}

