/*=============================================================================
    "pci-das6402.c" rev. 0.02            5-10-2000    (revision 0.01 4-18-2000)

    A Linux driver for the ComputerBoards PCI-DAS6402/16 data acquisition card
    (note: this driver is not for the ISA das6402)

    Copyright (C) 2000 Jim Henson's Creature Shop - LA
    written by Steve Rosenbluth
        (stever8@earthlink.net, stever@la.creatureshop.henson.com)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

==============================================================================*/
/*  epson dot-matrix printer condense code (137 chars per row on 8.5x11)       */
/* this condense code on dot matrix printers gives no line-wrap comments */

/*****************************************************************************
    _Release Notes_

    Revision 0.01 :    (Steve Rosenbluth)
        Loading and talking to card.
        using 8255 digital IO.

    Revision 0.02 :     (Steve Rosenbluth)
        Got external interrupts to work (both Linux and RTLinux).
        Getting ADC channel scans but no irq from them.
        File operations ioctl() and read() implemented.

*****************************************************************************/


#include <linux/version.h>
#include <linux/module.h>
#include <linux/config.h>
#include <linux/init.h>
#include <linux/types.h>  
#include <linux/ioport.h>

/*#include <linux/mm.h>*/
/*#include <linux/sched.h>*/
/*#include <linux/stat.h>*/
#include <linux/pci.h>
/*#include <asm/dma.h>*/
/*#include <sys/io.h>*/
#include <asm/io.h>
/*#include <asm/spinlock.h>*/
/*#include <asm/system.h>*/
/*       #include <sys/time.h>*/
#include <linux/malloc.h>                                                   /* for kmalloc(), kfree() */
#include <asm/uaccess.h>                                                    /* for verify_area() */
#include <asm/delay.h>                                                      /* for udelay() */

#include "pci-das6402.h"

#ifdef RTLINUX
    #include <rtlinux/rtl_core.h>                                           /* for rtl_hard_enable_irq, rtl_request_irq */
#endif


#define DEBUG_LEVEL 4                                                       /* decrease this to decrease verbosity */
#ifdef DEBUG_LEVEL
    static int debug_level = DEBUG_LEVEL;
    MODULE_PARM(debug_level, "i");
    #define DEBUG(n, args...) if (debug_level>(n)) printk(KERN_DEBUG "pci-das6402: " args)
#else
    #define DEBUG(n, args...)
#endif


/**************** prototypes **********************/
/* device functions */
void pci_das6402_config_8255(pci_das6402_dev_priv_t *dev);       /* configure 8255 digital io chip */
void pci_das6402_write_8255(pci_das6402_dev_priv_t *dev);       /* write 8255 digital io chip ports */
int pci_das6402_init(pci_das6402_dev_priv_t *dev);                                       /* device specific initialization for each device */
/*static void plx_init(void *addr, u32 initctl);*/                          /* for initializing PLX9080 chip */
/* driver events */
void pci_das6402_16_irq_handler(int irq, void *dev_id, struct pt_regs *regs);     /* interrupt handler */
int pci_das6402_fop_ioctl(struct inode *inode, struct file *filp, unsigned int cmd, unsigned long arg);
int pci_das6402_fop_open(struct inode *inode_p, struct file *file_p);
ssize_t pci_das6402_fop_read(struct file *filp, char *buf, size_t count, loff_t *offset_p);
int pci_das6402_fop_release(struct inode *inode_p, struct file *file_p);
#ifdef RTLINUX
    unsigned int das6402_rtirq_handler(unsigned int irq, struct pt_regs *regs);     /* for Real Time Linux irqs */
#endif

/************* global variables ******************/
static const char dev_name[] = "pci-das6402";                                   /* driver identifier string */

static pci_das6402_dev_priv_t *dev_list_p=NULL;                             /* root of llist of instances of this device */
/*pci_das6402_dev_priv_t *dev;*/                                            /* global pointer to current on device list */
/*static int device_count=0;*/                                                  /* quantity of devices (cards) in use */

/* user can set these at load time */
static int irq=-1;
static unsigned long io=0x0;
static unsigned long mem=0x0;
MODULE_PARM(irq, "i");
MODULE_PARM(io, "i");
MODULE_PARM(mem, "i");

static char *version =
"pci-das6402.c  v0.02  May 10, 2000  (by Steve Rosenbluth)";

/*  specify our method entry points. struct defined in linux/fs.h */
struct file_operations pci_das6402_fops = {
    NULL,           /* lseek */
    pci_das6402_fop_read,
    NULL,           /* write */
    NULL,           /* readdir */
    NULL,           /* poll */
    pci_das6402_fop_ioctl,
    NULL,           /* mmap */
    pci_das6402_fop_open,
    NULL,           /* flush */
    pci_das6402_fop_release,
    NULL,           /* fsync */
    NULL,           /* fasync */
    NULL,           /* check_media_change */
    NULL,           /* revalidate */    
    NULL            /* lock */    
};

static int pci_das6402_major = 0;                                           /* dynamic major device number (assigned dynamically) */


#if 0
/*  plx_init is a derivative work. 
    The original code comes from cyclades.c
    copyright Randolph Bentson (bentson@grieg.seaslug.org)
    and/or Ivan Passos (ivan@cyclades.com).
    The original code is free software distributed under 
    the GNU General Public License. */
/*******************************************/
static void plx_init(void *addr, u32 initctl)
{
    /* Reset PLX via PCI Adapter Software Reset bit */
    cy_writel(addr + initctl, cy_readl(addr + initctl) | 0x40000000);
    udelay(100L);
    cy_writel(addr + initctl, cy_readl(addr + initctl) & ~0x40000000);

    /* Reload Config. Registers from EEPROM */
    cy_writel(addr + initctl, cy_readl(addr + initctl) | 0x20000000);
    udelay(100L);
    cy_writel(addr + initctl, cy_readl(addr + initctl) & ~0x20000000);
}
#endif


/*******************************************/
void pci_das6402_config_8255(pci_das6402_dev_priv_t *dev)                   /* config 8255 digital io chip port directions */
{
    DEBUG(3,"config_8255()\n");

    if (dev->dio_a_dir != -1)
        if (dev->dio_a_dir == IN)
            dev->dio_config_word |= 0x10;
    if (dev->dio_b_dir != -1)
        if (dev->dio_b_dir == IN)
            dev->dio_config_word |= 0x02;
    if (dev->dio_c_lo_dir != -1)
        if (dev->dio_c_lo_dir == IN)
            dev->dio_config_word |= 0x01;
    if (dev->dio_c_hi_dir != -1)
        if (dev->dio_c_hi_dir == IN)
            dev->dio_config_word |= 0x08;

    writeb(dev->dio_config_word,dev->dio_io_base+PRI_DIO_CONFIG_REG);              /* write config to the memory mapped port */
}


/*******************************************/
void pci_das6402_write_8255(pci_das6402_dev_priv_t *dev)       /* write 8255 digital io chip ports */
{
    DEBUG(3,"write_8255()\n");

    if (dev->dio_a_dir == OUT)
        writeb(dev->dout_val_a,dev->dio_io_base+PRI_DIO_A_PORT);                    /* write byte to the port */    
    if (dev->dio_b_dir == OUT)
        writeb(dev->dout_val_b,dev->dio_io_base+PRI_DIO_B_PORT);                    /* write byte to the port */    
    if ((dev->dio_c_lo_dir == OUT) || (dev->dio_c_hi_dir == OUT))
        writeb(dev->dout_val_c,dev->dio_io_base+PRI_DIO_C_PORT);      /* write byte to the port. what happens when port is in/out ?!? */

    writeb(dev->dout_val_aux,dev->dio_io_base+AUX_DOUT_PORT);                    /* write byte to the port */    
}


/*******************************************/
int pci_das6402_program_adc(pci_das6402_dev_priv_t *dev)
{
/*    u32 sample_interval_div;*/
    u32 aggregate_rate, counts;

    DEBUG(3,"pci_das6402_program_adc()\n");

    writew(0,dev->daq_io_base+SAMP_INTRVL_LO_REG);                       /* mysterious DAQ initialization from manual */
    writew(0xFF,dev->daq_io_base+SAMP_INTRVL_HI_REG);                       /* mysterious DAQ initialization */

    dev->daq_ctl_reg1_mask = 1;                                              /* why 1 ?!? */
    writew(dev->daq_ctl_reg1_mask,dev->daq_io_base+CONTROL_REG_1);              /* prevents ADC conversion, inactive trig mode */

    dev->daq_ctl_reg0_mask = SOFT_TRIG|SOFT_GATE|AGATE_LEVEL_SENS|GATE_AFTER_SEQ;

    writew(INTERN_CHAN_Q,dev->daq_io_base+HDWRE_CFG_REG);                   /* use internal channel queue counter to go through chans */
    writew(DAQ_BUFF_1K|DAC_BUFF_1K,dev->daq_io_base+MEM_CFG_REG); /* 1k fifo size for ADC, 1k for DAC */
    writew(0,dev->daq_io_base+Q_FIFO_PTR_CLEAR_REG);                        /* reset queue pointer to home */
    writew(0,dev->daq_io_base+ADC_FIFO_PTR_CLR_REG);                        /* reset adc buff ptr, clear pipeline regs PIPE0 & PIPE1 */

    dev->daq_ctl_reg1_mask |= MULTI_CHAN_MODE;
    writew(dev->daq_ctl_reg1_mask,dev->daq_io_base+CONTROL_REG_1);

    writew(dev->num_scans & 0xFFFF,dev->daq_io_base+SAMP_SCAN_CNT_LO_REG);  /* specify LSWord and MSByte of scan quantity  */
    writew((dev->num_scans >> 16) & 0xFF,dev->daq_io_base+SAMP_SCAN_CNT_HI_REG);

    writew(dev->hi_scan_chan,dev->daq_io_base+Q_HIGH_REG);                      /* specify last channel of scan for internal queue counter */
    /* set input queue ctr(mux) first chan, volt gain, polarity, */
    writew(dev->lo_scan_chan|BI_TEN_VOLT|SINGLE_ENDED,dev->daq_io_base+Q_LOAD_REG); 

    writew((dev->adc_rate_divisor-3) & 0xFFFF,dev->daq_io_base+SAMP_INTRVL_LO_REG);
    writew((dev->adc_rate_divisor) >> 16,dev->daq_io_base+SAMP_INTRVL_HI_REG);

    aggregate_rate = dev->sample_rate * (dev->hi_scan_chan - dev->lo_scan_chan + 1);
    counts = (INTERNAL_CLOCK_HZ/aggregate_rate) - 3;
    writew(counts & 0xFFFF,dev->daq_io_base+DELAY_INTRVL_LO_REG);
    writew(counts >> 16,dev->daq_io_base+DELAY_INTRVL_HI_REG);

/*former    writew(DAQ_IRQ_ENAB|IRQ_END_SCAN,dev->daq_io_base+INT_ENAB_REG);*/
/*new */   writew(DAQ_IRQ_ENAB|IRQ_END_SCAN|DAQ_DONE,dev->daq_io_base+INT_ENAB_REG);

    /* enable pci interrupts through PCI9080 chip */
    writel(PCI_INT_ENAB|PCI_LOCAL_INT_ENAB|LOCAL_INT_OUT_ENAB|LOCAL_DOORBELL_INT_ENAB,
        dev->pci_9080_cfg_base+PCI_9080_INT_CTL_STAT_REG);

    dev->daq_ctl_reg0_mask |= DAQ_ENAB;
    writew(dev->daq_ctl_reg0_mask,dev->daq_io_base+CONTROL_REG_0);      /* enable data acq, cause acq after scan, level sensitive gate */

    dev->daq_ctl_reg1_mask |= SOFT_DAQ_GATE;
    writew(dev->daq_ctl_reg1_mask,dev->daq_io_base+CONTROL_REG_1);          /* un-gate data acquisition */

    writew(0,dev->daq_io_base+DAQ_SOFT_START_REG);                          /* start the conversions */


#if 0   /* fifth try - external irqs: works OK ! */
    writew(EXTERN_INT_ENAB,dev->daq_io_base+INT_ENAB_REG);                  /* enable the external irq line */
    writew(EXTERN_INT_FALLING,dev->daq_io_base+HDWRE_CFG_REG);
    writel(PCI_INT_ENAB|PCI_LOCAL_INT_ENAB,dev->pci_9080_cfg_base+PCI_9080_INT_CTL_STAT_REG);
#endif

#if 0   /* fourth try - continuous sampling, not done */

    writew(0,dev->daq_io_base+SAMP_INTRVL_LO_REG);                       /* mysterious DAQ initialization ?!?*/
    writew(0xFF,dev->daq_io_base+SAMP_INTRVL_HI_REG);                       /* mysterious DAQ initialization ?!?*/

    dev->daq_ctl_reg1_mask = 1;                                              /* why 1 ?!? */
    writew(dev->daq_ctl_reg1_mask,dev->daq_io_base+CONTROL_REG_1);              /* prevents ADC conversion, inactive trig mode */

    dev->daq_ctl_reg0_mask = SOFT_TRIG|SOFT_GATE|AGATE_LEVEL_SENS|GATE_AFTER_SEQ|TRIG_2_FALL_EDGE;

    writew(DMA_CHAN0_DAC|INTERN_CHAN_Q,dev->daq_io_base+HDWRE_CFG_REG);     /* use internal channel queue counter to go through chans */

    writew(0xFF7F,dev->daq_io_base+MEM_CFG_REG); /* FF7F ?!? */ 
    writew(0,dev->daq_io_base+Q_FIFO_PTR_CLEAR_REG);                        /* reset queue pointer to home */
    writew(0,dev->daq_io_base+ADC_FIFO_PTR_CLR_REG);                        /* reset adc buff ptr, clear pipeline regs PIPE0 & PIPE1 */

    dev->daq_ctl_reg1_mask |= MULTI_CHAN_MODE;
    writew(dev->daq_ctl_reg1_mask,dev->daq_io_base+CONTROL_REG_1);

    writew(dev->num_scans & 0xFFFF,dev->daq_io_base+SAMP_SCAN_CNT_LO_REG);  /* specify LSWord and MSByte of scan quantity  */
    writew((dev->num_scans >> 16) & 0xFF,dev->daq_io_base+SAMP_SCAN_CNT_HI_REG);

    writew(dev->hi_scan_chan,dev->daq_io_base+Q_HIGH_REG);                  /* specify last channel of scan for internal queue counter */
    /* set input queue ctr(mux) first chan, volt gain, polarity, */
    writew(dev->lo_scan_chan|BI_TEN_VOLT|SINGLE_ENDED,dev->daq_io_base+Q_LOAD_REG); 

#if 0
    writew((dev->adc_rate_divisor-3) & 0xFFFF,dev->daq_io_base+SAMP_INTRVL_LO_REG);
    writew((dev->adc_rate_divisor) >> 16,dev->daq_io_base+SAMP_INTRVL_HI_REG);
#endif

    aggregate_rate = dev->sample_rate * (dev->hi_scan_chan - dev->lo_scan_chan + 1);
    counts = (INTERNAL_CLOCK_HZ/aggregate_rate) - 3;
    writew(counts & 0xFFFF,dev->daq_io_base+DELAY_INTRVL_LO_REG);
    writew(counts >> 16,dev->daq_io_base+DELAY_INTRVL_HI_REG);

    /* dma threshold config */
    writel(0x4000000,dev->pci_9080_cfg_base+PCI_9080_DMA_THRESH_REG);

    /* set up scatter gather here */

    writew(DAQ_OVERRUN,dev->daq_io_base+INT_ENAB_REG);

    /* enable pci interrupts through PCI9080 chip ?!? */
    writel(0xF090900,dev->pci_9080_cfg_base+PCI_9080_INT_CTL_STAT_REG);
    writeb(0x13,dev->pci_9080_cfg_base+PCI_9080_DMA_CH1_CMD_STAT_REG+1); /* bad reg?!? */

    dev->daq_ctl_reg0_mask |= DAQ_ENAB;
    writew(dev->daq_ctl_reg0_mask,dev->daq_io_base+CONTROL_REG_0);      /* enable data acq, cause acq after scan, level sensitive gate */

    dev->daq_ctl_reg1_mask |= SOFT_DAQ_GATE;
    writew(dev->daq_ctl_reg1_mask,dev->daq_io_base+CONTROL_REG_1);          /* un-gate data acquisition */

    writew(0,dev->daq_io_base+DAQ_SOFT_START_REG);                          /* start the conversions */
#endif


#if 0   /* third try : for single scan & irq. exactly like 2nd CB pseudocode... */

#if 0
    writew(0,dev->daq_io_base+SAMP_INTRVL_LO_REG);                       /* mysterious DAQ initialization ?!?*/
    writew(0xFF,dev->daq_io_base+SAMP_INTRVL_HI_REG);                       /* mysterious DAQ initialization ?!?*/
#endif

    dev->daq_ctl_reg1_mask = 1;                                              /* why 1 ?!? */
    writew(dev->daq_ctl_reg1_mask,dev->daq_io_base+CONTROL_REG_1);              /* prevents ADC conversion, inactive trig mode */

    dev->daq_ctl_reg0_mask = SOFT_TRIG|SOFT_GATE|AGATE_LEVEL_SENS|GATE_AFTER_SEQ|TRIG_2_FALL_EDGE;

    writew(DMA_CHAN0_DAC|INTERN_CHAN_Q,dev->daq_io_base+HDWRE_CFG_REG);                   /* use internal channel queue counter to go through chans */
/*    writew(DAQ_BUFF_1K|DAC_BUFF_1K,dev->daq_io_base+MEM_CFG_REG);*/ /* FF7F ?!? */ /* 1k fifo size for ADC, 1k for DAC */
    writew(0xFF7F,dev->daq_io_base+MEM_CFG_REG); /* FF7F ?!? */ 
    writew(0,dev->daq_io_base+Q_FIFO_PTR_CLEAR_REG);                        /* reset queue pointer to home */
    writew(0,dev->daq_io_base+ADC_FIFO_PTR_CLR_REG);                        /* reset adc buff ptr, clear pipeline regs PIPE0 & PIPE1 */

    dev->daq_ctl_reg1_mask |= MULTI_CHAN_MODE;
    writew(dev->daq_ctl_reg1_mask,dev->daq_io_base+CONTROL_REG_1);

    writew(dev->num_scans & 0xFFFF,dev->daq_io_base+SAMP_SCAN_CNT_LO_REG);  /* specify LSWord and MSByte of scan quantity  */
    writew((dev->num_scans >> 16) & 0xFF,dev->daq_io_base+SAMP_SCAN_CNT_HI_REG);

    writew(dev->hi_scan_chan,dev->daq_io_base+Q_HIGH_REG);                      /* specify last channel of scan for internal queue counter */
    /* set input queue ctr(mux) first chan, volt gain, polarity, */
    writew(dev->lo_scan_chan|BI_TEN_VOLT|SINGLE_ENDED,dev->daq_io_base+Q_LOAD_REG); 

#if 0
    writew((dev->adc_rate_divisor-3) & 0xFFFF,dev->daq_io_base+SAMP_INTRVL_LO_REG);
    writew((dev->adc_rate_divisor) >> 16,dev->daq_io_base+SAMP_INTRVL_HI_REG);
#endif

    aggregate_rate = dev->sample_rate * (dev->hi_scan_chan - dev->lo_scan_chan + 1);
    counts = (INTERNAL_CLOCK_HZ/aggregate_rate) - 3;
    writew(counts & 0xFFFF,dev->daq_io_base+DELAY_INTRVL_LO_REG);
    writew(counts >> 16,dev->daq_io_base+DELAY_INTRVL_HI_REG);

    /* dma threshold config */
    writel(0x4000000,dev->pci_9080_cfg_base+PCI_9080_DMA_THRESH_REG);

    dev->daq_ctl_reg0_mask |= SAMPLE_CNT_ENAB;
    writew(dev->daq_ctl_reg0_mask,dev->daq_io_base+CONTROL_REG_0);

    /* supposed to set up irq handler here ... */

    writew(DAQ_DONE|DAQ_OVERRUN,dev->daq_io_base+INT_ENAB_REG);

    /* enable pci interrupts through PCI9080 chip ?!? */
    writel(0xF090900,dev->pci_9080_cfg_base+PCI_9080_INT_CTL_STAT_REG);

    dev->daq_ctl_reg0_mask |= DAQ_ENAB;
    writew(dev->daq_ctl_reg0_mask,dev->daq_io_base+CONTROL_REG_0);      /* enable data acq, cause acq after scan, level sensitive gate */

    dev->daq_ctl_reg1_mask |= SOFT_DAQ_GATE;
    writew(dev->daq_ctl_reg1_mask,dev->daq_io_base+CONTROL_REG_1);          /* un-gate data acquisition */

    writew(0,dev->daq_io_base+DAQ_SOFT_START_REG);                          /* start the conversions */
#endif


#if 0   /* second try at single scan & irq */
    writew(0,dev->daq_io_base+SAMP_INTRVL_LO_REG);                       /* mysterious DAQ initialization ?!?*/
    writew(0xFF,dev->daq_io_base+SAMP_INTRVL_HI_REG);                       /* mysterious DAQ initialization ?!?*/

    dev->daq_ctl_reg1_mask = 1;                                              /* why 1 ?!? */
    writew(dev->daq_ctl_reg1_mask,dev->daq_io_base+CONTROL_REG_1);              /* prevents ADC conversion, inactive trig mode */

    dev->daq_ctl_reg0_mask = SOFT_TRIG|SOFT_GATE|AGATE_LEVEL_SENS|GATE_AFTER_SEQ;

    /* why not write ctl_reg0 here ?!? */
    /*writew(dev->daq_ctl_reg0_mask,dev->daq_io_base+CONTROL_REG_0);*/ /* just testing ?!? */

    writew(INTERN_CHAN_Q,dev->daq_io_base+HDWRE_CFG_REG);                   /* use internal channel queue counter to go through chans */
    writew(DAQ_BUFF_1K|DAC_BUFF_1K,dev->daq_io_base+MEM_CFG_REG); /* FF7F ?!? */ /* 1k fifo size for ADC, 1k for DAC */
    writew(0,dev->daq_io_base+Q_FIFO_PTR_CLEAR_REG);                        /* reset queue pointer to home */
    writew(0,dev->daq_io_base+ADC_FIFO_PTR_CLR_REG);                        /* reset adc buff ptr, clear pipeline regs PIPE0 & PIPE1 */

    dev->daq_ctl_reg1_mask |= MULTI_CHAN_MODE;
    writew(dev->daq_ctl_reg1_mask,dev->daq_io_base+CONTROL_REG_1);

    writew(dev->num_scans & 0xFFFF,dev->daq_io_base+SAMP_SCAN_CNT_LO_REG);  /* specify LSWord and MSByte of scan quantity  */
    writew((dev->num_scans >> 16) & 0xFF,dev->daq_io_base+SAMP_SCAN_CNT_HI_REG);

    writew(dev->hi_scan_chan,dev->daq_io_base+Q_HIGH_REG);                      /* specify last channel of scan for internal queue counter */
    /* set input queue ctr(mux) first chan, volt gain, polarity, */
    writew(dev->lo_scan_chan|BI_TEN_VOLT|SINGLE_ENDED,dev->daq_io_base+Q_LOAD_REG); 

    writew((dev->adc_rate_divisor-3) & 0xFFFF,dev->daq_io_base+SAMP_INTRVL_LO_REG);
    writew((dev->adc_rate_divisor) >> 16,dev->daq_io_base+SAMP_INTRVL_HI_REG);

    aggregate_rate = dev->sample_rate * (dev->hi_scan_chan - dev->lo_scan_chan + 1);
    counts = (INTERNAL_CLOCK_HZ/aggregate_rate) - 3;
    writew(counts & 0xFFFF,dev->daq_io_base+DELAY_INTRVL_LO_REG);
    writew(counts >> 16,dev->daq_io_base+DELAY_INTRVL_HI_REG);

    /* dma threshold config was here */

    writew(DAQ_IRQ_ENAB|IRQ_END_SCAN,dev->daq_io_base+INT_ENAB_REG);

    /* enable pci interrupts through PCI9080 chip */
    writel(PCI_INT_ENAB|PCI_LOCAL_INT_ENAB|LOCAL_INT_OUT_ENAB|LOCAL_DOORBELL_INT_ENAB,
        dev->pci_9080_cfg_base+PCI_9080_INT_CTL_STAT_REG);

    /* dma command status was here */

    dev->daq_ctl_reg0_mask |= DAQ_ENAB;
    writew(dev->daq_ctl_reg0_mask,dev->daq_io_base+CONTROL_REG_0);      /* enable data acq, cause acq after scan, level sensitive gate */

    dev->daq_ctl_reg1_mask |= SOFT_DAQ_GATE;
    writew(dev->daq_ctl_reg1_mask,dev->daq_io_base+CONTROL_REG_1);          /* un-gate data acquisition */

    writew(0,dev->daq_io_base+DAQ_SOFT_START_REG);                          /* start the conversions */
#endif

#if 0   /* first try at single scan & irq. based on CB code for DMA xferrs */
    dev->daq_ctl_reg1_mask = 1;                                              /* why 1 ?!? */
    writew(dev->daq_ctl_reg1_mask,dev->daq_io_base+CONTROL_REG_1);              /* prevents ADC conversion, inactive trig mode */

    dev->daq_ctl_reg0_mask = SOFT_TRIG|SOFT_GATE|AGATE_LEVEL_SENS|GATE_AFTER_SEQ;

    /* why not write ctl_reg0 here ?!? */
    /*writew(dev->daq_ctl_reg0_mask,dev->daq_io_base+CONTROL_REG_0);*/ /* just testing ?!? */

    writew(INTERN_CHAN_Q,dev->daq_io_base+HDWRE_CFG_REG);                   /* use internal channel queue counter to go through chans */
    writew(DAQ_BUFF_1K|DAC_BUFF_1K,dev->daq_io_base+MEM_CFG_REG); /* FF7F ?!? */ /* 1k fifo size for ADC, 1k for DAC */
    writew(0,dev->daq_io_base+Q_FIFO_PTR_CLEAR_REG);                        /* reset queue pointer to home */
    writew(0,dev->daq_io_base+ADC_FIFO_PTR_CLR_REG);                        /* reset adc buff ptr, clear pipeline regs PIPE0 & PIPE1 */

    dev->daq_ctl_reg1_mask |= MULTI_CHAN_MODE;
    writew(dev->daq_ctl_reg1_mask,dev->daq_io_base+CONTROL_REG_1);

    writew(dev->num_scans & 0xFFFF,dev->daq_io_base+SAMP_SCAN_CNT_LO_REG);  /* specify LSWord and MSByte of scan quantity  */
    writew((dev->num_scans>>16) & 0xFF,dev->daq_io_base+SAMP_SCAN_CNT_HI_REG);

    writew(dev->hi_scan_chan,dev->daq_io_base+Q_HIGH_REG);                      /* specify last channel of scan for internal queue counter */
    /* set input queue ctr(mux) first chan, volt gain, polarity, */
    writew(dev->lo_scan_chan|BI_TEN_VOLT|SINGLE_ENDED,dev->daq_io_base+Q_LOAD_REG); 

    /* dma threshold config was here */

    writew(DAQ_IRQ_ENAB|IRQ_END_SCAN,dev->daq_io_base+INT_ENAB_REG);

    /* enable pci interrupts through PCI9080 chip ?!? */
/*    writel(PCI_INT_ENAB|PCI_LOCAL_INT_ENAB|LOCAL_INT_OUT_ENAB,
        dev->pci_9080_cfg_base+PCI_9080_INT_CTL_STAT_REG);*/

    /* dma command status was here */

    dev->daq_ctl_reg0_mask |= DAQ_ENAB;
    writew(dev->daq_ctl_reg0_mask,dev->daq_io_base+CONTROL_REG_0);      /* enable data acq, cause acq after scan, level sensitive gate */

    dev->daq_ctl_reg1_mask |= SOFT_DAQ_GATE;
    writew(dev->daq_ctl_reg1_mask,dev->daq_io_base+CONTROL_REG_1);          /* un-gate data acquisition */

    writew(0,dev->daq_io_base+DAQ_SOFT_START_REG);                          /* start the conversions */
#endif

    return 0;
}


/* FIXME: phase out arg and store global config on ioctl ?!? */
/* note that magic numbers have not been assigned to cmd, so errant ioctls could reconfigure this card */
/*******************************************/
int pci_das6402_fop_ioctl(struct inode *inode_p, struct file *file_p, unsigned int cmd, unsigned long arg)
{
    int ret;
    ctl_msg_struct_t *ctl_p;                                                /* this struct sent to us from user, holds configuration */
    pci_das6402_dev_priv_t *dev;
    static int last_command=-1;
    u16 status;
    u32 plx9080_status;

    DEBUG(3,"fop_ioctl(): cmd is %d, arg is %p\n",cmd,(void *)arg);

    if (dev_list_p)
        dev = dev_list_p;                                                   /* FIXME: implement llist of devs */
    else {
        printk(KERN_WARNING "pci_das6402_fop_ioctl(): module has not been initialized properly!");
        return ERROR;
    }

    /* see if pointer is valid. defined in uaccess.h */
    ret = verify_area(VERIFY_READ,(ctl_msg_struct_t *)arg,sizeof(ctl_msg_struct_t));
    if (ret != 0)                                                           /* problem */
    {
        printk("pci_das6402_fop_ioctl(): Couldn't read user struct!\n");
        return ret;
    }
    ctl_p = (ctl_msg_struct_t *)arg;

    switch (ctl_p->conversion_mode)
    {
    case CONTIN_ACQUISITION:
        dev->burst_acq = FALSE;                                   /* allow acquisition to be continuous */
        DEBUG(3, "set contin acq\n");
        break;
    case BURST_ACQUISITION:
        dev->burst_acq = TRUE;                                    /* do a scan of channels */
        DEBUG(3, "set burst acq\n");
        break;
    }

    switch (cmd)
    {
    case CONFIGURE_ACQUISITION:                              /* configure the accquisition mode */
        if (ctl_p->dio_a_dir != -1)                                         /* config dio port directions */
            dev->dio_a_dir = ctl_p->dio_a_dir;
        if (ctl_p->dio_b_dir != -1)
            dev->dio_b_dir = ctl_p->dio_b_dir;
        if (ctl_p->dio_c_lo_dir != -1)
            dev->dio_c_lo_dir = ctl_p->dio_c_lo_dir;
        if (ctl_p->dio_c_hi_dir != -1)
            dev->dio_c_hi_dir = ctl_p->dio_c_hi_dir;
        pci_das6402_config_8255(dev);                                       /* configure the 8255 dio chip */
        dev->send_dig_ins = TRUE;  /*?!? necessary on this card ?*/                      /* this change forces a new digital sample */
#if 0
        if (ctl_p->adc_rate_divisor > 0)
        {
            DEBUG(5,"clk=%d / rate=%d = %d\n",INTERNAL_CLOCK_HZ,ctl_p->adc_rate_divisor,(INTERNAL_CLOCK_HZ/ctl_p->adc_rate_divisor));
            if ((INTERNAL_CLOCK_HZ/ctl_p->adc_rate_divisor) <= MAX_SAMPLE_RATE_HZ)
            {   /* FIXME: check for single/multi channel - 200,100KHz */
                dev->adc_rate_divisor = ctl_p->adc_rate_divisor;            /* set counter divisor, start acq uses this */
                DEBUG(3, "adc rate div: %u\n",dev->adc_rate_divisor);
            }
        }
#endif
        if ((ctl_p->sample_rate > 0) && (ctl_p->sample_rate <= MAX_SAMPLE_RATE_HZ))
        {
            /* FIXME: check for single/multi channel - 200,100KHz */
            dev->sample_rate = ctl_p->sample_rate;
            dev->adc_rate_divisor = INTERNAL_CLOCK_HZ/dev->sample_rate;     /* set counter divisor */
            DEBUG(3, "sample_rate: %d, rate divisor: %u\n",dev->sample_rate,dev->adc_rate_divisor);
        }
        if ((ctl_p->num_scans <= MAX_SCANS) && (ctl_p->num_scans > 0))
        {
            dev->num_scans = ctl_p->num_scans;                              /* set quantity of scans for the card to convert */
            if (dev->num_scans > 1) {
                dev->multiscanning = TRUE;
                DEBUG(3, "multi-scanning\n");
            } else
                dev->multiscanning = FALSE;
            dev->cur_scan = 0;                                                   /* re-init */
            DEBUG(3, "num scans: %d\n",dev->num_scans);
        }
        if ((ctl_p->lo_scan_chan > -1) && (ctl_p->lo_scan_chan < MAX_CHANNELS))
        {
            dev->lo_scan_chan = ctl_p->lo_scan_chan;
            if ((ctl_p->hi_scan_chan > -1) && (ctl_p->hi_scan_chan < MAX_CHANNELS) && (ctl_p->hi_scan_chan >= ctl_p->lo_scan_chan))
            {
                dev->hi_scan_chan = ctl_p->hi_scan_chan;
                dev->channel_scan = dev->hi_scan_chan - dev->lo_scan_chan + 1;   /* set quantity of channels for the card to scan (1-64) */
            }
            dev->chan_cnt = 0;                                              /* re-init num samples gotten */
            DEBUG(3, "num chan: %d, low chan: %d, hi chan: %d\n",dev->channel_scan,dev->lo_scan_chan,dev->hi_scan_chan);
        }
            /* accept any single ended or bipolar 10Volt, 5V, 2.5V, 1.25V */
            if ((ctl_p->volt_range==BI_TEN_VOLT)||(ctl_p->volt_range==BI_FIVE_VOLT)||(ctl_p->volt_range==BI_TWO_VOLT)||
                (ctl_p->volt_range==BI_ONE_VOLT)||(ctl_p->volt_range==UNI_TEN_VOLT)||(ctl_p->volt_range==UNI_FIVE_VOLT)||
                (ctl_p->volt_range==UNI_TWO_VOLT)||(ctl_p->volt_range==UNI_ONE_VOLT))
        {
            dev->volt_range = (ctl_p->volt_range << 8);                     /* clear the two lsb's of input mode, "0" is 10V range */
            DEBUG(3, "volt range code : %u\n",dev->volt_range);
        }
        /*pci_das6402_program_adc(dev);*/                                          /* now program the adc scan and counters */
        break;
    case START_ACQUISITION:                                                 /* start accquisition */
        pci_das6402_program_adc(dev);                                          /* now program the adc scan and counters */
        break;
    case STOP_ACQUISITION:                                                                 /* stop accquisition */
        /* all this is just for testing. remove it ?!? */
        status = readw(dev->daq_io_base+PIPE1_READ_REG);
        DEBUG(3,"PIPE1_READ_REG = %u\n",status);
        /*writew(0,dev->daq_io_base+ADC_FIFO_PTR_CLR_REG);*/               /* reset adc buff ptr, clear pipeline regs PIPE0 & PIPE1 */
        break;
    case DIGITAL_OUTPUT:                                                    /* digital output */
        dev->dout_val_a = ctl_p->dout_val_a;                    /* store the byte */
        dev->dout_val_b = ctl_p->dout_val_b;                    /* store the byte */
        dev->dout_val_c = ctl_p->dout_val_c;                    /* store the byte */
        dev->dout_val_aux = ctl_p->dout_val_aux;                    /* store the byte */
        DEBUG(3,"dig out 'a' val : %d\n",dev->dout_val_a);
        pci_das6402_write_8255(dev);                                        /* actually write the bytes out */
        break;        
    case 4: 
#if 0
        /* FIXME: move to config ?!? */
        das16_program_timers(ctl_p->adc_rate_divisor,ctl_p->aux_adc_rate_divisor, DAS_START);
#endif
        break;
    case QUERY_ACQUISITION:  /* case 6 */
        DEBUG(3," -query-\n");
        plx9080_status = readl(dev->pci_9080_cfg_base+PCI_9080_INT_CTL_STAT_REG);
        DEBUG(3,"PCI_9080_INT_CTL_STAT_REG = %u\n",plx9080_status);
        status = readw(dev->daq_io_base+HDWRE_STAT_REG);
        DEBUG(3,"HDWRE_STAT_REG = %u\n",status);
        status = readw(dev->daq_io_base+ADC_READ_PTR_REG);
        DEBUG(3,"ADC_READ_PTR_REG = %u\n",status);
        status = readw(dev->daq_io_base+ADC_WRITE_PTR_REG);
        DEBUG(3,"ADC_WRITE_PTR_REG = %u\n",status);
        status = readw(dev->daq_io_base+LO_USR_XFER_CTR_REG);
        DEBUG(3,"LO_USR_XFER_CTR_REG = %u\n",status);
        status = readw(dev->daq_io_base+PRE_POST_REG);
        DEBUG(3,"PRE_POST_REG = %u\n",status);
        break;
    default:
        DEBUG(1, "pci_das6402_fop_ioctl: bad command (%u)\n",cmd);
        return -EINVAL;
        break;
    }
    last_command = cmd;

    return 0;
}   /* pci_das6402_fop_ioctl */


/*******************************************/
int pci_das6402_fop_open(struct inode *inode_p, struct file *file_p)
{
    DEBUG(3,"fop_open()\n");
#if 0
    /*in fs.h: MODE_READ 1, FMODE_WRITE 2*/
    /*printk(KERN_INFO "fop_open(): f_mode is 0x%x\n",file_p->f_mode);*/

#endif
    MOD_INC_USE_COUNT;
    return 0;
}


/*******************************************/
ssize_t pci_das6402_fop_read(struct file *filp, char *buf, size_t count, loff_t *offset_p)
{
    /*struct inode* inode = filp->f_dentry->d_inode;*/
    /*unsigned int minor = MINOR(inode->i_rdev);*/
    /*char *data_p;*/
    /*int packets_read=0, packet_count=0;*/
    /*unsigned int interrupt_state=0;*/
    int i, wordcount;
    pci_das6402_dev_priv_t *dev;
    u16 tempbuf[MAX_CHANNELS]/*, addr*/;

    DEBUG(3,"fop_read()\n");

    if (dev_list_p == NULL)
    {
        DEBUG(3,"fop_read(): module has not been initialized properly!\n");
        return -ENODEV; /* correct val ?!? */
    }
    else
    {
        dev = dev_list_p;

    if (filp->f_flags & O_NONBLOCK)                /* if reader is nonblocking */
    {
        if (/*adc write ptr reg ?!? == */ 0)   /* no data avail in FIFO */
        {
            DEBUG(3,"no results, returning 0\n");
            return 0;   /* ?!? -E */
        }
    }
    else                                          /* reader will block */
    {
        /* FIXME: implement ?!? */
        /* interruptible_sleep_on(dev->wait_queue); */
    }

#if 0
    while (/*packet_count > 0*/)
    {
        copy_to_user(buf, in, RESULT_SIZE_B);  /* actually write data to user */
        /*rtl_critical(interrupt_state);*/                                      /* mask irqs so we don't get preempted ? */
        /*CLEAR_ADC_RESULT(dev);*/                                              /* decrement table qty by one */
        /*rtl_end_critical(interrupt_state);*/            
	words_read += 1;                                  /* increment count of how many packets user has read */
    }
#endif

#if 0   /* this works ! */
    /* quick test: one sample at a time  ?!? */
    tempbuf[0] = readw(dev->daq_io_base+ADC_FIFO_REG);
    copy_to_user(buf, tempbuf, 1*RESULT_SIZE_B);  /* actually write data to user */
    return (1*RESULT_SIZE_B);   /* qty of bytes written */
#endif

    /*addr = readw(dev->daq_io_base+ADC_WRITE_PTR_REG);*/
    /*writew(addr,dev->daq_io_base+);*/
    
    /* quick test: one 64 channel scan  ?!? */
    wordcount = count/2;
    if (wordcount > MAX_CHANNELS)
        wordcount = MAX_CHANNELS;
    for (i=0; i < wordcount; i++)
        tempbuf[i] = readw(dev->daq_io_base+ADC_FIFO_REG);
    copy_to_user(buf, tempbuf, MAX_CHANNELS*RESULT_SIZE_B);             /* actually write data to user */
    return (MAX_CHANNELS*RESULT_SIZE_B);                                /* qty of bytes written */

#if 0
    if (words_read) {                             /* if we wrote any */
        inode->i_atime = CURRENT_TIME;
            /*printk("fop_read(): returning %d\n",read);*/
	return (words_read*sizeof(u16));   /* qty of bytes written */
    }
#endif

    }   /* end else have device */

    return 0;
}   /* pci_das6402_fop_read */


/*******************************************/
int pci_das6402_fop_release(struct inode *inode_p, struct file *file_p)
{
    DEBUG(3,"fop_release()\n");
    MOD_DEC_USE_COUNT;
    return 0;
}


/* The ordinary Linux interrupt handler */
/*******************************/
void pci_das6402_16_irq_handler(int irq, void *dev_id, struct pt_regs *regs)
{
    pci_das6402_dev_priv_t *dev=NULL;
    u16 irq_status;

/*    DEBUG(3,"pci_das6402_16_irq_handler()\n");*/
    printk("pci_das6402_16_irq_handler()\n");
    if (readl(dev->pci_9080_cfg_base+PCI_9080_INT_CTL_STAT_REG) & LOCAL_INT_ACTIVE) {
        DEBUG(3,"Our IRQ.\n");
    }else
        DEBUG(3,"NOT our irq !");

    if (dev_list_p)
    {
        dev = dev_list_p;

        irq_status = readw(dev->daq_io_base+HDWRE_STAT_REG);
        DEBUG(4,"irq_status: %u\n",irq_status);
        if (irq_status & EXTERN_INT_FLAG) {
            DEBUG(3,"External irq...\n");
        } else if (irq_status & DAQ_INT_FLAG)
            DEBUG(3,"DAQ irq...\n");
    }

    /*wake_up_interruptible(&pci_das6402_task_queue); FIXME: implement ?!? */
}


#ifdef RTLINUX
/* The RTLinux interrupt hadler */
/*******************************/
unsigned int das6402_rtirq_handler(unsigned int irq, struct pt_regs *regs)
{
    pci_das6402_dev_priv_t *dev=NULL;
    u16 irq_status;

    printk("RTL irq : ");
    if (readl(dev->pci_9080_cfg_base+PCI_9080_INT_CTL_STAT_REG) & LOCAL_INT_ACTIVE) 
        DEBUG(3,"Our IRQ.\n");
    else
        DEBUG(3,"NOT our irq !");

    if (dev_list_p)
    {
        dev = dev_list_p;

        irq_status = readw(dev->daq_io_base+HDWRE_STAT_REG);
        if (irq_status & EXTERN_INT_FLAG)
            printk("EXT");
        rtl_hard_enable_irq(dev->irq);                                            /* reenable this irq @ mask reg for further handler calls*/
    }

    printk("\n");

    return 0;
}
#endif

/*******************************/
int pci_das6402_init(pci_das6402_dev_priv_t *dev)                                        /* device specific initialization for each device */
{
    dev->dio_config_word = 128;

    dev->sample_rate = 50000;
    dev->adc_rate_divisor = INTERNAL_CLOCK_HZ/50000;

    /* FIXME: init all private vars here */

    return 0;
}


/* init_module called when module first loads.
    Uses pci functions from /usr/src/linux/include/linux/pci.h
    Labeled portions of this code are from cyclades.c copyright Randolph Bentson (bentson@grieg.seaslug.org)
    and/or Ivan Passos (ivan@cyclades.com). */
/*******************************/
int init_module(void)
{
    bool done, got_one;
    int i, j, ret;
    unsigned long region_size;
    u16 cur_command, new_command/*,pwr_commmand*/;
    u32 pci_ioaddr, mask;
    u32 tempaddr[5][2];
    pci_das6402_dev_priv_t *dev=NULL, *curdev=NULL;                                            /* pointer to current on device list */
    struct pci_dev *found_dev_p=NULL;
    static u32 pci_addresses[] = {
        PCI_BASE_ADDRESS_0, PCI_BASE_ADDRESS_1, PCI_BASE_ADDRESS_2, PCI_BASE_ADDRESS_3, PCI_BASE_ADDRESS_4, PCI_BASE_ADDRESS_5, 0};

    printk(KERN_DEBUG "%s\n", version);

#ifdef CONFIG_PCI
    if (pci_present())
    {
        for (i=0,done=FALSE,got_one=FALSE; (i < MAX_6402_DEVICES) && (!done); i++)          /* look for our device on bus */
        {
            found_dev_p = pci_find_device(PCI_VENDOR_ID_CBOARDS, PCI_DEVICE_ID_CBOARDS_DAS6402_16, found_dev_p);
            if (found_dev_p)
            {
                got_one = TRUE;                                             /* flag we found at least one */
                DEBUG(4,"found_dev_p=%p, base_address[0]=%lu, irq=%u.\n",found_dev_p,found_dev_p->base_address[0],found_dev_p->irq);

                dev = kmalloc(sizeof(pci_das6402_dev_priv_t),GFP_KERNEL);        /* allocate first device */
                if (dev)
                {
                    if (i==0)                                               /* first device on list */
                        dev_list_p = dev;                                   /* capture addr of first device. just a hack for now... */
                    else
                        curdev->next_dev = dev;                             /* link the list forward */
                    curdev = dev;                                           /* steps llist, one behind */

                    pci_das6402_init(dev);                                  /* init all device private vars */
                    dev->bus = (found_dev_p->bus)->number;
                    dev->dev_fn = found_dev_p->devfn;
                    DEBUG(3,"bus=%d, devfn=%x\n",dev->bus,dev->dev_fn);
                    dev->irq = found_dev_p->irq;
                    dev->name = dev_name;                                   /* put addr of name string in private struct */
                    dev->id = i;                                            /* give each instance of device a number - (the iterator) */
                    /*DEBUG(0,"addr0=0x%p addr1=0x%p addr2=0x%p\n\taddr3=0x%p  irq=%u.\n",(void *)dev->io_base0,
                        (void *)dev->io_base1,(void *)dev->io_base2,(void *)dev->io_base3,dev->irq);*/
                    /*DEBUG(0,"addr0=0x%p addr1=0x%p addr2=0x%p\n\t\taddr3=0x%p  irq=%u.\n",(void *)found_dev_p->base_address[0],
                        (void *)found_dev_p->base_address[1],(void *)found_dev_p->base_address[2],(void *)found_dev_p->base_address[3],dev->irq);*/
                    DEBUG(0,"irq = %u\n",dev->irq);
#if 0               /* don't do this, use ioremap in kernel ! */
                    DEBUG(2,"Remapping addresses ...\n");
                    dev->io_base0 = 0x4000;
                    pcibios_write_config_dword(dev->bus,dev->dev_fn,PCI_BASE_ADDRESS_0,dev->io_base0);  /* remap addr  */
                    dev->io_base2 = 0x2000;
                    pcibios_write_config_dword(dev->bus,dev->dev_fn,PCI_BASE_ADDRESS_2,dev->io_base2);  /* remap addr  */
                    dev->io_base3 = 0x3000;
                    pcibios_write_config_dword(dev->bus,dev->dev_fn,PCI_BASE_ADDRESS_3,dev->io_base3);  /* remap addr  */
#endif
                    for (j=0; j < 4; j++)                                  /* check our addresses + their size */
                    {
                        pcibios_read_config_dword(dev->bus,dev->dev_fn,pci_addresses[j],&pci_ioaddr);   /* save current addr */
                        DEBUG(5,"Found IO region%d at 0x%p ...\n",j,(void *)pci_ioaddr);
                        pcibios_write_config_dword(dev->bus,dev->dev_fn,pci_addresses[j],~0);  /* set address to all 1s */
                        pcibios_read_config_dword(dev->bus,dev->dev_fn,pci_addresses[j],&mask); /* get size of region */
                        if (mask & PCI_BASE_ADDRESS_SPACE)
                            mask &= PCI_BASE_ADDRESS_IO_MASK;
                        else
                            mask &= PCI_BASE_ADDRESS_MEM_MASK;
                        region_size = (unsigned long)~mask+1;                   /* capture region size */
                        tempaddr[j][1] = region_size;                       /* store size w/ addr */
                        DEBUG(5,"its size is %lu\n",region_size);
                        pcibios_write_config_dword(dev->bus,dev->dev_fn,pci_addresses[j],pci_ioaddr);  /* restore addr */
                        if (pci_ioaddr > 0xFFFFF)                           /* if its a high PCI region */
                        {
                            tempaddr[j][0] = (u32)ioremap(pci_ioaddr,region_size);  /* remap high addresses */
                            DEBUG(3,"Region 0x%p size=%lu, remapped to 0x%p\n",(void *)pci_ioaddr,region_size,(void *)tempaddr[j][0]);
                        }
                        else
                        {
                            tempaddr[j][0] = pci_ioaddr;                   /* save low address w/out remapping */
                            DEBUG(3,"Region 0x%p size=%lu, not remapped.\n",(void *)pci_ioaddr,region_size);
                        }
                    }
                    dev->pci_9080_cfg_base = (void *)tempaddr[0][0];
                    dev->pci_iomap_cfg_base = (void *)(tempaddr[1][0] & ~0xf);
                    dev->pci_iomap_cfg_base_size = tempaddr[1][1];
                    dev->daq_io_base = (u16 *)tempaddr[2][0];                            /* capture base addr of 16 bit regs */
                    dev->dio_io_base = (u8 *)tempaddr[3][0];                             /* capture base addr of 8 bit regs  */
                    DEBUG(3,"Read %u from DIO port A (0x%p)\n",readb(dev->dio_io_base+PRI_DIO_A_PORT),dev->dio_io_base);

#if 0
                    /* this block of code derived from cyclades.c. see notice above. */
                    plx_init(dev->pci_9080_cfg_base,PCI_9080_EEPROM_CTL_REG);
		    /* For some yet unknown reason, once the PLX9060 reloads
		       the EEPROM, the IRQ is lost and, thus, we have to
		       re-write it to the PCI config. registers.
		       This will remain here until we find a permanent fix. */
		    pci_write_config_byte(found_dev_p, PCI_INTERRUPT_LINE,dev->irq);
                    /* now enable interrupts at pci chip level*/
		    cy_writew(dev->pci_9080_cfg_base+PCI_9080_INT_CTL_STAT_REG,
                        cy_readw(dev->pci_9080_cfg_base+PCI_9080_INT_CTL_STAT_REG)|PCI_INT_ENAB|PCI_LOCAL_INT_ENAB);
#endif
                    /* register 2nd area w/ kernel (area not really used though) */
                    request_region((unsigned)dev->pci_iomap_cfg_base,dev->pci_iomap_cfg_base_size,dev_name);

                    #ifdef RTLINUX
                        DEBUG(0,"Using Real Time Linux interrupts.\n");
                        ret = rtl_request_irq(dev->irq,das6402_rtirq_handler); /* install our handler with RTLinux executive */
                        if (ret)                                              /* nonzero return means problems */
                        {
                                printk(KERN_WARNING "das6402: ERROR installing RT interrupt handler: !\n");
                                if (ret == -EBUSY)
                                    printk("RT-Interrupt is already in use.\n");
                                else if (ret == -EINVAL)
                                    printk("Invalid interrupt number, or null handler.\n");
                        }
                        else
                        {
                            rtl_hard_enable_irq(dev->irq);                  /* enable this level at mask register */
                            DEBUG(3, "Requested and hard enabled irq %d w/ handler %p\n",dev->irq,das6402_rtirq_handler);
                        }
                    #else
                        /* should do this when device is first opened, but we share irq level */
                        ret = request_irq(dev->irq,pci_das6402_16_irq_handler,SA_SHIRQ,dev_name,NULL); /* register our irq handler w/ kernel */
                        if (ret)
                            DEBUG(1,"Failed to get interrupt %d !\n",dev->irq);
                    #endif
                    pci_read_config_word(found_dev_p,PCI_COMMAND,&cur_command);     /* see what card command bits say */
                    DEBUG(3,"card command = %d\n",cur_command);
                    new_command = PCI_COMMAND_MASTER|PCI_COMMAND_IO|PCI_COMMAND_MEMORY;
                    if (cur_command != new_command)
                    {
                        DEBUG(2,"card command reg doesnt have IO,mem,mastering enabled\n\t\tWriting command...");
                        new_command |= cur_command;
                        pci_write_config_word(found_dev_p,PCI_COMMAND,new_command); /* write new config to card */
                    }

                    /* Register device major number (is set to 0): accept dynamic number. */
                    /* read /proc/devices after to find what major was assigned */
                    ret = register_chrdev(pci_das6402_major,dev_name, &pci_das6402_fops); /* our fops get registered here */
                    if (ret < 0)
                    {
                        DEBUG(0,"pci_das6402: can't get major %d\n",pci_das6402_major);
                    }
                    else
                    {
                        pci_das6402_major = ret;
                        DEBUG(0,"device major number is %d\n",pci_das6402_major);
                    }
                    pci_das6402_init(dev);                                  /* init private struct vars, etc */
                } /* end if allocated a device successfully */
                else
                {
                    printk("pci_das6402: ERROR, failed to allocate a device !\n");
                    done = TRUE;                                                /* stop searching bus */
                }
            }   /* end if found our card on pci bus */
            else
            {
                if (!got_one)
                    DEBUG(0,"didn't find our device on pci bus.\n");
                done = TRUE;                                                /* stop searching bus */
            }
        }
    }
    if (irq != -1)  /* FIXME: these just to quiet compiler warnings */
        ;
    if (io != 0)  
        ;
    if (mem != 0)  
        ;

#else
    DEBUG(0,"ERROR, pci support not compiled into kernel!\n");
    return -ENODEV;
#endif

    DEBUG(1,"module loaded.\n");
    return 0;
}


/*******************************/
void cleanup_module(void)
{
    pci_das6402_dev_priv_t *dev=NULL;

    DEBUG(3,"cleanup_module()\n");

    if (dev_list_p)
    {                                                   /* FIXME: release all devices */
        dev = dev_list_p;

        #ifdef RTLINUX
            rtl_free_irq(dev->irq);                                   /* unregister our RTLinux handler */
        #else
            free_irq(dev->irq,NULL);                                   /* release our irq */
        #endif

        release_region((unsigned)dev->pci_iomap_cfg_base,dev->pci_iomap_cfg_base_size); /* tell kernel we're done w/ io space */
        unregister_chrdev(pci_das6402_major,dev_name);                  /* tellkernel device node is available */
        /* FIXME: free all devs on llist ! */
        kfree(dev);
    }

    DEBUG(0,"module unloaded.\n");
}

