///////////////////////////////////////////////////////////////////////////////
//
// Copyright  1999 Zentropic Computing LLC.
//
// Authors:            	Stuart Hughes
// Contact:            	sehughes@zentropix.com
// Original date:      	Monday November 29th 1999
// Ident:              	@(#)$Id: regression.c,v 1.2 2000/06/12 19:50:27 seh Exp $
// Description:        	This file implements some simple regression tests
//			that will give a pass/fail summary on stdout.
//			to run it type insert the base RTL/RTAI
//			modules (e.g ../rtai and ./rtai_fifos.o for RTAI),
//			then type: ./regression
//
// License:
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//
// common user/kernel
//
///////////////////////////////////////////////////////////////////////////////
#define FIFO		0
#define MSG		"hello"
#define FILENAME	"/dev/rtf0"
#define MSGLEN		6

///////////////////////////////////////////////////////////////////////////////
//
// kernel module
//
///////////////////////////////////////////////////////////////////////////////
#if __KERNEL__

#include <linux/module.h>
#include <linux/kernel.h>
#include <asm/io.h>
#include <rtai_fifos.h>

int init_module(void)
{
	int err;
	if( (err = rtf_create(FIFO, MSGLEN * 2)) < 0) {
	    printk("rtf_create: %d\n", err);
	    return err;
        }
	if( (err = rtf_put(FIFO, MSG, MSGLEN)) != 6) {
	   printk("rtf_put, expected 6, got %d\n", err);
	}
	if( (err = rtf_put(FIFO, MSG, MSGLEN)) != 6) {
	   printk("rtf_put, expected 6, got %d\n", err);
	}
	return 0;
}

void cleanup_module(void)
{
	rtf_destroy(FIFO);
}

///////////////////////////////////////////////////////////////////////////////
//
// user process
//
///////////////////////////////////////////////////////////////////////////////
#else

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <signal.h>
#include <time.h>
#include <sys/time.h>
#include <rtai_fifos.h>

int any_err	= 0;
int rc 		= 0;

void timeout(int sig) { rc = 1; }

void result(char *msg)
{
    if(rc) {
	any_err++;
    }
    printf("%-60s : %s\n", msg, rc ? "failed" : "passed");
    rc		= 0;
}

int left_to_read(fd)
{
	int n = 0;
	if(rc = ioctl(fd, FIONREAD, &n) < 0) {
		perror("ioctl FIONREAD");
		exit(1);
	}
	return n;
}

int main()
{
    int fd, n, nrdy, then;
    char buf[128], *msg;
    fd_set rfds;
    struct timeval tv;

    signal(SIGALRM, timeout);
   
    printf(	"\nFifo regression tests\n"
		  "----------------------\n");

    system("/sbin/rmmod regression 2>/dev/null");
    if( system("/sbin/insmod ./regression.o") ) {
	fprintf(stderr, "can't insert regression.o, bailing out\n");
	exit(1);
    }

    msg = "opening fifo with data in it doesn't block";
    alarm(2);
    if( (fd = open(FILENAME, O_RDONLY)) < 0 ) {
	perror("open " FILENAME);
    }
    result(msg);

    msg = "reading less than available data doesn't block";
    alarm(2);
    *buf = 0;
    if( (n = read(fd, buf, MSGLEN)) < 0 ) {
	perror("read");
    }
    result(msg);

    msg = "check data read was correct";
    rc = strcmp(buf, "hello");
    result(msg);

    msg = "reading more than available data doesn't block";
    alarm(2);
    then   = time(0);
    if( (n = read(fd, buf, sizeof(buf))) < 0 ) {
	perror("read");
    }
    rc = ((time(0) - then) < 2) ? 0 : 1;
    result(msg);

    msg = "non-blocking read when no data available does not block";
    if( close(fd) != 0 ) {
	perror("close");
    }
    if( (fd = open(FILENAME, O_RDONLY | O_NONBLOCK)) < 0 ) {
	perror("open " FILENAME);
    }
    alarm(3);
    then   = time(0);
    if( (n = read(fd, buf, sizeof(buf))) < 0 ) {
	if( errno != EAGAIN) {
		perror("non-blocking read");
	}
    }
    rc = ((time(0) - then) <= 1) ? 0 : 1;
    result(msg);

    msg = "reading when no data available blocks";
    if( close(fd) != 0 ) {
	perror("close");
    }
    msg = "non-blocking read when no data available does not block";
    if( (fd = open(FILENAME, O_RDONLY)) < 0 ) {
	perror("open " FILENAME);
    }
    alarm(3);
    then   = time(0);
    if( (n = read(fd, buf, sizeof(buf))) < 0 ) {
	perror("read");
    }
    rc = ((time(0) - then) < 2) ? 1 : 0;
    result(msg);

    if( close(fd) != 0 ) {
	perror("close");
    }
    //////////////////////////////
    // tests using select
    //////////////////////////////
    printf(	"\nTests using select\n"
		  "----------------------\n");
    system("/sbin/rmmod regression");

    if( system("/sbin/insmod ./regression.o") ) {
	fprintf(stderr, "can't insert regression.o, bailing out\n");
	exit(1);
    }

    if( (fd = open(FILENAME, O_RDONLY)) < 0 ) {
	perror("open " FILENAME);
	exit(1);
    }

    msg = "reading less than available data doesn't block";
    *buf = 0;
    FD_ZERO(&rfds);
    FD_SET(fd, &rfds);
    tv.tv_sec  = 2;
    tv.tv_usec = 0;
    if((nrdy = select(fd+1, &rfds, NULL, NULL, &tv)) < 0) {
	perror("select");
    }
    rc = nrdy ? 0 : 1;
    result(msg);
    
    msg = "check data read was correct";
    if( (n = read(fd, buf, MSGLEN)) < 0 ) {
	perror("read");
    }
    rc = strcmp(buf, "hello");
    result(msg);

    msg = "reading more than available data doesn't block";
    alarm(2);
    then   = time(0);
    if( (n = read(fd, buf, sizeof(buf))) < 0 ) {
	perror("read");
    }
    alarm(0);
    rc = ((time(0) - then) <= 1) ? 0 : 1;
    result(msg);

    msg = "non-blocking read when no data available does not block";
    FD_ZERO(&rfds);
    FD_SET(fd, &rfds);
    tv.tv_sec  = 0;
    tv.tv_usec = 0;
    if((nrdy = select(fd+1, &rfds, NULL, NULL, &tv)) < 0) {
	perror("select");
    }
    rc = nrdy ? 1 : 0;
    result(msg);

    msg = "reading when no data available blocks";
    FD_ZERO(&rfds);
    FD_SET(fd, &rfds);
    tv.tv_sec  = 2;
    tv.tv_usec = 0;
    if((nrdy = select(fd+1, &rfds, NULL, NULL, &tv)) < 0) {
	perror("select");
    }
    rc = nrdy ? 1 : 0;
    result(msg);


    if( close(fd) != 0 ) {
	perror("close");
    }
    system("/sbin/rmmod regression");

    printf("\nAll tests complete with %d error(s)\n", any_err);
    exit(any_err);
}

#endif

