/*
COPYRIGHT (C) 1999  Paolo Mantegazza (mantegazza@aero.polimi.it)

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.
*/

/* 
ACKNOWLEDGEMENTS: 
- nice proc file contributed by Steve Papacharalanbous (stevep@zentropix.com);
- rt_printk is just an RTAI adapted copy of David Schleef rt_printk idea of 
  hacking Linux printk so that it could be safely used within NMT_RTL modules.
- from rt_printk it was easily go ahead and add also "rt_print_to_screen", for
  those that do not want messages to be logged.
- bug fixes for rtf_resize and signal handling in all wait functions provided
  by Stuart Hughes (sehughes@zentropix.com).
*/

/* 
ACKNOWLEDGEMENT NOTE: besides naming conventions and the idea of a fifo handler
function, the only remaining code from RTL original fifos, as written and 
copyrighted by Michael Barabanov, should be the function "check_blocked" 
(modified to suite my style). However I like to remark that I owe to that code 
my first understanding of Linux devices drivers.
*/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/version.h>
#include <linux/errno.h>
#include <linux/mm.h>
#include <linux/malloc.h>
#include <linux/poll.h>
#include <linux/tty_driver.h>
#include <linux/console.h>

#include <asm/io.h>

#ifdef CONFIG_PROC_FS
#include <linux/stat.h>
#include <linux/proc_fs.h>
#include <rtai_proc_fs.h>
#endif

#include <rtai.h>
#include "rtai_fifos.h"

typedef struct lx_queue {
	struct lx_queue *prev;
	struct lx_queue *next;
	struct lx_task_struct *task;
} QUEUE;

typedef struct lx_semaphore {
	int free;
	QUEUE queue;
} SEM;

typedef struct lx_task_struct {
	QUEUE queue;
	int priority;
	struct wait_queue *waitq;
	struct wait_queue linux_task;
	int blocked;
} LX_TASK;

typedef struct lx_mailbox {
	SEM sndsem, rcvsem;
	LX_TASK *waiting_task;
	char *bufadr;
	int size, fbyte, avbs, frbs;
	spinlock_t buflock;
} MBX;

typedef struct rt_fifo_struct {
	MBX mbx;		// MUST BE THE FIRST!
	int opncnt;
	int malloc_type;
	int pol_asyn_pended;
	struct wait_queue *pollq;
	struct fasync_struct *asynq;
	int (*handler)(unsigned int arg);
	SEM sem;
} FIFO;

// proc filesystem additions.
extern struct proc_dir_entry proc_rtai;
static int rtai_proc_fifo_register(void);
static void rtai_proc_fifo_unregister(void);
// End of proc filesystem additions. 

#define rtf_save_flags_and_cli()  rt_spin_lock_irqsave(&rtf_lock)
#define rtf_restore_flags(x)      rt_spin_unlock_irqrestore((x), &rtf_lock)

static int fifo_srq, async_sig;
static spinlock_t rtf_lock = SPIN_LOCK_UNLOCKED;

#define MAX_FIFOS 64
static FIFO fifo[MAX_FIFOS];

#define MAXREQS 64	// KEEP IT A POWER OF 2!!!
static struct { int in, out; struct wait_queue **waitq[MAXREQS]; } taskq;
static struct { int in, out; FIFO *fifo[MAXREQS]; } pol_asyn_q;

static int do_nothing(unsigned int arg) { return 0; }

static inline int check_blocked(sigset_t signal, sigset_t blocked)
{
	int i = 0;
	do {
		if ( signal.sig[i] & ~blocked.sig[i] ) {
			return 1;
		}
	} while (++i < _NSIG_WORDS);
	return 0;
}

static inline void mbx_sem_signal(SEM *sem, FIFO *fifop)
{
	unsigned long flags;
	LX_TASK *task;

	flags = rtf_save_flags_and_cli();
	if ((task = (sem->queue.next)->task)) {
		sem->queue.next = task->queue.next;
		(task->queue.next)->prev = &(sem->queue);
		task->blocked = 0;
		if ((task->linux_task.task)->state == TASK_INTERRUPTIBLE) {
			taskq.waitq[taskq.in] = &(task->waitq);
			taskq.in = (taskq.in + 1) & (MAXREQS - 1);
			rt_pend_linux_srq(fifo_srq);
		}
	} else {
		sem->free = 1;
		if (fifop && !(fifop->pol_asyn_pended) &&
		    (((MBX *)fifop)->avbs || ((MBX *)fifop)->frbs) &&
		    (fifop->pollq != WAIT_QUEUE_HEAD(&(fifop->pollq)) || 
								fifop->asynq)) {
			fifop->pol_asyn_pended = 1;
			pol_asyn_q.fifo[pol_asyn_q.in] = fifop;
			pol_asyn_q.in = (pol_asyn_q.in + 1) & (MAXREQS - 1);
			rt_pend_linux_srq(fifo_srq);
		}
	}
	rtf_restore_flags(flags);
}

static inline void mbx_signal(MBX *mbx)
{
	unsigned long flags;
	LX_TASK *task;

	flags = rtf_save_flags_and_cli();
	if ((task = mbx->waiting_task)) {
		mbx->waiting_task = 0;
		task->blocked = 0;
		if ((task->linux_task.task)->state == TASK_INTERRUPTIBLE) {
			taskq.waitq[taskq.in] = &(task->waitq);
			taskq.in = (taskq.in + 1) & (MAXREQS - 1);
			rt_pend_linux_srq(fifo_srq);
		}
	}
	rtf_restore_flags(flags);
}

static inline int mbx_sem_wait_if(SEM *sem)
{
	unsigned long flags;
	int free;
	flags = rtf_save_flags_and_cli();
	if ((free = sem->free)) {
		sem->free = 0;
	}
	rtf_restore_flags(flags);
	return free;
}

static inline int mbx_sem_wait(SEM *sem)
{
	unsigned long flags;
	LX_TASK task;
	QUEUE *q;

	task.queue.task = &task;
	task.priority = 1E6 - current->rt_priority;
	task.blocked = 1;
	task.waitq = 0;
	task.linux_task.task = current;
	__add_wait_queue(&task.waitq, &task.linux_task);

	flags = rtf_save_flags_and_cli();
	if (!(sem->free)) {
		q = &(sem->queue);
		while ((q = q->next) != &(sem->queue) &&
			 (q->task)->priority <= task.priority);
		task.queue.prev = q->prev;
		task.queue.next = q;
		(q->prev)->next = &(task.queue);
		q->prev = &(task.queue);
		current->state = TASK_INTERRUPTIBLE;
		rtf_restore_flags(flags);
		schedule();
		flags = rtf_save_flags_and_cli();
		if (task.blocked) { 
			(task.queue.prev)->next = task.queue.next;
			(task.queue.next)->prev = task.queue.prev;
			if (!((sem->queue.next)->task)) {
				sem->free = 1;
			}
		}
		rtf_restore_flags(flags);
		if (check_blocked(current->signal, current->blocked)) {
			return -ERESTARTSYS;
		}
	} else {
		sem->free = 0;
		rtf_restore_flags(flags);
		task.blocked = 0;
	}
	return task.blocked;
}

static inline int mbx_wait(MBX *mbx, int *fravbs)
{
	unsigned long flags;
	LX_TASK task;

	task.blocked = 1;
	task.waitq = 0;
	task.linux_task.task = current;
	__add_wait_queue(&task.waitq, &task.linux_task);

	flags = rtf_save_flags_and_cli();
	if (!(*fravbs)) {
		mbx->waiting_task = &task;
		current->state = TASK_INTERRUPTIBLE;
		rtf_restore_flags(flags);
		schedule();
		mbx->waiting_task = 0;
		if (check_blocked(current->signal, current->blocked)) {
			if (task.blocked) {
				if (fravbs == &mbx->avbs) {
					mbx_sem_signal(&(mbx->rcvsem), (FIFO *)mbx);
				} else {
					mbx_sem_signal(&(mbx->rcvsem), (FIFO *)mbx);
				}
			}
			return -ERESTARTSYS;
		}
	} else {
		rtf_restore_flags(flags);
		task.blocked = 0;
	}
	return task.blocked;
}

static inline int mbx_sem_wait_timed(SEM *sem, int delay)
{
	unsigned long flags;
	LX_TASK task;
	QUEUE *q;

	task.queue.task = &task;
	task.priority = 1E6 - current->rt_priority;
	task.blocked = 1;
	task.waitq = 0;
	task.linux_task.task = current;
	__add_wait_queue(&task.waitq, &task.linux_task);

	flags = rtf_save_flags_and_cli();
	if (!(sem->free)) {
		q = &(sem->queue);
		while ((q = q->next) != &(sem->queue) &&
			 (q->task)->priority <= task.priority);
		task.queue.prev = q->prev;
		task.queue.next = q;
		(q->prev)->next = &(task.queue);
		q->prev = &(task.queue);
		current->state = TASK_INTERRUPTIBLE;
		rtf_restore_flags(flags);
		schedule_timeout(delay);
		flags = rtf_save_flags_and_cli();
		if (task.blocked) { 
			(task.queue.prev)->next = task.queue.next;
			(task.queue.next)->prev = task.queue.prev;
			if (!((sem->queue.next)->task)) {
				sem->free = 1;
			}
		}
		rtf_restore_flags(flags);
		if (check_blocked(current->signal, current->blocked)) {
			return -ERESTARTSYS;
		}
	} else {
		sem->free = 0;
		rtf_restore_flags(flags);
		task.blocked = 0;
	}
	return task.blocked;
}

static inline int mbx_wait_timed(MBX *mbx, int *fravbs, int delay)
{
	unsigned long flags;
	LX_TASK task;

	task.blocked = 1;
	task.waitq = 0;
	task.linux_task.task = current;
	__add_wait_queue(&task.waitq, &task.linux_task);

	flags = rtf_save_flags_and_cli();
	if (!(*fravbs)) {
		mbx->waiting_task = &task;
		current->state = TASK_INTERRUPTIBLE;
		rtf_restore_flags(flags);
		schedule_timeout(delay);
		mbx->waiting_task = 0;
		if (check_blocked(current->signal, current->blocked)) {
			if (task.blocked) {
				if (fravbs == &mbx->avbs) {
					mbx_sem_signal(&(mbx->rcvsem), (FIFO *)mbx);
				} else {
					mbx_sem_signal(&(mbx->rcvsem), (FIFO *)mbx);
				}
			}
			return -ERESTARTSYS;
		}
	} else {
		rtf_restore_flags(flags);
		task.blocked = 0;
	}
	return task.blocked;
}

#define MOD_SIZE(indx) ((indx) < mbx->size ? (indx) : (indx) - mbx->size)

static inline int mbx_put(MBX *mbx, char **msg, int msg_size, int lnx)
{
	int tocpy, last_byte;
	unsigned long flags;

	flags = rt_spin_lock_irqsave(&(mbx->buflock));
	while (mbx->frbs && msg_size > 0) {
		last_byte = MOD_SIZE(mbx->fbyte + mbx->avbs);
		if ((tocpy = mbx->size - last_byte) > msg_size) {
			tocpy = msg_size;
		}
		if (tocpy > mbx->frbs) {
			tocpy = mbx->frbs;
		}
		rt_spin_unlock_irqrestore(flags, &(mbx->buflock));
		if (lnx) {
			copy_from_user(mbx->bufadr + last_byte, *msg, tocpy);
		} else {
			memcpy(mbx->bufadr + last_byte, *msg, tocpy);
		}
		msg_size  -= tocpy;
		*msg      += tocpy;
		flags = rt_spin_lock_irqsave(&(mbx->buflock));
		mbx->frbs -= tocpy;
		mbx->avbs += tocpy;
	}
	rt_spin_unlock_irqrestore(flags, &(mbx->buflock));
	return msg_size;
}

static inline int mbx_get(MBX *mbx, char **msg, int msg_size, int lnx)
{
	int tocpy;
	unsigned long flags;

	flags = rt_spin_lock_irqsave(&(mbx->buflock));
	while (mbx->avbs && msg_size > 0) {
		if ((tocpy = mbx->size - mbx->fbyte) > msg_size) {
			tocpy = msg_size;
		}
		if (tocpy > mbx->avbs) {
			tocpy = mbx->avbs;
		}
		rt_spin_unlock_irqrestore(flags, &(mbx->buflock));
		if (lnx) {
			copy_to_user(*msg, mbx->bufadr + mbx->fbyte, tocpy);
		} else {
			memcpy(*msg, mbx->bufadr + mbx->fbyte, tocpy);
		}
		msg_size  -= tocpy;
		*msg      += tocpy;
		flags = rt_spin_lock_irqsave(&(mbx->buflock));
		mbx->fbyte = MOD_SIZE(mbx->fbyte + tocpy);
		mbx->frbs += tocpy;
		mbx->avbs -= tocpy;
	}
	rt_spin_unlock_irqrestore(flags, &(mbx->buflock));
	return msg_size;
}

static inline void mbx_sem_init(SEM *sem, int value)
{
	sem->free = value;
	sem->queue.prev = &(sem->queue);
	sem->queue.next = &(sem->queue);
	sem->queue.task = 0;
}

static inline int mbx_sem_delete(SEM *sem)
{
	unsigned long flags;
	LX_TASK *task;

	flags = rtf_save_flags_and_cli();
	while ((task = (sem->queue.next)->task)) {
		sem->queue.next = task->queue.next;
		(task->queue.next)->prev = &(sem->queue);
		if ((task->linux_task.task)->state == TASK_INTERRUPTIBLE) {
			taskq.waitq[taskq.in] = &(task->waitq);
			taskq.in = (taskq.in + 1) & (MAXREQS - 1);
			rt_pend_linux_srq(fifo_srq);
		}
	}
	rtf_restore_flags(flags);
	return 0;
}

static inline void mbx_init(MBX *mbx, int size, char *bufadr)
{
	mbx_sem_init(&(mbx->sndsem), 1);
	mbx_sem_init(&(mbx->rcvsem), 1);
	mbx->waiting_task = 0;
	mbx->bufadr = bufadr;
	mbx->size = mbx->frbs = size;
//	mbx->buflock.lock = mbx->fbyte = mbx->avbs = 0;
	mbx->fbyte = mbx->avbs = 0;
	spin_lock_init(&(mbx->buflock));
}

static inline int mbx_delete(MBX *mbx)
{
	if (mbx_sem_delete(&(mbx->sndsem)) || mbx_sem_delete(&(mbx->rcvsem))) {
		return -EFAULT;
	}
	return 0;
}

static inline int mbx_send(MBX *mbx, void *msg, int msg_size, int lnx)
{
	if (mbx_sem_wait(&(mbx->sndsem))) {
		return msg_size;
	}
	while (msg_size) {
		if (mbx_wait(mbx, &mbx->frbs)) {
			return msg_size;
		}
		msg_size = mbx_put(mbx, (char **)(&msg), msg_size, lnx);
		mbx_signal(mbx);
	}
	mbx_sem_signal(&(mbx->sndsem), (FIFO *)mbx);
	return 0;
}

static inline int mbx_send_wp(MBX *mbx, void *msg, int msg_size, int lnx)
{
	unsigned long flags;

	flags = rtf_save_flags_and_cli();
	if (mbx->sndsem.free && mbx->frbs) {
		mbx->sndsem.free = 0;
		rtf_restore_flags(flags);
		mbx_put(mbx, (char **)(&msg), 
		       msg_size > mbx->frbs ? mbx->frbs : msg_size, lnx);
		mbx_signal(mbx);
		mbx_sem_signal(&(mbx->sndsem), (FIFO *)mbx);
		return 0;
	}
	rtf_restore_flags(flags);
	return msg_size;
}

static int mbx_send_timed(MBX *mbx, void *msg, int msg_size, int delay, int lnx)
{
	if (mbx_sem_wait_timed(&(mbx->sndsem), delay)) {
		return msg_size;
	}
	while (msg_size) {
		if (mbx_wait_timed(mbx, &(mbx->frbs), delay)) {
			return msg_size;
		}
		msg_size = mbx_put(mbx, (char **)(&msg), msg_size, lnx);
		mbx_signal(mbx);
	}
	mbx_sem_signal(&(mbx->sndsem), (FIFO *)mbx);
	return 0;
}

static inline int mbx_receive(MBX *mbx, void *msg, int msg_size, int lnx)
{
	if (mbx_sem_wait(&(mbx->rcvsem))) {
		return msg_size;
	}
	while (msg_size) {
		if (mbx_wait(mbx, &mbx->avbs)) {
			return msg_size;
		}
		msg_size = mbx_get(mbx, (char **)(&msg), msg_size, lnx);
		mbx_signal(mbx);
	}
	mbx_sem_signal(&(mbx->rcvsem), (FIFO *)mbx);
	return 0;
}

static inline int mbx_receive_wjo(MBX *mbx, void *msg, int msg_size, int lnx)
{
	if (mbx_sem_wait(&(mbx->rcvsem))) {
		return msg_size;
	}
	if (msg_size) {
		if (mbx_wait(mbx, &mbx->avbs)) {
			return msg_size;
		}
		msg_size = mbx_get(mbx, (char **)(&msg), msg_size, lnx);
		mbx_signal(mbx);
	}
	mbx_sem_signal(&(mbx->rcvsem), (FIFO *)mbx);
	return msg_size;
}

static inline int mbx_receive_wp(MBX *mbx, void *msg, int msg_size, int lnx)
{
	unsigned long flags;

	flags = rtf_save_flags_and_cli();
	if (mbx->rcvsem.free && mbx->avbs) {
		mbx->rcvsem.free = 0;
		rtf_restore_flags(flags);
		mbx_get(mbx, (char **)(&msg), 
		       msg_size > mbx->avbs ? mbx->avbs : msg_size, lnx);
		mbx_signal(mbx);
		mbx_sem_signal(&(mbx->rcvsem), (FIFO *)mbx);
		return 0;
	}
	rtf_restore_flags(flags);
	return msg_size;
}

static int mbx_receive_timed(MBX *mbx, void *msg, int msg_size, int delay, int lnx)
{
	if (mbx_sem_wait_timed(&(mbx->rcvsem), delay)) {
		return msg_size;
	}
	while (msg_size) {
		if (mbx_wait_timed(mbx, &(mbx->avbs), delay)) {
			return msg_size;
		}
		msg_size = mbx_get(mbx, (char **)(&msg), msg_size, lnx);
		mbx_signal(mbx);
	}
	mbx_sem_signal(&(mbx->rcvsem), (FIFO *)mbx);
	return 0;
}

static void rt_printk_sysreq_handler(void);

void rtf_sysrq_handler(void)
{
	FIFO *fifop;
	rt_printk_sysreq_handler();
	while (taskq.out != taskq.in) {
		wake_up_interruptible(taskq.waitq[taskq.out]);
		taskq.out = (taskq.out + 1) & (MAXREQS - 1);
	}
	while (pol_asyn_q.out != pol_asyn_q.in) {
		fifop = pol_asyn_q.fifo[pol_asyn_q.out];
		fifop->pol_asyn_pended = 0;
		if ((fifop = pol_asyn_q.fifo[pol_asyn_q.out])->pollq) {
			wake_up_interruptible(&(fifop->pollq));
		}
		if (fifop->asynq) { 
			kill_fasync(fifop->asynq, async_sig);
		}
		pol_asyn_q.out = (pol_asyn_q.out + 1) & (MAXREQS - 1);
	}
}

#define VALID_FIFO	if (minor >= MAX_FIFOS) { return -ENODEV; } \
			if (!(fifo[minor].opncnt)) { return -EINVAL; }

int rtf_reset(unsigned int minor)
{
	MBX *mbx;
	VALID_FIFO;
	if (mbx_delete(mbx = &(fifo[minor].mbx))) {
		return -EFAULT;
	}
	mbx_init(mbx, mbx->size, mbx->bufadr);
	return 0;
}

int rtf_resize(unsigned int minor, int size)
{
	void *bufadr, *msg;
	int malloc_type;
	MBX *mbx;
	
	VALID_FIFO;
	malloc_type = fifo[minor].malloc_type;
	if (size <= PAGE_SIZE*32) {
		if (!(bufadr = kmalloc(size, GFP_KERNEL))) {
			return -ENOMEM;
		}
		fifo[minor].malloc_type = 'k';
	} else {
		if (!(bufadr = vmalloc(size))) {
			return -ENOMEM;
		}
		fifo[minor].malloc_type = 'v';
	}
	msg = bufadr;
	mbx = &(fifo[minor].mbx);
	mbx->avbs = 1E9 - mbx_get(mbx, (char **)&msg, 1E9, 0);
	if (malloc_type == 'k') {
		kfree(fifo[minor].mbx.bufadr); 
	} else {
		vfree(fifo[minor].mbx.bufadr); 
	}
	mbx->bufadr = bufadr;
	mbx->size = size > mbx->size ? size : mbx->size;
	mbx->fbyte = 0;
	mbx->frbs = mbx->size - mbx->avbs;
	return size;
}

int rtf_create(unsigned int minor, int size)
{
	void *buf;
	if (minor >= MAX_FIFOS) {
		return -ENODEV;
	}
	if (!(fifo[minor].opncnt)) {
		if (size <= PAGE_SIZE*32) {
			if (!(buf = kmalloc(size, GFP_KERNEL))) {
				return -ENOMEM;
			}
			fifo[minor].malloc_type = 'k';
		} else {
			if (!(buf = vmalloc(size))) {
				return -ENOMEM;
			}
			fifo[minor].malloc_type = 'v';
		}
		fifo[minor].handler = do_nothing;
		mbx_init(&(fifo[minor].mbx), size, buf);
		mbx_sem_init(&(fifo[minor].sem), 0);
		fifo[minor].pol_asyn_pended = 0;
		fifo[minor].asynq = 0;;
	} else {
		if (size > fifo[minor].mbx.size) {
			rtf_resize(minor, size);
		}
	}
	MOD_INC_USE_COUNT;
	(fifo[minor].opncnt)++;
	return 0;
}

int rtf_destroy(unsigned int minor)
{
	VALID_FIFO;
	MOD_DEC_USE_COUNT;
	(fifo[minor].opncnt)--;
	if(!(fifo[minor].opncnt)) {
		if (fifo[minor].malloc_type == 'k') {
			kfree(fifo[minor].mbx.bufadr); 
		} else {
			vfree(fifo[minor].mbx.bufadr); 
		}
		fifo[minor].handler = do_nothing;
		mbx_delete(&(fifo[minor].mbx));
		fifo[minor].pol_asyn_pended = 0;
		fifo[minor].asynq = 0;;
	}
	return fifo[minor].opncnt;
}

int rtf_create_handler(unsigned int minor, int (*handler) (unsigned int fifo))
{
	if (minor >= MAX_FIFOS || !handler) {
		return -EINVAL;
	}
	fifo[minor].handler = handler;
	return 0;
}

int rtf_put(unsigned int minor, void *buf, int count)
{
	VALID_FIFO;
	count -= mbx_send_wp(&(fifo[minor].mbx), buf, count, 0);
	return count;
}

int rtf_get(unsigned int minor, void *buf, int count)
{
	VALID_FIFO;
	count -= mbx_receive_wp(&(fifo[minor].mbx), buf, count, 0);
	return count;
}

int rtf_sem_init(unsigned int minor, int value)
{
	VALID_FIFO;
	mbx_sem_init(&(fifo[minor].sem), value);
	return 0;
}

int rtf_sem_post(unsigned int minor)
{
	VALID_FIFO;
	mbx_sem_signal(&(fifo[minor].sem), 0);
	return 0;
}

int rtf_sem_trywait(unsigned int minor)
{
	VALID_FIFO;
	return mbx_sem_wait_if(&(fifo[minor].sem));
}

int rtf_sem_delete(unsigned int minor)
{
	VALID_FIFO;
	return mbx_sem_delete(&(fifo[minor].sem));
}

static int rtf_open(struct inode *inode, struct file *filp)
{
#define DEFAULT_SIZE 1000
	return rtf_create(MINOR(inode->i_rdev), DEFAULT_SIZE);
}

static int rtf_fasync(int fd, struct file *filp, int mode)
{	
	int minor;
	minor = MINOR((filp->f_dentry->d_inode)->i_rdev);
	return fasync_helper(fd, filp, mode, &(fifo[minor].asynq));
	if (!mode) {
		fifo[minor].asynq = 0;
	}
}

static int rtf_release(struct inode *inode, struct file *filp)
{
	int minor;
	minor = MINOR(inode->i_rdev);
	wake_up_interruptible(&(fifo[minor].pollq));
	rtf_fasync(-1, filp, 0);
	return rtf_destroy(minor);
}

static ssize_t rtf_read(struct file *filp, char *buf, size_t count, loff_t* ppos)
{
	struct inode *inode = filp->f_dentry->d_inode;
	unsigned int minor = MINOR(inode->i_rdev);
	int handler_ret;

	if (filp->f_flags & O_NONBLOCK) {
		count -= mbx_receive_wp(&(fifo[minor].mbx), buf, count, 1);
		if (!count) {
			return -EAGAIN;
		}
	} else {
		count -= mbx_receive_wjo(&(fifo[minor].mbx), buf, count, 1);
	}

	if (count) {
		inode->i_atime = CURRENT_TIME;
//		if ((handler_ret = (fifo[minor].handler)(minor)) < 0) {
		if ((handler_ret = ((int (*)(int, ...))(fifo[minor].handler))(minor, 'r')) < 0) {
			return handler_ret;
		}
		return count;
	}
	return 0;

	return count;
}

static ssize_t rtf_write(struct file *filp, const char *buf, size_t count, loff_t* ppos)
{
	struct inode *inode = filp->f_dentry->d_inode;
	unsigned int minor = MINOR(inode->i_rdev);
	int handler_ret;

	if (filp->f_flags & O_NONBLOCK) {
		count -= mbx_send_wp(&(fifo[minor].mbx), (char *)buf, count, 1);
		if (!count) {
			return -EAGAIN;
		}
	} else {
		count -= mbx_send(&(fifo[minor].mbx), (char *)buf, count, 1);
	}

	inode->i_ctime = inode->i_mtime = CURRENT_TIME;
//	if ((handler_ret = (fifo[minor].handler)(minor)) < 0) {
	if ((handler_ret = ((int (*)(int, ...))(fifo[minor].handler))(minor, 'w')) < 0) {
		return handler_ret;
	}

	return count;
}

#define DELAY(x) (((x)*HZ + 500)/1000)

static int rtf_ioctl(struct inode *inode, struct file *filp, unsigned int cmd, unsigned long arg)
{	
	unsigned int minor;
	FIFO *fifop;

	fifop = fifo + (minor = MINOR(inode->i_rdev));
	switch(cmd) {
		case RESET: {
			return rtf_reset(minor);
		}
		case RESIZE: {
			return rtf_resize(minor, arg);
		}
		case SUSPEND_TIMED: {
			current->state = TASK_INTERRUPTIBLE;
			schedule_timeout(DELAY(arg));
			if (check_blocked(current->signal, current->blocked)) {
				return -ERESTARTSYS;
			}
			return 0;
		}
		case OPEN_SIZED: {
			return rtf_create(minor, arg);
		}
		case READ_ALL_AT_ONCE: {
			struct { char *buf; int count; } args;
			int handler_ret;
			copy_from_user(&args, (void *)arg, sizeof(args));
			args.count -= mbx_receive(&(fifop->mbx), args.buf, args.count, 1);
			if (args.count) {
				inode->i_atime = CURRENT_TIME;
//				if ((handler_ret = (fifop->handler)(minor)) < 0) {
				if ((handler_ret = ((int (*)(int, ...))(fifo[minor].handler))(minor, 'r')) < 0) {
					return handler_ret;
				}
				return args.count;
			}
			return 0;
		}
		case READ_TIMED: {
			struct { char *buf; int count, delay; } args;
			int handler_ret;
			copy_from_user(&args, (void *)arg, sizeof(args));
			if (!args.delay) {
				args.count -= mbx_receive_wp(&(fifop->mbx), args.buf, args.count, 1);
				if (!args.count) {
					return -EAGAIN;
				}
			} else {
				args.count -= mbx_receive_timed(&(fifop->mbx), args.buf, args.count, DELAY(args.delay), 1);
			}
			if (args.count) {
				inode->i_atime = CURRENT_TIME;
//				if ((handler_ret = (fifop->handler)(minor)) < 0) {
				if ((handler_ret = ((int (*)(int, ...))(fifo[minor].handler))(minor, 'r')) < 0) {
					return handler_ret;
				}
				return args.count;
			}
			return 0;
		}
		case WRITE_TIMED: {
			struct { char *buf; int count, delay; } args;
			int handler_ret;
			copy_from_user(&args, (void *)arg, sizeof(args));
			if (!args.delay) {
				args.count -= mbx_send_wp(&(fifop->mbx), args.buf, args.count, 1);
				if (!args.count) {
					return -EAGAIN;
				}
			} else {
				args.count -= mbx_send_timed(&(fifop->mbx), args.buf, args.count, DELAY(args.delay), 1);
			}
			inode->i_ctime = inode->i_mtime = CURRENT_TIME;
//			if ((handler_ret = (fifop->handler)(minor)) < 0) {
			if ((handler_ret = ((int (*)(int, ...))(fifo[minor].handler))(minor, 'w')) < 0) {
				return handler_ret;
			}
			return args.count;
		}
		case RTF_SEM_INIT: {
			mbx_sem_init(&(fifop->sem), arg);
			return 0;
		}
		case RTF_SEM_WAIT: {
			return mbx_sem_wait(&(fifop->sem));
		}
		case RTF_SEM_TRYWAIT: {
			return mbx_sem_wait_if(&(fifop->sem));
		}
		case RTF_SEM_TIMED_WAIT: {
			return mbx_sem_wait_timed(&(fifop->sem), DELAY(arg));
		}
		case RTF_SEM_POST: {
			mbx_sem_signal(&(fifop->sem), 0);
			return 0;
		}
		case RTF_SEM_DESTROY: {
			mbx_sem_delete(&(fifop->sem));
			return 0;
		}
		case SET_ASYNC_SIG: {
			async_sig = arg;
			return 0;
		}
		case FIONREAD: 
			return put_user(fifo[minor].mbx.avbs, (int *) arg);
		default:
			printk("cmd %d is not implemented\n", cmd);
			return -ENOSYS;
	}
	return 0;
}

static unsigned int rtf_poll(struct file *filp, poll_table *wait)
{
	unsigned int minor, nleft, space, mask = 0;

	minor = MINOR((filp->f_dentry->d_inode)->i_rdev);

	nleft = fifo[minor].mbx.avbs;
	space = fifo[minor].mbx.frbs;

	poll_wait(filp, &(fifo[minor].pollq), wait);

	if (nleft) {
		mask |= POLLIN | POLLRDNORM;
	}
	if (space) {
		mask |= POLLOUT | POLLWRNORM;
	}
	return mask;
}

static loff_t rtf_llseek(struct file *filp, loff_t offset, int origin)
{
	return rtf_reset(MINOR((filp->f_dentry->d_inode)->i_rdev));
}

static struct file_operations rtf_fops =
{
	rtf_llseek, rtf_read, rtf_write, NULL, rtf_poll, rtf_ioctl, NULL,
	rtf_open, NULL, rtf_release, NULL, rtf_fasync, NULL, NULL, NULL
};

int init_module(void)
{
	int minor;

	if (register_chrdev(RTAI_MAJOR, "rtai_fifo", &rtf_fops)) {
		printk("RTAI-FIFO: cannot register major %d.\n", RTAI_MAJOR);
		return -EIO;
	}
	if ((fifo_srq = rt_request_srq(0, rtf_sysrq_handler, 0)) < 0) {
		printk("RTAI-FIFO: no srq available in rtai.\n");
		return fifo_srq;
	}
	taskq.in = taskq.out = pol_asyn_q.in = pol_asyn_q.out = 0;
	async_sig = SIGIO;
	for (minor = 0; minor < MAX_FIFOS; minor++) {
		fifo[minor].opncnt = fifo[minor].pol_asyn_pended = 0;
		init_waitqueue(&fifo[minor].pollq);
		fifo[minor].asynq = 0;;
		mbx_sem_init(&(fifo[minor].sem), 0);
	}
#ifdef CONFIG_PROC_FS
	rtai_proc_fifo_register();
#endif
	return 0;
}

void cleanup_module(void)
{
	if (rt_free_srq(fifo_srq) < 0) {
		printk("RTAI-FIFO: rtai srq %d illegal or already free.\n", fifo_srq);
	}
#ifdef CONFIG_PROC_FS
        rtai_proc_fifo_unregister();
#endif
	unregister_chrdev(RTAI_MAJOR, "rtai_fifo");
}

/* ----------------------< proc filesystem section >----------------------*/

static int rtai_read_fifos(char* buf, char** start, off_t offset, int len, int unused)
{
	int i;
	char tmp_str[12];

	len = sprintf(buf, "\nRTAI Real Time fifos status.\n\n" );
	if (len > LIMIT) {
		return(len);
	}
	len += sprintf(buf + len, "    fifo No  Open Cnt  Buff Size  malloc type\n" );
	if (len > LIMIT) {
		return(len);
	}
	len += sprintf( buf + len, "    -----------------------------------------\n" );
	if (len > LIMIT) {
		return(len);
	}
/*
* Display the status of all open RT fifos.
*
*/
	for (i = 0; i < MAX_FIFOS; i++) {
		if (fifo[i].opncnt > 0) {
			if (fifo[i].malloc_type == 'v') {
				strcpy( tmp_str, "vmalloc");
			} else {
				strcpy( tmp_str, "kmalloc");
			}
			len += sprintf(buf + len, "    %-8d %-9d %-10d %-10s\n", i,
                        		        fifo[i].opncnt,
                                 		fifo[i].mbx.size,
						tmp_str );
			if (len > LIMIT) {
				return(len);
			}
		} /* End if - fifo is open. */
	} /* End for loop - loop for all fifos. */
	return len;

}  /* End function - rtai_read_fifos */

static struct proc_dir_entry proc_fifo_entry = {
        0,                      /* low_ino: inode is dynamic. */
        5, "fifos",         /* length of name and name. */
        S_IFREG | S_IRUGO,      /* mode. */
        1, 0, 0,                /* nlinks, owner, group. */
        0,                      /* size -- not used. */
        NULL,                   /* operations -- use default. */
        &rtai_read_fifos,       /* function used to read data. */
        /* That's it */
};

static int rtai_proc_fifo_register(void) 
{
	int r_c;
	if((r_c = proc_register( &proc_rtai, &proc_fifo_entry )) < 0) {
		return(r_c);
	}
	return(r_c);
}  /* End function - rtai_proc_fifo_register */

static void rtai_proc_fifo_unregister(void) 
{
  proc_unregister( &proc_rtai, proc_fifo_entry.low_ino );
}  /* End function - rtai_proc_fifo_unregister */

/* ------------------< end of proc filesystem section >------------------*/

/*
 *  rt_printk.c, hacked from linux/kernel/printk.c.
 *
 * Modified for RT support, David Schleef.
 *
 * Adapted to RTAI and restyled his way by Paolo Mantegazza.
 */

#define PRINTK_BUF_LEN	(8192)
#define TEMP_BUF_LEN	(512)

static char rt_printk_buf[PRINTK_BUF_LEN];
static int buf_front, buf_back;

static char buf[TEMP_BUF_LEN];

int rt_printk(const char *fmt, ...)
{
	va_list args;
	int len, i;
	unsigned long flags;

	flags = rtf_save_flags_and_cli();
	va_start(args, fmt);
	len = vsprintf(buf, fmt, args);
	va_end(args);
	if (buf_front + len >= PRINTK_BUF_LEN) {
		i = PRINTK_BUF_LEN - buf_front;
		memcpy(rt_printk_buf + buf_front, buf, i);
		memcpy(rt_printk_buf, buf + i, len - i);
		buf_front = len - i;
	} else {
		memcpy(rt_printk_buf + buf_front, buf, len);
		buf_front += len;
	}
	rtf_restore_flags(flags);
	rt_pend_linux_srq(fifo_srq);

	return len;
}

static void rt_printk_sysreq_handler(void)
{
	int tmp;

	while(1) {
		tmp = buf_front;
		if (buf_back  > tmp) {
			printk("%.*s", PRINTK_BUF_LEN - buf_back, rt_printk_buf + buf_back);
			buf_back = 0;
		}
		if (buf_back == tmp) {
			break;
		}
		printk("%.*s", tmp - buf_back, rt_printk_buf + buf_back);
		buf_back = tmp;
	}
}

int rt_print_to_screen(const char *fmt, ...)
{
	va_list args;
        struct console *c;
        unsigned long flags;
        int len;

	flags = rtf_save_flags_and_cli();
	va_start(args, fmt);
	len = vsprintf(buf, fmt, args);
	va_end(args);
        c = console_drivers;
        while(c) {
                if ((c->flags & CON_ENABLED) && c->write)
                        c->write(c, buf, len);
                c = c->next;
        }
	rtf_restore_flags(flags);
	return len;
}
